package org.kink_lang.kink;

import java.util.List;
import java.util.Locale;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * A <a href="../../../../../manual/api/kink-STR.html#type-str">str val</a>.
 *
 * @see Vm#str
 */
public class StrVal extends Val {

    /** The content of the str. */
    private StrCell cell;

    /** The count of runes in the str. */
    private int runeCount = UNINITIALIZED_RUNE_COUNT;

    /** runeCount field has not been initialized. */
    private static final int UNINITIALIZED_RUNE_COUNT = -1;

    /** The cache of (upper 32bits = rune pos, lower 32bits = char pos). */
    private long posCache;

    /**
     * Constructs a str val.
     *
     * @param vm the vm.
     * @param string the string.
     */
    StrVal(Vm vm, String string) {
        this(vm, new StrCell(string));
    }

    /**
     * Constructs a str val with the binary tree.
     */
    private StrVal(Vm vm, StrCell cell) {
        super(vm, null);
        Preconds.checkArg(cell.charCountEstimate() <= vm.str.getMaxLength(),
                "length must not exceed the max");
        this.cell = cell;
    }

    /**
     * Returns the string.
     *
     * @return the string.
     */
    public String string() {
        this.cell = this.cell.singleCell();
        return this.cell.back();
    }

    /**
     * Returns the count of runes in the str.
     */
    int runeCount() {
        if (this.runeCount == UNINITIALIZED_RUNE_COUNT) {
            var s = string();
            this.runeCount = s.codePointCount(0, s.length());
        }
        return this.runeCount;
    }

    /**
     * Returns the char length of the string.
     */
    int charCountEstimate() {
        return this.cell.charCountEstimate();
    }

    /**
     * Returns whether the string is empty.
     *
     * @return whether the string is empty.
     */
    public boolean isEmpty() {
        return this.cell.isEmpty();
    }

    /**
     * Converts the rune pos index to the corresponding char pos index.
     */
    int runePosToCharPos(int runePos) {
        long cache = this.posCache;
        int cachedRunePos = (int) (cache >>> 32);
        int cachedCharPos = (int) cache;
        int charPos = string().offsetByCodePoints(cachedCharPos, runePos - cachedRunePos);
        this.posCache = (((long) runePos) << 32) | Integer.toUnsignedLong(charPos);
        return charPos;
    }

    /**
     * Returns a str val concatenating {@code this} and {@code tail}.
     *
     * @param tail str val to concatenate.
     * @return a str val concatenating {@code this} and {@code tail}.
     */
    public StrVal concat(StrVal tail) {
        Preconds.checkArg(
                (long) charCountEstimate() + tail.charCountEstimate() <= vm.str.getMaxLength(),
                "length must not exceed the max");

        if (isEmpty()) {
            return tail;
        } else if (tail.isEmpty()) {
            return this;
        }

        StrCell resultCell = this.cell.concat(tail.string());
        return new StrVal(vm, resultCell);
    }

    @Override
    SharedVars sharedVars() {
        return vm.str.sharedVars;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "StrVal(%s)", string());
    }

    /**
     * Returns properties which determine equality of the type.
     */
    private List<Object> properties() {
        return List.of(this.vm, string());
    }

    @Override
    public int hashCode() {
        return properties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof StrVal argStr
            && properties().equals(argStr.properties());
    }

    /**
     * Containing the string, as a reversed cons list.
     *
     * It is optimized for concatenation like (((x + y) + z) + w).
     *
     * @param cellCount the number of cells, including {@code this}.
     * @param charCountEstimate etsimated count of chars.
     * @param front cell on the front side; may be null.
     * @param back string on the back side.
     */
    private record StrCell(
            int cellCount,
            int charCountEstimate,
            @Nullable StrCell front,
            String back) {

        /**
         * Makes a single cell containing {@code str}.
         */
        StrCell(String str) {
            this(1, str.length(), null, str);
        }

        /**
         * Whether it is empty.
         */
        boolean isEmpty() {
            return front() == null && back.isEmpty();
        }

        /**
         * Concatenate strings this on the front side, and {@code tail} at the back side.
         */
        StrCell concat(String tail) {
            return new StrCell(cellCount() + 1,
                    charCountEstimate() + tail.length(),
                    this,
                    tail);
        }

        /**
         * Returns a single cell containing the string of {@code this} cell.
         */
        StrCell singleCell() {
            var isSingle = front() == null;
            if (isSingle) {
                return this;
            }

            String[] strs = new String[cellCount()];
            int i = cellCount() - 1;
            StrCell cell = this;
            while (cell != null) {
                strs[i] = cell.back();
                cell = cell.front();
                -- i;
            }

            StringBuilder sb = new StringBuilder();
            for (String s : strs) {
                sb.append(s);
            }
            return new StrCell(sb.toString());
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
