package org.kink_lang.kink;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.internal.function.ThrowingFunction2;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * The helper for {@linkplain TraceVal traces}.
 *
 * @see Vm#trace
 */
public final class TraceHelper {

    /** The vm. */
    private final Vm vm;

    /** The SNIP trace. */
    private TraceVal snip;

    /** The empty trace. */
    private TraceVal empty;

    /** Shared vars of trace vals. */
    SharedVars sharedVars;

    /**
     * Constructs the helper.
     */
    TraceHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a trace with a sym.
     *
     * @param sym the sym.
     * @return a trace with the sym.
     */
    public TraceVal of(String sym) {
        return new TraceVal(vm, sym, vm.location.of("", "", 0), false);
    }

    /**
     * Returns a trace with a loc.
     *
     * @param loc the loc.
     * @return a trace with the loc.
     */
    public TraceVal of(LocationVal loc) {
        return new TraceVal(vm, "", loc, false);
    }

    /**
     * Returns a trace with a sym.
     *
     * @param sym the sym.
     * @param loc the loc.
     * @return a trace with the loc.
     */
    public TraceVal of(String sym, LocationVal loc) {
        return new TraceVal(vm, sym, loc, false);
    }

    /**
     * Returns a trace with no sym nand loc, on the tail.
     *
     * @return a trace with no sym nand loc, on the tail.
     */
    public TraceVal snip() {
        return this.snip;
    }

    /**
     * Returns a trace with no sym nand loc, and not on the tail.
     *
     * @return a trace with no sym nand loc, and not on the tail.
     */
    public TraceVal empty() {
        return this.empty;
    }

    /**
     * Initializes the helper.
     */
    void init() {
        this.snip = new TraceVal(vm, "", vm.location.of("", "", 0), true);
        this.empty = new TraceVal(vm, "", vm.location.of("", "", 0), false);
        Map<Integer, Val> vars = new HashMap<>();
        addUnaryOp(vars, "Trace", "snip?", (c, t) -> vm.bool.of(t.isSnip()));
        addUnaryOp(vars, "Trace", "sym", this::symMethod);
        addUnaryOp(vars, "Trace", "location", this::locationMethod);
        addUnaryOp(vars, "Trace", "tail?", (c, trace) -> vm.bool.of(trace.isTail()));
        addUnaryOp(vars, "Trace", "on_tail", this::onTailMethod);
        vars.put(vm.sym.handleFor("op_eq"), vm.fun.make("Trace.op_eq").take(1).action(
                    c -> vm.bool.of(c.recv().equals(c.arg(0)))));
        addUnaryOp(vars, "Trace", "repr", this::reprMethod);
        addUnaryOp(vars, "Trace", "desc", (c, trace) -> vm.str.of(trace.desc()));
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Trace.sym {{{1

    /**
     * Implementation of Trace.sym.
     */
    private HostResult symMethod(HostContext c, TraceVal trace) {
        return trace.emptySym() ? vm.str.of("") : vm.str.of(trace.sym());
    }

    // }}}1

    // Trace.loc {{{1

    /**
     * Implementation of Trace.location.
     */
    private HostResult locationMethod(HostContext c, TraceVal trace) {
        return trace.emptyLoc() ? vm.location.of("", "", 0) : trace.location();
    }

    // }}}1

    // Trace.on_tail {{{

    /**
     * Implementation of Trace.on_tail.
     */
    private HostResult onTailMethod(HostContext c, TraceVal trace) {
        return trace.onTail();
    }

    // }}}

    // Trace.repr {{{1

    /**
     * Implementation of Trace.repr.
     */
    private HostResult reprMethod(HostContext c, TraceVal trace) {
        return c.call(vm.graph.format("(trace sym={} {} tail?={})",
                    vm.graph.of(vm.str.of(trace.sym())),
                    vm.graph.repr(trace.location()),
                    vm.graph.of(vm.str.of(Boolean.toString(trace.isTail())))));
    }

    // }}}1

    /**
     * Adds an unary operator method fun.
     */
    private void addUnaryOp(Map<Integer, Val> vars,
            String recvDesc,
            String name,
            ThrowingFunction2<CallContext, TraceVal, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s", recvDesc, name);
        vars.put(vm.sym.handleFor(name), vm.fun.make(desc).take(0).action(c -> {
            Val recv = c.recv();
            if (! (recv instanceof TraceVal trace)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be trace, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(recv)));
            }
            return action.apply(c, trace);
        }));
    }

}

// vim: et sw=4 sts=4 fdm=marker
