package org.kink_lang.kink;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * The helper for {@linkplain VarrefVal varref vals}.
 *
 * @see Vm#varref
 */
public class VarrefHelper {

    /** The vm. */
    private final Vm vm;

    /** The sym handle of "pass_arg". */
    private int passArgHandle;

    /** The sym handle of "pass_nothing". */
    private int passNothingHandle;

    /** The sym handle of "pass_rest". */
    private int passRestHandle;

    /** The sym handle of "repr". */
    private int reprHandle;

    /** Shared vars of varref vals. */
    SharedVars sharedVars;

    /**
     * Constructs the helper.
     */
    VarrefHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Returns a varref.
     *
     * @param owner the owner of the var.
     * @param symHandle the handle of the sym of the var.
     * @return a varref.
     */
    public VarrefVal of(Val owner, int symHandle) {
        return new VarrefVal(owner, symHandle);
    }

    /**
     * Returns a varref.
     *
     * @param owner the owner of the var.
     * @param sym the sym of the var.
     * @return a varref.
     */
    public VarrefVal of(Val owner, String sym) {
        return new VarrefVal(owner, vm.sym.handleFor(sym));
    }

    /**
     * Initializes the helper.
     */
    void init() {
        this.passArgHandle = vm.sym.handleFor("pass_arg");
        this.passNothingHandle = vm.sym.handleFor("pass_nothing");
        this.passRestHandle = vm.sym.handleFor("pass_rest");
        this.reprHandle = vm.sym.handleFor("repr");

        Map<Integer, Val> vars = new HashMap<>();
        addMethod(vars, "Varref", "owner", "", 0, (c, desc, varref) -> varref.owner());
        addMethod(vars, "Varref", "sym", "", 0, (c, desc, varref) -> vm.str.of(varref.sym()));
        addMethod(vars, "Varref", "op_store", "(Target)", 1, this::opStoreMethod);
        addMethod(vars, "Varref", "load", "", 0, this::loadMethod);
        addMethod(vars, "Varref", "have_val?", "", 0, this::haveValPMethod);
        addMethod(vars, "Varref", "require_from", "(Prefix)", 1, this::requireFromMethod);
        addMethod(vars, "Varref", "rest", "", 0, this::restMethod);
        addMethod(vars, "Varref", "opt", "", 0, this::optMethod);
        addMethod(vars, "Varref", "repr", "", 0, this::reprMethod);
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Varref.op_store(Val) {{{1

    /**
     * Implementation of Varref.op_store.
     */
    private HostResult opStoreMethod(CallContext c, String desc, VarrefVal varref) {
        varref.owner().setVar(varref.symHandle(), c.arg(0));
        return vm.nada;
    }

    // }}}1
    // Varref.load {{{1

    /**
     * Implementation of Varref.load.
     */
    private HostResult loadMethod(HostContext c, String desc, VarrefVal varref) {
        Val target = varref.owner().getVar(varref.symHandle());
        return target != null
            ? target
            : c.raise(String.format(Locale.ROOT, "%s: :%s is empty", desc, varref.sym()));
    }

    // }}}1

    // Varref.have_val? {{{1

    /**
     * Implementation of Varref.have_val?.
     */
    private HostResult haveValPMethod(HostContext c, String desc, VarrefVal varref) {
        Val owner = varref.owner();
        int symHandle = varref.symHandle();
        return vm.bool.of(owner.hasVar(symHandle));
    }

    // }}}1

    // Varref.require_from(Prefix) {{{1

    /**
     * Implementation of Varref.require_from(Prefix).
     */
    private HostResult requireFromMethod(
            CallContext c,
            String desc,
            VarrefVal varref) throws Throwable {
        if (! (c.arg(0) instanceof StrVal prefix)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Prefix must be a str, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        String modName = prefix.string() + varref.sym();
        var successCont = vm.fun.make().take(1).action(cc -> {
            var mod = cc.arg(0);
            varref.owner().setVar(varref.symHandle(), mod);
            return vm.nada;
        });
        return vm.mod.require(c, modName, successCont, null, null);
    }

    // }}}1

    // Varref.rest {{{1

    /**
     * Implementation of Varref.rest.
     */
    private HostResult restMethod(HostContext c, String desc, VarrefVal varref) {
        return makeRestParam(varref.owner(), varref.symHandle());
    }

    /**
     * Makes a rest_param.
     */
    private Val makeRestParam(Val owner, int symHandle) {
        Val param = vm.newVal();
        String prefix = "Rest_param.pass_rest(Rest_args)";
        param.setVar(passRestHandle, vm.fun.make(prefix).take(1).action(c -> {
            owner.setVar(symHandle, c.arg(0));
            return vm.nada;
        }));
        param.setVar(reprHandle, vm.fun.make("Rest_param.repr").take(0).action(c -> {
            return vm.str.of(String.format(Locale.ROOT, ":%s.rest", vm.sym.symFor(symHandle)));
        }));
        return param;
    }

    // }}}1
    // Varref.opt {{{1

    /**
     * Implementation of Varref.opt.
     */
    private HostResult optMethod(HostContext c, String desc, VarrefVal varref) {
        return makeOptParam(varref.owner(), varref.symHandle());
    }

    /**
     * Makes an opt_param.
     */
    private Val makeOptParam(Val owner, int symHandle) {
        Val param = vm.newVal();
        param.setVar(passArgHandle, vm.fun.make("Opt_param.pass_arg(Arg)").take(1).action(c -> {
            owner.setVar(symHandle, vm.vec.of(c.arg(0)));
            return vm.nada;
        }));
        param.setVar(passNothingHandle, vm.fun.make("Opt_param.pass_nothing").take(0).action(c -> {
            owner.setVar(symHandle, vm.vec.of());
            return vm.nada;
        }));
        param.setVar(reprHandle, vm.fun.make("Opt_param.repr").take(0).action(c -> {
            return vm.str.of(String.format(Locale.ROOT, ":%s.opt", vm.sym.symFor(symHandle)));
        }));
        return param;
    }

    // }}}1
    // Varref.repr {{{1

    /**
     * Implementation of Varref.repr.
     */
    private HostResult reprMethod(HostContext c, String desc, VarrefVal varref) {
        return vm.str.of(":" + varref.sym());
    }

    // }}}1

    /**
     * Add a method to vars.
     */
    private void addMethod(
            Map<Integer, Val> vars,
            String recvDesc,
            String sym,
            String argsDesc,
            int arity,
            ThrowingFunction3<CallContext, String, VarrefVal, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s%s", recvDesc, sym, argsDesc);
        var fun = vm.fun.make(desc).take(arity).action(c -> {
            if (! (c.recv() instanceof VarrefVal recv)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be varref, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(c.recv())));
            }
            return action.apply(c, desc, recv);
        });
        vars.put(vm.sym.handleFor(sym), fun);
    }

}

// vim: et sw=4 sts=4 fdm=marker
