package org.kink_lang.kink;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;

import org.kink_lang.kink.internal.function.ThrowingFunction3;
import org.kink_lang.kink.internal.function.ThrowingFunction4;
import org.kink_lang.kink.hostfun.HostFunReaction;
import org.kink_lang.kink.hostfun.HostResult;
import org.kink_lang.kink.hostfun.graph.GraphNode;
import org.kink_lang.kink.hostfun.HostContext;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.internal.intrinsicsupport.ArgsSupport;
import org.kink_lang.kink.internal.vec.VecInternal;
import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.num.NumOperations;

/**
 * The helper of {@linkplain VecVal vec vals}.
 *
 * @see Vm#vec
 */
public class VecHelper {

    /** The vm. */
    private final Vm vm;

    /** The empty vecInternal. */
    private VecInternal empty;

    /** The sym handle of op_store. */
    private int storeHandle;

    /** The sym handle of pass_arg. */
    private int passArgHandle;

    /** The sym handle of pass_nothing. */
    private int passNothingHandle;

    /** The sym handle of pass_rest. */
    private int passRestHandle;

    /** The sym handle of op_eq. */
    private int opEqHandle;

    /** The sym handle of op_lt. */
    private int opLtHandle;

    /** The sym handle of repr_vec. */
    private int reprVecHandle;

    /** The sym handle of each. */
    private int eachHandle;

    /** The sym handle of sort. */
    private int sortHandle;

    /** The sym handle of all?. */
    private int allP;

    /** The sym handle of any?. */
    private int anyP;

    /** The sym handle of have?. */
    private int haveP;

    /** The sym handle of new. */
    private int newHandle;

    /** The fun {(:X :Y) X &lt; Y }. */
    private FunVal defaultPrecedesFun;

    /** Trait of vecs. */
    SharedVars sharedVars;

    /**
     * Constructs the helper.
     */
    VecHelper(Vm vm) {
        this.vm = vm;
    }

    /**
     * Initializes the helper.
     */
    void init() {
        this.empty = VecInternal.of(vm, new Val[0], 0, 0);
        this.storeHandle = vm.sym.handleFor("op_store");
        this.passArgHandle = vm.sym.handleFor("pass_arg");
        this.passNothingHandle = vm.sym.handleFor("pass_nothing");
        this.passRestHandle = vm.sym.handleFor("pass_rest");
        this.opEqHandle = vm.sym.handleFor("op_eq");
        this.opLtHandle = vm.sym.handleFor("op_lt");
        this.reprVecHandle = vm.sym.handleFor("repr_vec");
        this.eachHandle = vm.sym.handleFor("each");
        this.sortHandle = vm.sym.handleFor("sort");
        this.allP = vm.sym.handleFor("all?");
        this.anyP = vm.sym.handleFor("any?");
        this.haveP = vm.sym.handleFor("have?");
        this.newHandle = vm.sym.handleFor("new");

        this.defaultPrecedesFun = vm.fun.make("(compare)")
            .take(2).action(c -> c.call(c.arg(0), opLtHandle).args(c.arg(1)));

        Map<Integer, Val> vars = new HashMap<>();
        addMethod(vars, "Vec", "size", "", 0,
                (c, vec, desc) -> vm.num.of(vec.vecInternal().size()));
        addMethod(vars, "Vec", "empty?", "", 0,
                (c, vec, desc) -> vm.bool.of(vec.vecInternal().size() == 0));
        addMethod(vars, "Vec", "get", "(Ind)", 1, this::getMethod);
        addMethod(vars, "Vec", "front", "", 0, this::frontMethod);
        addMethod(vars, "Vec", "back", "", 0, this::backMethod);
        addMethod(vars, "Vec", "set", "(Ind Elem)", 2, this::setMethod);
        addMethod(vars, "Vec", "dup", "", 0, this::dupMethod);
        addMethod(vars, "Vec", "rev", "", 0, this::revMethod);
        addTakeDropMethod(vars, "take_front", 0, 0, 1, 0);
        addTakeDropMethod(vars, "take_back", -1, 1, 0, 1);
        addTakeDropMethod(vars, "drop_front", 1, 0, 0, 1);
        addTakeDropMethod(vars, "drop_back", 0, 0, -1, 1);
        addMethod(vars, "Vec", "slice", "(From_pos To_pos)", 2, this::sliceMethod);
        addMethod(vars, "Vec", "clear", "", 0, this::clearMethod);
        addMethod(vars, "Vec", "clear_slice", "(From_pos To_pos)", 2, this::clearSliceMethod);
        addMethod(vars, "Vec", "push_front", "(Elem)", 1, this::pushFront);
        addMethod(vars, "Vec", "push_back", "(Elem)", 1, this::pushBack);
        addMethod(vars, "Vec", "push_at", "(Ind Elem)", 2, this::pushAt);
        addMethod(vars, "Vec", "push_each_front", "(Eacher)", 1, this::pushEachFront);
        addMethod(vars, "Vec", "push_each_back", "(Eacher)", 1, this::pushEachBack);
        addMethod(vars, "Vec", "push_each_at", "(Ind Eacher)", 2, this::pushEachAt);
        addMethod(vars, "Vec", "pop_front", "", 0, this::popFrontMethod);
        addMethod(vars, "Vec", "pop_back", "", 0, this::popBackMethod);
        addMethod(vars, "Vec", "pop_at", "(Ind)", 1, this::popAtMethod);
        addMethod(vars, "Vec", "concat", "", 0, this::concatMethod);
        addMethod(vars, "Vec", "chunk", "(Chunk_size)", 1, this::chunkMethod);
        addMethod(vars, "Vec", "just", "", 0, this::justMethod);
        addMethod(vars, "Vec", "just_or", "($cont_empty)", 1, this::justOrMethod);
        addMethod(vars, "Vec", "with_just_or", "($cont_just, $cont_empty)", 2,
                this::withJustOrMethod);
        addMethod(vars, "Vec", "each", "($proc_elem)", 1, this::eachMethod);
        addMethod(vars, "Vec", "map", "($trans)", 1, this::mapMethod);
        addMethod(vars, "Vec", "concat_map", "($trans_to_vec)", 1, this::concatMapMethod);
        addMethod(vars, "Vec", "filter", "($match?)", 1, this::filterMethod);
        addMethod(vars, "Vec", "count", "($match?)", 1, this::countMethod);
        addMethod(vars, "Vec", "fold", "(Init $combine)", 2, this::foldMethod);
        addMethod(vars, "Vec", "reduce", "($combine)", 1, this::reduceMethod);
        addMethod(vars, "Vec", "scan", "(Init $combine)", 2, this::scanMethod);
        addMethod(vars, "Vec", "scan_inside", "($combine)", 1, this::scanInsideMethod);
        addMethod(vars, "Vec", "take_while", "($match?)", 1, this::takeWhileMethod);
        addMethod(vars, "Vec", "drop_while", "($match?)", 1, this::dropWhileMethod);
        addMethod(vars, "Vec", "all?", "($match?)", 1, this::allPMethod);
        addMethod(vars, "Vec", "any?", "($match?)", 1, this::anyPMethod);
        addMethod(vars, "Vec", "have?", "(Target)", 1, this::havePMethod);
        addMethod(vars, "Vec", "have_all?", "(For_aller)", 1, this::haveAllPMethod);
        addMethod(vars, "Vec", "have_any?", "(For_anyer)", 1, this::haveAnyPMethod);
        addMethod(vars, "Vec", "search", "(Min_ind $accept?)", 2, this::searchMethod);
        vars.put(vm.sym.handleFor("sort"),
                vm.fun.make("Vec.sort($precede?)").takeMinMax(0, 1).action(this::sortMethod));
        addMethod(vars, "Vec", "iter", "", 0, this::iterMethod);
        addMethod(vars, "Vec", "iter_from", "(Min_ind)", 1, this::iterFromMethod);
        addBinOp(vars, "Vec", "op_add", "Arg_vec", this::opAddMethod);
        addMethod(vars, "Vec", "op_store", "(Rhs)", 1, this::opStoreMethod);
        addMethod(vars, "Vec", "op_mul", "(Count)", 1, this::opMulMethod);
        addBinOp(vars, "Vec", "op_eq", "Arg_vec", this::opEqMethod);
        addBinOp(vars, "Vec", "op_lt", "Arg_vec", this::opLtMethod);
        addMethod(vars, "Vec", "repr", "", 0, this::reprMethod);
        this.sharedVars = vm.sharedVars.of(vars);
    }

    // Vec.get {{{1

    /**
     * Implementation of Vec.get.
     */
    private HostResult getMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(c.arg(0), size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        return vecInternal.get(ind);
    }

    // }}}1

    // Vec.front {{{1

    /**
     * Implementation of Vec.front.
     */
    private HostResult frontMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise(desc + ": vec is empty");
        }
        return vecInternal.get(0);
    }

    // }}}1

    // Vec.back {{{1

    /**
     * Implementation of Vec.back.
     */
    private HostResult backMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise(desc + ": vec is empty");
        }
        return vecInternal.get(size - 1);
    }

    // }}}1

    // Vec.set {{{1

    /**
     * Implementation of Vec.set.
     */
    private HostResult setMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(c.arg(0), size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        vec.setVecInternal(vecInternal.set(ind, c.arg(1)));
        return vm.nada;
    }

    // }}}1

    // Vec.dup {{{1

    /**
     * Implementation of Vec.dup.
     */
    private HostResult dupMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        return new VecVal(vm, vecInternal.copy());
    }

    // }}}1

    // Vec.rev {{{1

    /**
     * Implementation of Vec.rev.
     */
    private HostResult revMethod(CallContext c, VecVal vec, String desc) {
        VecInternal src = vec.vecInternal();
        int size = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, size);
        for (int i = 0; i < size; ++ i) {
            dest = dest.append(src.get(size - i - 1));
        }
        return new VecVal(vm, dest);
    }

    // }}}1

    // Vec.take_front, take_back, drop_front, drop_back {{{1

    /**
     * Adds take_xxx/drop_xxx method to vars.
     */
    private void addTakeDropMethod(
            Map<Integer, Val> vars,
            String sym,
            int fromNumCoef, int fromSizeCoef, int toNumCoef, int toSizeCoef) {
        addMethod(vars, "Vec", sym, "(N)", 1, (c, vec, desc) -> {
            VecInternal vecInternal = vec.vecInternal();
            int size = vecInternal.size();

            Val numVal = c.arg(0);
            int num = NumOperations.getPosIndex(numVal, size);
            if (num < 0) {
                return c.call(vm.graph.raiseFormat(
                            "{}: N must be an int num in [0, {}], but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.num.of(size)),
                            vm.graph.repr(numVal)));
            }
            int from = fromNumCoef * num + fromSizeCoef * size;
            int to = toNumCoef * num + toSizeCoef * size;
            return new VecVal(vm, vecInternal.copyRange(from, to));
        });
    }

    // }}}1

    // Vec.slice {{{1

    /**
     * Implementation of Vec.slice.
     */
    private HostResult sliceMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        GraphNode error = checkRange(desc, size, c.arg(0), c.arg(1));
        if (error != null) {
            return c.call(error);
        }

        BigDecimal from = ((NumVal) c.arg(0)).bigDecimal();
        BigDecimal to = ((NumVal) c.arg(1)).bigDecimal();
        return new VecVal(vm, vecInternal.copyRange(from.intValue(), to.intValue()));
    }

    // }}}1

    // Vec.clear {{{1

    /**
     * Implementation of Vec.clear.
     */
    private HostResult clearMethod(CallContext c, VecVal vec, String desc) {
        vec.setVecInternal(this.empty);
        return vm.nada;
    }

    // }}}1

    // Vec.clear_slice {{{1

    /**
     * Implementation of Vec.clear_slice.
     */
    private HostResult clearSliceMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        GraphNode error = checkRange(desc, size, c.arg(0), c.arg(1));
        if (error != null) {
            return c.call(error);
        }

        BigDecimal from = ((NumVal) c.arg(0)).bigDecimal();
        BigDecimal to = ((NumVal) c.arg(1)).bigDecimal();
        vec.setVecInternal(vecInternal.removeRange(from.intValue(), to.intValue()));
        return vm.nada;
    }

    // }}}1

    // Vec.push_front(Elem) {{{1

    /**
     * Implementation of Vec.push_front.
     */
    private HostResult pushFront(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        vec.setVecInternal(vecInternal.insert(0, c.arg(0)));
        return vm.nada;
    }

    // }}}1

    // Vec.push_back(Elem) {{{1

    /**
     * Implementation of Vec.push_back.
     */
    private HostResult pushBack(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        vec.setVecInternal(vecInternal.append(c.arg(0)));
        return vm.nada;
    }

    // }}}1

    // Vec.push_at(Ind Elem) {{{1

    /**
     * Implementation of Vec.push_at.
     */
    private HostResult pushAt(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getPosIndex(c.arg(0), size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: required int num in [0, {}] for Ind, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        vecInternal = vecInternal.insert(ind, c.arg(1));
        vec.setVecInternal(vecInternal);
        return vm.nada;
    }

    // }}}1

    // Vec.push_each_xxx {{{1

    /**
     * Implementation of Vec.push_each_front.
     */
    private HostResult pushEachFront(CallContext c, VecVal vec, String desc) {
        Val eacher = c.arg(0);
        if (eacher instanceof VecVal eacherVec) {
            VecInternal vecInternal = vec.vecInternal();
            vec.setVecInternal(vecInternal.insertAll(0, eacherVec.vecInternal()));
            return vm.nada;
        } else if (eacher.hasVar(eachHandle)) {
            return c.call(eacher, eachHandle).args(pushEachAtConsume(vec, 0));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "{}: Eacher must have .each, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(eacher)));
        }
    }

    /**
     * Implementation of Vec.push_each_back.
     */
    private HostResult pushEachBack(CallContext c, VecVal vec, String desc) {
        Val eacher = c.arg(0);
        if (eacher instanceof VecVal eacherVec) {
            VecInternal vecInternal = vec.vecInternal();
            int size = vecInternal.size();
            vec.setVecInternal(vecInternal.insertAll(size, eacherVec.vecInternal()));
            return vm.nada;
        } else if (eacher.hasVar(eachHandle)) {
            return c.call(eacher, eachHandle).args(pushEachBackConsume(vec));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "{}: Eacher must have .each, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(eacher)));
        }
    }

    /**
     * Called by Adder.each via Vec.push_each_back.
     */
    private FunVal pushEachBackConsume(VecVal vec) {
        return vm.fun.make("Vec.push_each_back-call").take(1).action(c -> {
            VecInternal vecInternal = vec.vecInternal();
            vec.setVecInternal(vecInternal.append(c.arg(0)));
            return vm.nada;
        });
    }

    /**
     * Implementation of Vec.push_each_at.
     */
    private HostResult pushEachAt(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getPosIndex(c.arg(0), size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind must be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }

        Val adderVal = c.arg(1);
        if (adderVal instanceof VecVal adder) {
            vec.setVecInternal(vecInternal.insertAll(ind, adder.vecInternal()));
            return vm.nada;
        } else if (adderVal.hasVar(eachHandle)) {
            return c.call(adderVal, eachHandle).args(pushEachAtConsume(vec, ind));
        } else {
            return c.call(vm.graph.raiseFormat(
                        "{}: Eacher must have .each, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(adderVal)));
        }
    }

    /**
     * Called by Adder.each via Vec.push_each_at and .push_each_front.
     */
    private FunVal pushEachAtConsume(VecVal vec, int initialInd) {
        AtomicInteger indAtomic = new AtomicInteger(initialInd);
        return vm.fun.make("Vec.push_each_at-consume").take(1).action(c -> {
            int ind = indAtomic.getAndIncrement();
            VecInternal vecInternal = vec.vecInternal();
            int size = vecInternal.size();
            if (ind > size) {
                return c.raise("Vec.push_each_at-consume: vec was modified during loop");
            }
            vec.setVecInternal(vecInternal.insert(ind, c.arg(0)));
            return vm.nada;
        });
    }

    // }}}1

    // Vec.pop_front {{{1

    /**
     * Implementation of Vec.pop_front.
     */
    private HostResult popFrontMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise(desc + ": could not pop from empty vec");
        }
        Val result = vecInternal.get(0);
        vec.setVecInternal(vecInternal.remove(0));
        return result;
    }

    // }}}1

    // Vec.pop_back {{{1

    /**
     * Implementation of Vec.pop_back.
     */
    private HostResult popBackMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return c.raise(desc + ": could not pop from empty vec");
        }
        Val result = vecInternal.get(size - 1);
        vec.setVecInternal(vecInternal.remove(size - 1));
        return result;
    }

    // }}}1

    // Vec.pop_at {{{1

    /**
     * Implementation of Vec.pop_at.
     */
    private HostResult popAtMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int ind = NumOperations.getElemIndex(c.arg(0), size);
        if (ind < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Ind msut be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        Val result = vecInternal.get(ind);
        vec.setVecInternal(vecInternal.remove(ind));
        return result;
    }

    // }}}1

    // Vec.concat {{{1

    /**
     * Implementation of Vec.concat.
     */
    private HostResult concatMethod(CallContext c, VecVal vecs, String desc) {
        VecInternal src = vecs.vecInternal();
        int srcSize = src.size();
        VecInternal result = this.empty;
        for (int i = 0; i < srcSize; ++ i) {
            Val elem = src.get(i);
            if (! (elem instanceof VecVal elemVec)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: Vec must be a vec of vecs, but the #{} elem was {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.num.of(i)),
                            vm.graph.repr(elem)));
            }
            result = result.insertAll(result.size(), elemVec.vecInternal());
        }
        return new VecVal(vm, result);
    }

    // }}}1

    // Vec.chunk(Chunk_size) {{{1

    /**
     * Implementation of Vec.chunk(Chunk_size).
     */
    private HostResult chunkMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int vecSize = vecInternal.size();
        Val chunkSizeVal = c.arg(0);
        if (! (chunkSizeVal instanceof NumVal chunkSizeNum)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Chunk_size must be an int num, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(chunkSizeVal)));
        }
        BigDecimal chunkSizeDec = chunkSizeNum.bigDecimal();
        if (vecInternal.size() == 0) {
            boolean sizeIsPositiveInt = chunkSizeDec.signum() > 0
                && chunkSizeDec.remainder(BigDecimal.ONE).signum() == 0;
            if (! sizeIsPositiveInt) {
                return c.call(vm.graph.raiseFormat(
                            "{}: Chunk_size must be an int num, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(chunkSizeVal)));
            }
            return vm.vec.of();
        }

        int chunkSize = NumOperations.getPosIndex(chunkSizeDec, vecSize);
        if (chunkSize < 1 || vecSize % chunkSize != 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Chunk_size must be a positive divisor of {}, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(vecSize)),
                        vm.graph.repr(chunkSizeVal)));
        }
        int numChunks = vecSize / chunkSize;
        VecInternal result = VecInternal.ofCapa(vm, numChunks);
        for (int i = 0; i < numChunks; ++ i) {
            int from = i * chunkSize;
            int to = from + chunkSize;
            VecInternal chunk = vecInternal.copyRange(from, to);
            result = result.append(new VecVal(vm, chunk));
        }
        return new VecVal(vm, result);
    }

    // }}}1

    // Vec.just {{{1

    /**
     * Implementation of Vec.just.
     */
    private HostResult justMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size != 1) {
            String msg = String.format(Locale.ROOT,
                    "%s: Vec must be a 1-elem vec, but the number of elems was %d", desc, size);
            return c.raise(msg);
        }
        return vecInternal.get(0);
    }

    // }}}1

    // Vec.just_or {{{1

    /**
     * Implementation of Vec.just_or.
     */
    private HostResult justOrMethod(CallContext c, VecVal vec, String desc) {
        var contEmptyVal = c.arg(0);
        if (! (contEmptyVal instanceof FunVal contEmpty)) {
            return c.call(vm.graph.raiseFormat("{}: $cont_empty must be fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(contEmptyVal)));
        }

        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        switch (size) {
            case 0: {
                return c.call(contEmpty);
            }
            case 1: {
                return vecInternal.get(0);
            }
            default: {
                String msg = String.format(Locale.ROOT,
                        "%s: Vec must be 0-elem or 1-elem vec, but the number of elems was %d",
                        desc, size);
                return c.raise(msg);
            }
        }
    }

    // }}}1

    // Vec.with_just_or {{{1

    /**
     * Implementation of Vec.with_just_or.
     */
    private HostResult withJustOrMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size > 1) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Vec must be 0-elem or 1-elem vec, but Vec contains {} elements",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size))));
        }

        Val contJustVal = c.arg(0);
        if (! (contJustVal instanceof FunVal contJust)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $cont_just must be fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(contJustVal)));
        }

        Val contEmptyVal = c.arg(1);
        if (! (contEmptyVal instanceof FunVal contEmpty)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $cont_empty must be fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(contEmptyVal)));
        }

        return size == 0
            ? c.call(contEmpty)
            : c.call(contJust).args(vecInternal.get(0));
    }

    // }}}1

    // Vec.each {{{1

    /**
     * Implementation of Vec.each($on_elem).
     */
    private HostResult eachMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal procElem)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $cont_elem must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        return size ==0
            ? vm.nada
            : eachLoop(c, vecInternal, 0, size, procElem);
    }

    /**
     * Iteration of Vec.each.
     */
    private HostResult eachLoop(
            HostContext c, VecInternal vecInternal, int ind, int size, FunVal onElem) {
        Val elem = vecInternal.get(ind);
        int nextInd = ind + 1;
        if (nextInd < size) {
            return c.call(onElem).args(elem)
                .on((cc, ignored) -> eachLoop(cc, vecInternal, nextInd, size, onElem));
        } else {
            return c.call(onElem).args(elem);
        }
    }

    // }}}1

    // Vec.map {{{1

    /**
     * Implementation of Vec.map.
     */
    private HostResult mapMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal trans)) {
            return c.call(vm.graph.raiseFormat("{}: $trans must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal src = vec.vecInternal();
        int size = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, size);
        return mapLoop(c, src, dest, 0, trans);
    }

    /**
     * Iteration of Vec.map.
     */
    private HostResult mapLoop(
            HostContext c, VecInternal src, VecInternal dest, int index, FunVal trans) {
        if (index >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(trans).args(src.get(index)).on((cc, result) -> mapLoop(
                    cc, src, dest.append(result), index + 1, trans));
    }

    // }}}1

    // Vec.concat_map {{{1

    /**
     * Implementation of Vec.concat_map.
     */
    private HostResult concatMapMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal trans)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $trans_to_vec must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal src = vec.vecInternal();
        VecInternal dest = this.empty;
        return concatMapLoop(c, src, dest, 0, trans, desc);
    }

    /**
     * Iteration of Vec.concat_map.
     */
    private HostResult concatMapLoop(
            HostContext c,
            VecInternal src, VecInternal dest,
            int srcIndex,
            FunVal trans,
            String desc) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(trans).args(src.get(srcIndex)).on((cc, resultVal) -> {
            if (! (resultVal instanceof VecVal result)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: expected vec for fun result, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(resultVal)));
            }
            VecInternal newDest = dest.insertAll(dest.size(), result.vecInternal());
            return concatMapLoop(cc, src, newDest, srcIndex + 1, trans, desc);
        });
    }

    // }}}1

    // Vec.filter {{{1

    /**
     * Implementation of Vec.filter.
     */
    private HostResult filterMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? msut be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal src = vec.vecInternal();
        VecInternal dest = this.empty;
        return filterLoop(c, src, dest, 0, match);
    }

    /**
     * Iteration of Vec.filter.
     */
    private HostResult filterLoop(
            HostContext c, VecInternal src, VecInternal dest, int srcIndex, FunVal pred) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        Val elem = src.get(srcIndex);
        return c.call(pred).args(elem).on((cc, bool) -> {
            if (! vm.bool.isBool(bool)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec.filter: expected bool for fun result, but got {}",
                            vm.graph.repr(bool)));
            }
            VecInternal newDest = bool == vm.bool.trueVal
                ? dest.append(elem)
                : dest;
            return filterLoop(cc, src, newDest, srcIndex + 1, pred);
        });
    }

    // }}}1

    // Vec.count {{{1

    /**
     * Implementation of Vec.count.
     */
    private HostResult countMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        return countLoop(c, vecInternal, 0, 0, match, desc);
    }

    /**
     * Iteration of Vec.count.
     */
    private HostResult countLoop(
            HostContext c, VecInternal vecInternal, int index, int num, FunVal match, String desc) {
        if (index >= vecInternal.size()) {
            return vm.num.of(num);
        }
        return c.call(match).args(vecInternal.get(index)).on((cc, bool) -> {
            if (! vm.bool.isBool(bool)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: expected bool for fun result, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(bool)));
            }
            int newNum = num + (bool == vm.bool.trueVal ? 1 : 0);
            return countLoop(cc, vecInternal, index + 1, newNum, match, desc);
        });
    }

    // }}}1

    // Vec.fold {{{1

    /**
     * Implementation of Vec.fold.
     */
    private HostResult foldMethod(CallContext c, VecVal vec, String desc) {
        var combineVal = c.arg(1);
        if (! (combineVal instanceof FunVal combine)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $combine must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(combineVal)));
        }
        VecInternal vecInternal = vec.vecInternal();
        return foldLoop(c, vecInternal, 0, c.arg(0), combine);
    }

    /**
     * Iteration of Vec.fold.
     */
    private HostResult foldLoop(
            HostContext c,
            VecInternal vecInternal,
            int index,
            Val accum,
            FunVal combine) {
        int size = vecInternal.size();
        if (index >= size) {
            return accum;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(combine).args(accum, elem)
                .on((cc, newAccum) -> foldLoop(cc, vecInternal, nextIndex, newAccum, combine));
        } else {
            return c.call(combine).args(accum, elem);
        }
    }

    // }}}1

    // Vec.reduce {{{1

    /**
     * Implementation of Vec.reduce.
     */
    private HostResult reduceMethod(CallContext c, VecVal vec, String desc) {
        var combineVal = c.arg(0);
        if (! (combineVal instanceof FunVal combine)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $combine must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(combineVal)));
        }
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        if (size == 0) {
            return vm.vec.of();
        } else {
            Val first = vecInternal.get(0);
            return reduceLoop(c, vecInternal, 1, first, combine);
        }
    }

    /**
     * Iteration of Vec.reduce.
     */
    private HostResult reduceLoop(
            HostContext c,
            VecInternal vecInternal,
            int index,
            Val accum,
            FunVal combine) {
        if (index >= vecInternal.size()) {
            return vm.vec.of(accum);
        }
        return c.call(combine).args(accum, vecInternal.get(index)).on((cc, newAccum) -> {
            return reduceLoop(cc, vecInternal, index + 1, newAccum, combine);
        });
    }

    // }}}1

    // Vec.scan, Vec.scan_inside {{{1

    /**
     * Implementation of Vec.scan.
     */
    private HostResult scanMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(1) instanceof FunVal combine)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $combine must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }
        VecInternal src = vec.vecInternal();
        int srcSize = src.size();
        VecInternal dest = VecInternal.ofCapa(vm, srcSize + 1);
        dest = dest.insert(0, c.arg(0));
        return scanLoop(c, src, dest, 0, c.arg(0), combine);
    }

    /**
     * Implementation of Vec.scan_inside.
     */
    private HostResult scanInsideMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal combine)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $combine must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal src = vec.vecInternal();
        int srcSize = src.size();
        if (srcSize == 0) {
            return vm.vec.of();
        }
        VecInternal dest = VecInternal.ofCapa(vm, srcSize);
        Val first = src.get(0);
        dest = dest.insert(0, first);
        return scanLoop(c, src, dest, 1, first, combine);
    }

    /**
     * Iteration of Vec.scan.
     */
    private HostResult scanLoop(
            HostContext c,
            VecInternal src, VecInternal dest,
            int srcIndex, Val accum, FunVal combine) {
        if (srcIndex >= src.size()) {
            return new VecVal(vm, dest);
        }
        return c.call(combine).args(accum, src.get(srcIndex)).on((cc, newAccum) -> {
            VecInternal newDest = dest.append(newAccum);
            return scanLoop(cc, src, newDest, srcIndex + 1, newAccum, combine);
        });
    }

    // }}}1

    // Vec.take_while {{{1

    /**
     * Implementation of Vec.take_while.
     */
    private HostResult takeWhileMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        return takeWhileLoop(c, vecInternal, 0, match, desc);
    }

    /**
     * Iteration of Vec.take_while.
     */
    private HostResult takeWhileLoop(
            HostContext c, VecInternal src, int srcIndex, FunVal match, String desc) {
        int srcSize = src.size();
        if (srcIndex >= srcSize) {
            return new VecVal(vm, src.copy());
        }

        Val elem = src.get(srcIndex);
        return c.call(match).args(elem).on((cc, result) -> {
            if (! vm.bool.isBool(result)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: $match? must return bool, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(result)));
            }
            if (result == vm.bool.falseVal) {
                return new VecVal(vm, src.copyRange(0, srcIndex));
            }
            return takeWhileLoop(cc, src, srcIndex + 1, match, desc);
        });
    }

    // }}}1

    // Vec.drop_while {{{1

    /**
     * Implementation of Vec.drop_while.
     */
    private HostResult dropWhileMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        return dropWhileLoop(c, vecInternal, 0, vecInternal.size(), match, desc);
    }

    /**
     * Iteration of Vec.drop_while.
     */
    private HostResult dropWhileLoop(
            HostContext c, VecInternal src, int srcIndex, int srcSize, FunVal match, String desc) {
        if (srcIndex >= srcSize) {
            return vm.vec.of();
        }

        Val elem = src.get(srcIndex);
        return c.call(match).args(elem).on((cc, result) -> {
            if (! vm.bool.isBool(result)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: $match? must return bool, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(result)));
            }
            if (result == vm.bool.falseVal) {
                return new VecVal(vm, src.copyRange(srcIndex, srcSize));
            }
            return dropWhileLoop(cc, src, srcIndex + 1, srcSize, match, desc);
        });
    }

    // }}}1

    // Vec.all? {{{1

    /**
     * Implementation of Vec.all?
     */
    private HostResult allPMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        return allPLoop(c, vecInternal, 0, match, desc);
    }

    /**
     * Iteration of Vec.all?.
     */
    private HostResult allPLoop(
            HostContext c, VecInternal vecInternal, int index, FunVal match, String desc) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.trueVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(match).args(elem).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "{}: $match? must return bool, but got {}",
                                vm.graph.of(vm.str.of(desc)),
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.falseVal) {
                    return vm.bool.falseVal;
                }
                return allPLoop(cc, vecInternal, nextIndex, match, desc);
            });
        } else {
            return c.call(match).args(elem);
        }
    }

    // }}}1

    // Vec.any? {{{1

    /**
     * Implementation of Vec.any?.
     */
    private HostResult anyPMethod(CallContext c, VecVal vec, String desc) {
        if (! (c.arg(0) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }
        VecInternal vecInternal = vec.vecInternal();
        return forAnyPLoop(c, vecInternal, 0, match, desc);
    }

    /**
     * Iteration of Vec.any?.
     */
    private HostResult forAnyPLoop(
            HostContext c, VecInternal vecInternal, int index, FunVal match, String desc) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.falseVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(match).args(elem).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "{}: $match? must return bool, but got {}",
                                vm.graph.of(vm.str.of(desc)),
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.trueVal) {
                    return vm.bool.trueVal;
                }
                return forAnyPLoop(cc, vecInternal, nextIndex, match, desc);
            });
        } else {
            return c.call(match).args(elem);
        }
    }

    // }}}1

    // Vec.have? {{{1

    /**
     * Implementation of Vec.have?.
     */
    private HostResult havePMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        return havePLoop(c, vecInternal, 0, c.arg(0), desc);
    }

    /**
     * Iteration of Vec.have?.
     */
    private HostResult havePLoop(
            HostContext c, VecInternal vecInternal, int index, Val target, String desc) {
        int size = vecInternal.size();
        if (index >= size) {
            return vm.bool.falseVal;
        }

        Val elem = vecInternal.get(index);
        int nextIndex = index + 1;
        if (nextIndex < size) {
            return c.call(target, this.opEqHandle).args(elem).on((cc, result) -> {
                if (! vm.bool.isBool(result)) {
                    return cc.call(vm.graph.raiseFormat(
                                "{}: .op_eq must return bool, but got {} for the #{} elem",
                                vm.graph.of(vm.str.of(desc)),
                                vm.graph.of(vm.num.of(index)),
                                vm.graph.repr(result)));
                }
                if (result == vm.bool.trueVal) {
                    return vm.bool.trueVal;
                }
                return havePLoop(cc, vecInternal, nextIndex, target, desc);
            });
        } else {
            return c.call(target, this.opEqHandle).args(elem);
        }
    }

    // }}}1

    // Vec.have_all? {{{1

    /**
     * Implementation of Vec.have_all?.
     */
    private HostResult haveAllPMethod(CallContext c, VecVal vec, String desc) {
        var accept = vm.fun.make().take(1).action(cc -> cc.call(vec, haveP).args(cc.arg(0)));
        return c.call(c.arg(0), allP).args(accept);
    }

    // }}}1

    // Vec.have_any? {{{1

    /**
     * Implementation of Vec.have_any?.
     */
    private HostResult haveAnyPMethod(CallContext c, VecVal vec, String desc) {
        var accept = vm.fun.make().take(1).action(cc -> cc.call(vec, haveP).args(cc.arg(0)));
        return c.call(c.arg(0), anyP).args(accept);
    }

    // }}}1

    // Vec.search(Min_ind $accept?) {{{1

    /**
     * Implementation of Vec.search.
     */
    private HostResult searchMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int minInd = NumOperations.getPosIndex(c.arg(0), size);
        if (minInd < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Min_ind must be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        if (! (c.arg(1) instanceof FunVal match)) {
            return c.call(vm.graph.raiseFormat(
                        "{}: $match? must be a fun, but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(1))));
        }
        return searchMethodLoop(c, vecInternal, minInd, match, size, desc);
    }

    /**
     * Loop of Vec.search.
     */
    private HostResult searchMethodLoop(
            HostContext c,
            VecInternal vecInternal,
            int ind,
            FunVal match,
            int size,
            String desc) {
        if (ind >= size) {
            return vm.vec.of();
        }

        Val elem = vecInternal.get(ind);
        return c.call(match).args(elem).on((cc, r) -> {
            if (! vm.bool.isBool(r)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: $match? must return a bool, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(r)));
            }
            if (r.equals(vm.bool.trueVal)) {
                return vm.vec.of(vm.num.of(ind));
            }
            return searchMethodLoop(cc, vecInternal, ind + 1, match, size, desc);
        });
    }

    // }}}1

    // Vec.sort($precede?) {{{1

    /**
     * Implementation of Vec.sort.
     */
    private HostResult sortMethod(CallContext c) {
        Val precedes = c.argCount() == 0
            ? this.defaultPrecedesFun
            : c.arg(0);
        return c.call("kink/_vec/SORT_VEC", sortHandle).args(c.recv(), precedes);
    }

    // }}}1
    // Vec.iter and .iter_from {{{1

    /**
     * Implementation of Vec.iter.
     */
    private HostResult iterMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        FunVal ifun = forwardIfun(desc + ".ifun", vecInternal, 0, size);
        return c.call("kink/iter/ITER", newHandle).args(ifun);
    }

    /**
     * Implementation of Vec.iter_from.
     */
    private HostResult iterFromMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        int size = vecInternal.size();
        int minInd = NumOperations.getPosIndex(c.arg(0), size);
        if (minInd < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Min_ind must be an int num in [0, {}], but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.repr(c.arg(0))));
        }
        FunVal ifun = forwardIfun(desc + ".ifun", vecInternal, minInd, size);
        return c.call("kink/iter/ITER", newHandle).args(ifun);
    }

    /**
     * Ifun for Vec.iter.
     */
    private FunVal forwardIfun(String prefix, VecInternal vecInternal, int ind, int size) {
        return vm.fun.make(prefix).take(2).action(c -> {
            Val proc = c.arg(0);
            if (! (proc instanceof FunVal)) {
                return c.call(vm.graph.raiseFormat("{}: $proc must be a fun, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(proc)));
            }

            Val fin = c.arg(1);
            if (! (fin instanceof FunVal)) {
                return c.call(vm.graph.raiseFormat("{}: $fin must be a fun, but got {}",
                            vm.graph.of(vm.str.of(prefix)),
                            vm.graph.repr(fin)));
            }
            if (ind < size) {
                Val head = vecInternal.get(ind);
                FunVal tail = forwardIfun(prefix, vecInternal, ind + 1, size);
                return c.call((FunVal) proc).args(head, tail);
            } else {
                return c.call((FunVal) fin);
            }
        });
    }

    // }}}1

    // Vec.op_add {{{1

    /**
     * Implementation of Vec.op_add.
     */
    private HostResult opAddMethod(CallContext c, VecVal left, VecVal right, String desc) {
        VecInternal result = left.vecInternal().copy();
        int leftSize = result.size();
        result = result.insertAll(leftSize, right.vecInternal());
        return new VecVal(vm, result);
    }

    // }}}1

    // Vec.op_store {{{1

    /**
     * Implementation of Vec.op_store.
     */
    private HostResult opStoreMethod(CallContext c, VecVal lhs, String desc) {
        Val arg = c.arg(0);
        if (! (arg instanceof VecVal rhs)) {
            return c.call(ArgsSupport.raiseNotVecRhs(vm, arg));
        }

        VecInternal lhsVi = lhs.vecInternal();
        VecInternal rhsVi = rhs.vecInternal();
        int lhsSize = lhsVi.size();
        int rhsSize = rhsVi.size();

        int mandatoryCount = getMandatoryParamsCount(lhsSize, lhsVi);
        int optCount = getOptParamsCount(lhsSize, lhsVi, mandatoryCount);

        int restInd = mandatoryCount + optCount;
        @Nullable Val restParam = restInd == lhsSize ? null : lhsVi.get(restInd);
        if (restInd < lhsSize - 1 || restInd == lhsSize - 1 && ! restParam.hasVar(passRestHandle)) {
            return c.call(vm.graph.raiseFormat("{}: unexpected lhs param: {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(restParam)));
        }

        if (rhsSize < mandatoryCount || restParam == null && rhsSize > mandatoryCount + optCount) {
            FunVal fun = ArgsSupport.wrongNumberOfArgs(vm, mandatoryCount, vm.graph.repr(lhs), rhs);
            return c.call(fun);
        }

        return opStoreMandatoryLoop(c, 0,
                lhsVi,
                rhsSize, rhsVi,
                mandatoryCount, optCount, restParam);
    }

    /**
     * Returns the count of mandatory params.
     */
    private int getMandatoryParamsCount(int lhsSize, VecInternal lhsVi) {
        for (int i = 0; i < lhsSize; ++ i) {
            Val param = lhsVi.get(i);
            if (! (param instanceof VarrefVal || param.hasVar(storeHandle))) {
                return i;
            }
        }
        return lhsSize;
    }

    /**
     * Returns the count of opt params.
     */
    private int getOptParamsCount(int lhsSize, VecInternal lhsVi, int offset) {
        for (int i = 0; offset + i < lhsSize; ++ i) {
            Val param = lhsVi.get(offset + i);
            if (! param.hasVar(passArgHandle)) {
                return i;
            }
        }
        return lhsSize - offset;
    }

    /**
     * Loop for mandatory args.
     */
    private HostResult opStoreMandatoryLoop(
            HostContext c,
            int ind,
            VecInternal lhs,
            int rhsSize,
            VecInternal rhs,
            int mandatoryCount, int optCount, @Nullable Val restParam) {
        while (ind < mandatoryCount) {
            Val param = lhs.get(ind);
            Val rval = rhs.get(ind);

            if (! (param instanceof VarrefVal)) {
                int nextInd = ind + 1;
                return c.call(param, storeHandle).args(rval)
                    .on((cc, r) -> opStoreMandatoryLoop(
                                cc, nextInd, lhs, rhsSize, rhs,
                                mandatoryCount, optCount, restParam));
            }

            VarrefVal varref = (VarrefVal) param;
            Val owner = varref.owner();
            int handle = varref.symHandle();
            owner.setVar(handle, rval);
            ++ ind;
        }
        return opStoreOptLoop(
                c, ind, lhs, rhsSize, rhs, mandatoryCount, optCount, restParam);
    }


    /**
     * Loop for optional args.
     */
    private HostResult opStoreOptLoop(
            HostContext c, int ind, VecInternal lhs, int rhsSize, VecInternal rhs,
            int mandatoryCount, int optCount, @Nullable Val restParam) {
        if (ind == mandatoryCount + optCount) {
            return opStoreRest(c, ind, rhsSize, rhs, restParam);
        }

        Val param = lhs.get(ind);
        HostFunReaction next = (cc, r) -> opStoreOptLoop(
                cc, ind + 1, lhs, rhsSize, rhs, mandatoryCount, optCount, restParam);
        return (ind < rhsSize
                ? c.call(param, passArgHandle).args(rhs.get(ind))
                : c.call(param, passNothingHandle)
                ).on(next);
    }

    /**
     * Handles the rest args.
     */
    private HostResult opStoreRest(
            HostContext c, int ind, int rhsSize, VecInternal rhs, @Nullable Val restParam) {
        if (restParam == null) {
            return vm.nada;
        }
        VecInternal restArgs = rhs.copyRange(Math.min(ind, rhsSize), rhsSize);
        VecVal restArgsVec = new VecVal(vm, restArgs);
        return c.call(restParam, passRestHandle).args(restArgsVec);
    }

    // }}}1

    // Vec.op_mul(Count) {{{1

    /**
     * Implementation of Vec.op_mul(Count).
     */
    private HostResult opMulMethod(CallContext c, VecVal vec, String desc) {
        VecInternal vecInternal = vec.vecInternal();
        BigInteger count = NumOperations.getExactBigInteger(c.arg(0));
        if (count == null || count.signum() < 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: Count must be an int num in [0, infinity), but got {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.repr(c.arg(0))));
        }

        int size = vecInternal.size();
        if (size == 0) {
            return vm.vec.of();
        }

        // Size of vecInternal may be changed concurrently,
        // thus the resultSizeEstimate is only an estimate.
        BigInteger resultSizeEstimate = count.multiply(BigInteger.valueOf(size));
        if (resultSizeEstimate.compareTo(BigInteger.valueOf(Integer.MAX_VALUE)) > 0) {
            return c.call(vm.graph.raiseFormat(
                        "{}: too long result: size {} * Count {} is {}",
                        vm.graph.of(vm.str.of(desc)),
                        vm.graph.of(vm.num.of(size)),
                        vm.graph.of(c.arg(0)),
                        vm.graph.of(vm.num.of(resultSizeEstimate))));
        }

        int countInt = count.intValueExact();
        VecInternal dest = VecInternal.ofCapa(vm, size * countInt);
        for (int i = 0; i < countInt; ++ i) {
            dest = dest.insertAll(dest.size(), vecInternal);
        }
        return new VecVal(vm, dest);
    }

    // }}}1

    // Vec == Another_vec and Vec != Another_vec {{{1

    /**
     * Implementation of Vec.op_eq.
     */
    private HostResult opEqMethod(CallContext c, VecVal left, VecVal right, String desc) {
        VecInternal lv = left.vecInternal();
        VecInternal rv = right.vecInternal();
        int size = lv.size();
        if (rv.size() != size) {
            return vm.bool.falseVal;
        }
        if (size == 0) {
            return vm.bool.trueVal;
        }
        return eqLoop(c, lv, rv, 0, size, desc);
    }

    /**
     * Loop of Vec.op_eq.
     */
    private HostResult eqLoop(
            HostContext c, VecInternal lv, VecInternal rv, int ind, int size, String desc) {
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        if (ind + 1 >= size) {
            return c.call(lelem, opEqHandle).args(relem);
        }
        return c.call(lelem, opEqHandle).args(relem).on((cc, r) -> {
            if (! vm.bool.isBool(r)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: .op_eq of an elem must return bool, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(r)));
            }
            if (r == vm.bool.falseVal) {
                return vm.bool.of(false);
            }
            return eqLoop(cc, lv, rv, ind + 1, size, desc);
        });
    }

    // }}}1
    // Vec < Another_vec Vec <= Another_vec Vec >= Another_vec Vec > Another_vec {{{1

    /**
     * Implementation of Vec.op_lt.
     */
    private HostResult opLtMethod(CallContext c, VecVal left, VecVal right, String desc) {
        VecInternal lv = left.vecInternal();
        VecInternal rv = right.vecInternal();
        int lsize = lv.size();
        int rsize = rv.size();
        return lsize == 0 ? vm.bool.of(rsize != 0)
            : rsize == 0 ? vm.bool.falseVal
            : lsize == rsize ? compareSameSize(c, lv, rv, 0, lsize)
            : compareDifferentSize(c, lv, rv, 0, Math.min(lsize, rsize), lsize < rsize, desc);
    }

    /**
     * Loop of op_lt when the size of the two vecs are different.
     */
    private HostResult compareDifferentSize(
            HostContext c,
            VecInternal lv, VecInternal rv,
            int ind,
            int size,
            boolean atEnd,
            String desc) {
        if (ind == size) {
            return vm.bool.of(atEnd);
        }
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        return c.call(lelem, opLtHandle).args(relem).on((cc, accept) -> {
            if (! vm.bool.isBool(accept)) {
                return cc.call(vm.graph.raiseFormat(
                            "{}: .op_lt of an elem must return bool, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(accept)));
            }
            if (accept.equals(vm.bool.trueVal)) {
                return vm.bool.trueVal;
            }
            return cc.call(relem, opLtHandle).args(lelem).on((c3, reject) -> {
                if (! vm.bool.isBool(reject)) {
                    return c3.call(vm.graph.raiseFormat(
                                "{}: .op_lt of an elem must return bool, but got {}",
                                vm.graph.of(vm.str.of(desc)),
                                vm.graph.repr(reject)));
                }
                if (reject.equals(vm.bool.trueVal)) {
                    return vm.bool.falseVal;
                }
                return compareDifferentSize(c3, lv, rv, ind + 1, size, atEnd, desc);
            });
        });
    }

    /**
     * Loop of op_lt when the size of the two vecs are the same.
     */
    private HostResult compareSameSize(
            HostContext c,
            VecInternal lv, VecInternal rv,
            int ind,
            int size) {
        Val lelem = lv.get(ind);
        Val relem = rv.get(ind);
        if (ind + 1 == size) {
            return c.call(lelem, opLtHandle).args(relem);
        }
        return c.call(lelem, opLtHandle).args(relem).on((cc, accept) -> {
            if (! vm.bool.isBool(accept)) {
                return cc.call(vm.graph.raiseFormat(
                            "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                            vm.graph.repr(accept)));
            }
            if (accept.equals(vm.bool.trueVal)) {
                return vm.bool.trueVal;
            }
            return cc.call(relem, opLtHandle).args(lelem).on((c3, reject) -> {
                if (! vm.bool.isBool(reject)) {
                    return c3.call(vm.graph.raiseFormat(
                                "Vec1.op_lt(Vec2): expected bool result from .op_lt, but got {}",
                                vm.graph.repr(reject)));
                }
                if (reject.equals(vm.bool.trueVal)) {
                    return vm.bool.falseVal;
                }
                return compareSameSize(c3, lv, rv, ind + 1, size);
            });
        });
    }

    // }}}1

    // Vec.repr {{{1

    /**
     * Implementation of Vec.repr.
     */
    private HostResult reprMethod(HostContext c, VecVal vec, String desc) {
        return c.call("kink/_vec/REPR_VEC", reprVecHandle).args(vec);
    }

    // }}}1

    /**
     * Checks that the pair of fromVal and toVal is a valid range
     * in the sequence with {@code size},
     * and returns a node which yields an error message, or null.
     */
    @Nullable
    private GraphNode checkRange(String prefix, int size, Val fromVal, Val toVal) {
        if (! (fromVal instanceof NumVal)) {
            return vm.graph.raiseFormat("{}: From_pos must be a num, but got {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.repr(fromVal));
        }
        BigDecimal fromDecimal = ((NumVal) fromVal).bigDecimal();

        if (! (toVal instanceof NumVal)) {
            return vm.graph.raiseFormat("{}: To_pos must be a num, but got {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.repr(toVal));
        }
        BigDecimal toDecimal = ((NumVal) toVal).bigDecimal();

        if (! NumOperations.isRangePair(fromDecimal, toDecimal, size)) {
            return vm.graph.raiseFormat("{}: required range pair in [0, {}], but got {} and {}",
                    vm.graph.of(vm.str.of(prefix)),
                    vm.graph.of(vm.num.of(size)),
                    vm.graph.repr(fromVal),
                    vm.graph.repr(toVal));
        }

        return null;
    }

    /**
     * Adds a vec method to vars.
     */
    private void addMethod(
            Map<Integer, Val> vars,
            String recvDesc,
            String sym,
            String argsDesc,
            int arity,
            ThrowingFunction3<CallContext, VecVal, String, HostResult> action) {
        String desc = String.format(Locale.ROOT, "%s.%s%s", recvDesc, sym, argsDesc);
        int handle = vm.sym.handleFor(sym);
        var fun = vm.fun.make(desc).take(arity).action(c -> {
            if (! (c.recv() instanceof VecVal recvVec)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be vec, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(recvDesc)),
                            vm.graph.repr(c.recv())));
            }
            return action.apply(c, recvVec, desc);
        });
        vars.put(handle, fun);
    }

    /**
     * Adds a binary operator method to vars.
     */
    private void addBinOp(
            Map<Integer, Val> vars,
            String recvDesc,
            String sym,
            String argDesc,
            ThrowingFunction4<CallContext, VecVal, VecVal, String, HostResult> action) {
        String argsDesc = "(" + argDesc + ")";
        addMethod(vars, recvDesc, sym, argsDesc, 1, (c, recv, desc) -> {
            if (! (c.arg(0) instanceof VecVal argVec)) {
                return c.call(vm.graph.raiseFormat("{}: {} must be a vec, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.of(vm.str.of(argDesc)),
                            vm.graph.repr(c.arg(0))));
            }
            return action.apply(c, recv, argVec, desc);
        });
    }

    // factory methods {{{1

    /**
     * Returns a vec containing the elems.
     *
     * @param elems the elems of the vec.
     * @return a vec.
     */
    public VecVal of(List<? extends Val> elems) {
        return elems.isEmpty() ? of() : new VecVal(vm, VecInternal.of(vm, elems));
    }

    /**
     * Returns a new vec from the range of the array.
     *
     * @param vals the array of vals.
     * @param from the from index of the range (inclusive).
     * @param to the to index of the range (exclusive).
     * @return a new vec.
     */
    public VecVal of(Val[] vals, int from, int to) {
        Preconds.checkRange(from, to, vals.length);
        return from == to ? of() : new VecVal(vm, VecInternal.of(vm, vals, from, to));
    }

    /**
     * Returns a new empty vec.
     *
     * @return a new empty vec.
     */
    public VecVal of() {
        return new VecVal(vm, this.empty);
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0) {
        return new VecVal(vm, VecInternal.of(vm, e0));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1, Val e2) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1, e2));
    }

    /**
     * Returns a new vec.
     *
     * @param e0 the #0 elem.
     * @param e1 the #1 elem.
     * @param e2 the #2 elem.
     * @param e3 the #3 elem.
     * @return a new vec.
     */
    public VecVal of(Val e0, Val e1, Val e2, Val e3) {
        return new VecVal(vm, VecInternal.of(vm, e0, e1, e2, e3));
    }

    /**
     * Returns a new vec.
     *
     * @param elems the elems.
     * @return a new vec.
     */
    public VecVal of(Val... elems) {
        return new VecVal(vm, VecInternal.of(vm, elems, 0, elems.length));
    }

    // }}}1

}

// vim: et sw=4 sts=4 fdm=marker
