package org.kink_lang.kink.hostfun;

import org.kink_lang.kink.FunVal;
import org.kink_lang.kink.TraceVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.hostfun.graph.GraphNode;

import java.util.List;

/**
 * Context of an action of a host fun.
 *
 * <p>The context is valid during invocation of the lambda of
 * {@link HostFunBuilder#action(ThrowingFunction1)}
 * or {@link CallFlowToOn#on(HostFunReaction)}.</p>
 *
 * <p>See {@link org.kink_lang.kink.hostfun} for usage.</p>
 */
public interface HostContext {

    /**
     * Returns the traces of the current execution stack,
     * which is the result of
     * <a href="../../../../../../manual/language/eval.html#traces-instruction">traces</a>
     * instruction.
     *
     * <p>The traces are ordered from the bottom to the top,
     * unlike stack trace in Java.</p>
     *
     * @return the traces to the current execution stack.
     */
    List<TraceVal> traces();

    /**
     * Returns a HostResult to raise an exception.
     *
     * @param msg the message of the exception.
     * @return a HostResult to raise an exception.
     */
    HostResult raise(String msg);

    /**
     * Returns a HostResult to raise an exception made from the Java throwable.
     *
     * @param throwable the throwable of which a kink exception is made.
     * @return a HostResult to raise an exception.
     */
    HostResult raise(Throwable throwable);

    /**
     * Returns a call flow to call the fun, with nada as the recv and no args.
     *
     * <p>Example:</p>
     *
     * <pre>
     *  FunVal curry2 = vm.fun.make("curry2($fun)").take(1).action(c -&gt; {
     *      if (! (c.arg(0) instanceof FunVal fun)) {
     *          return c.call(vm.graph.raiseFormat("curry2($fun): required a fun, but got {}",
     *              vm.graph.repr(c.arg(0))));
     *      }
     *      FunVal curried = vm.fun.make().take(1).action(cc -&gt; {
     *          Val a0 = cc.arg(0);
     *          return vm.fun.make().take(1).action(c3 -&gt; {
     *              Val a1 = c3.arg(0);
     *              return c3.call(fun).args(a0, a1);
     *          });
     *      });
     *  });
     * </pre>
     *
     * @param fun the fun.
     * @return a call flow to call the fun.
     */
    CallFlowToRecv call(FunVal fun);

    /**
     * Returns a call flow to call a method of {@code ownerRecv} with no args.
     *
     * <p>The ownerRecv is also set as the recv of the call.</p>
     *
     * <p>Example:</p>
     *
     * <pre>
     *  int opAddHandle = vm.sym.handleFor("op_add");
     *  FunVal increment = vm.fun.make().take(1).action(c -&gt; {
     *      Val n = c.arg(0);
     *      // call “n + 1”
     *      return c.call(n, opAddHandle).args(vm.num.of(1));
     *  });
     * </pre>
     *
     * <p>If the specified var is empty,
     * or the target of the var is not a fun,
     * call() method returns a call flow which raises an exception.</p>
     *
     * @param ownerRecv the owner and the recv of the method.
     * @param symHandle the sym handle of the method.
     * @return a call flow with the fun.
     */
    CallFlowToRecv call(Val ownerRecv, int symHandle);

    /**
     * Returns a call flow to call a fun in the specified mod with no args.
     *
     * <p>nada is set as the recv of the call.</p>
     *
     * <p>Example:</p>
     *
     * <pre>
     *  int exit = vm.sym.handleFor("exit");
     *  FunVal abort = vm.fun.make().take(0).action(
     *      c -&gt; c.call("kink/PROCESS", exit).args(vm.num.of(1)));
     * </pre>
     *
     * <p>If the specified fun is empty,
     * or the target of the var is not a fun,
     * call() method returns a call flow which raises an exception.</p>
     *
     * @param modName the name of the mod containing the fun.
     * @param symHandle the sym handle of the fun.
     * @return a call flow.
     */
    CallFlowToArgs call(String modName, int symHandle);

    /**
     * Returns a call flow to evaluate the graph.
     *
     * <p>Example:</p>
     *
     * <pre>
     *  FunVal abs = vm.fun.make().take(0).action(c -&gt; {
     *      if (! (c.arg(0) instanceof NumVal num)) {
     *          return c.call(vm.graph.raiseFormat("unknown val {}", vm.graph.repr(c.arg(0))));
     *      }
     *      return vm.num.of(num.bigDecimal().abs());
     *  });
     * </pre>
     *
     * @param graph the graph to evaluate.
     * @return a call flow to evaluate the graph.
     */
    CallFlowToOn call(GraphNode graph);

}

// vim: et sw=4 sts=4 fdm=marker
