package org.kink_lang.kink.internal.callstack;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Support class for lnums in the call stack.
 */
public final class Lnums {

    /**
     * No instance.
     */
    private Lnums() {
    }

    /** Bits of each int num. */
    private static final int BITS = 20;

    /** Mask of each int num. */
    private static final long MASK = (1L << BITS) - 1;

    /**
     * Makes an lnum composing three int nums.
     * Each arg must be unsigned 20bit int.
     *
     * @param programCounter the program counter.
     * @param argCount the count of args of the frame.
     * @param dataStackUsage the used size of dataStack.
     *
     * @return the lnum.
     */
    public static long makeLnum(int programCounter, int argCount, int dataStackUsage) {
        // TODO add Preconds.checkIntArgMinMax
        Preconds.checkPosIndex(programCounter, (int) MASK);
        Preconds.checkPosIndex(argCount, (int) MASK);
        Preconds.checkPosIndex(dataStackUsage, (int) MASK);
        return ((long) programCounter) << BITS * 2
            | ((long) argCount) << BITS
            | dataStackUsage;
    }

    /**
     * Extracts programCounter from the lnum.
     *
     * @param lnum made by {@link makeLnum(int, int, int)}.
     * @return the programCounter.
     */
    public static int getProgramCounter(long lnum) {
        return (int) (lnum >>> BITS * 2 & MASK);
    }

    /**
     * Extracts argCount from the lnum.
     *
     * @param lnum made by {@link makeLnum(int, int, int)}.
     * @return the argCount.
     */
    public static int getArgCount(long lnum) {
        return (int) (lnum >>> BITS & MASK);
    }

    /**
     * Extracts dataStackUsage from the lnum.
     *
     * @param lnum made by {@link makeLnum(int, int, int)}.
     * @return the dataStackUsage.
     */
    public static int getDataStackUsage(long lnum) {
        return (int) (lnum & MASK);
    }

}

// vim: et sw=4 sts=4 fdm=marker
