package org.kink_lang.kink.internal.callstack;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;

import org.kink_lang.kink.internal.contract.Preconds;

/**
 * Ring buffer of traces of tail-calls.
 */
public class TailTraceRingBuffer {

    /** Index of the next element. */
    private int index;

    /** Whether all the elements of this.ring are filled. */
    private boolean isFull;

    /** Array of the ring of traces. */
    private final Trace[] ring;

    /**
     * Constructs a ring buffer with fields.
     */
    private TailTraceRingBuffer(int index, boolean isFull, Trace[] ring) {
        this.index = index;
        this.isFull = isFull;
        this.ring = ring;
    }

    /**
     * Makes a ring buffer.
     *
     * @param size the size of the ring buffer.
     * @return a new ring buffer with the specified size.
     */
    public static TailTraceRingBuffer withSize(int size) {
        return new TailTraceRingBuffer(0, false, new Trace[size]);
    }

    /**
     * Size of the ring buffer.
     */
    private int size() {
        return this.ring.length;
    }

    /**
     * Returns the traces.
     *
     * @return the traces.
     */
    public List<Trace> traces() {
        List<Trace> ringList = Arrays.asList(this.ring);

        if (this.isFull) {
            List<Trace> result = new ArrayList<>(size() + 1);
            result.add(Trace.SNIP);
            result.addAll(ringList.subList(this.index, size()));
            result.addAll(ringList.subList(0, this.index));
            return Collections.unmodifiableList(result);
        } else {
            return List.copyOf(ringList.subList(0, this.index));
        }
    }

    /**
     * Pushes the trace to the ring buffer.
     *
     * @param trace the trace.
     */
    public void push(Trace trace) {
        this.ring[this.index] = trace;
        ++ this.index;
        if (this.index == size()) {
            this.index = 0;
            this.isFull = true;
        }
    }

    /**
     * Make the ring buffer empty.
     */
    public void reset() {
        this.index = 0;
        this.isFull = false;
    }

    /**
     * Whether the ring buffer does not have any element.
     */
    private boolean isEmpty() {
        return this.index == 0 && ! this.isFull;
    }

    /**
     * New ring buffer equivalent to this buffer.
     *
     * @return a new ring buffer equivalent to this buffer.
     */
    public TailTraceRingBuffer makeCopy() {
        return isEmpty()
            ? new TailTraceRingBuffer(0, false, new Trace[size()])
            : new TailTraceRingBuffer(this.index, this.isFull, this.ring.clone());
    }

    /**
     * Copy traces from src.
     *
     * @param src buffer from which traces are copied.
     */
    public void copyFrom(TailTraceRingBuffer src) {
        Preconds.checkArg(size() == src.size(), "size of src must be equal to the size of this");
        if (src.isEmpty()) {
            reset();
        } else {
            this.index = src.index;
            this.isFull = src.isFull;
            System.arraycopy(src.ring, 0, this.ring, 0, size());
        }
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "TraceRingBuffer(%s %s)",
                size(), traces());
    }

    /**
     * Returns the equality properties.
     */
    private List<Object> getProperties() {
        return List.of(size(), traces());
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof TailTraceRingBuffer
            && getProperties().equals(((TailTraceRingBuffer) arg).getProperties());
    }

}

// vim: et sw=4 sts=4 fdm=marker
