package org.kink_lang.kink.internal.compile.javaclassir;

import java.lang.invoke.MethodType;
import java.util.ArrayList;
import java.util.List;

import org.objectweb.asm.Type;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.internal.callstack.Location;
import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.program.itree.LocalVar;

/**
 * Generates insns of loval vars in a ssa fun.
 */
public class FastLvarAccessGenerator implements LvarAccessGenerator {

    /** The allocation analysis of local vars. */
    private final AllocationSet allocationSet;

    /** The supplier of key strs. */
    private final KeyStrSupplier keySup;

    /** Trace accumulator. */
    private final TraceAccumulator traceAccum;

    /**
     * Constructs a generator.
     *
     * @param allocationSet the allocation analysis of local vars.
     * @param keySup the supplier of key strs.
     * @param traceAccum trace accumulator.
     */
    public FastLvarAccessGenerator(
            AllocationSet allocationSet,
            KeyStrSupplier keySup,
            TraceAccumulator traceAccum) {
        this.allocationSet = allocationSet;
        this.keySup = keySup;
        this.traceAccum = traceAccum;
    }

    /** Insns to produce the recv. */
    private static final List<Insn> RECV_INSNS = List.of(
            // => (dataStack)
            InsnsGenerator.LOAD_DATASTACK,
            // => (recv)
            InsnsGenerator.INVOKE_RECV,
            // contParam <-
            InsnsGenerator.STORE_CONTPARAM);

    /**
     * Insns to produce the arg.
     */
    private List<Insn> argInsns(int argIndex) {
        return List.of(
                InsnsGenerator.LOAD_DATASTACK,
                // => (dataStack argIndex)
                new Insn.PushInt(argIndex),
                // => (arg)
                InsnsGenerator.INVOKE_ARG,
                // contParam <-
                InsnsGenerator.STORE_CONTPARAM);
    }

    /**
     * Insns to produce the val of the local var on the datastack,
     * without checking null.
     */
    private List<Insn> stackInsnsNoCheck(int stackIndex) {
        // contParam = dataStack.getAtOffset(stackIndex + 1 + argCount);
        return List.of(
                InsnsGenerator.LOAD_DATASTACK,
                new Insn.PushInt(stackIndex + 1),
                InsnsGenerator.LOAD_ARGCOUNT,
                new Insn.AddInt(),
                InsnsGenerator.INVOKE_AT_OFFSET,
                InsnsGenerator.STORE_CONTPARAM);
    }

    /**
     * Insns to load stacked lvar to contParam,
     * checking the value is null.
     */
    private List<Insn> stackInsnsCheckNull(int stackIndex, String lvarName, Location loc) {
        List<Insn> insns = new ArrayList<>();

        // contParam = dataStack.getAtOffset(stackIndex + 1 + argCount)
        insns.addAll(stackInsnsNoCheck(stackIndex));

        // if contParam is null, transition to raise and return
        insns.addAll(LvarAccessGenerator.checkNull(lvarName, loc, keySup, traceAccum));

        return insns;
    }

    /**
     * Insns to produce the val of the free local var on the datastack,
     * without checking null.
     */
    private List<Insn> fieldInsnsNoCheck(int fieldIndex) {
        // contParam = this.valFieldX
        return List.of(
                new Insn.LoadThis(),
                new Insn.GetField(
                    JavaClassIr.TYPE_BASE,
                    "valField" + fieldIndex,
                    Type.getType(Val.class)),
                InsnsGenerator.STORE_CONTPARAM);
    }

    /**
     * Insns to produce the val of the free local var on the datastack,
     * checking the value is null.
     */
    private List<Insn> fieldInsnsCheckNull(int fieldIndex, String lvarName, Location loc) {
        List<Insn> insns = new ArrayList<>();

        // contParam = this.valFieldX
        insns.addAll(fieldInsnsNoCheck(fieldIndex));

        // if contParam is null, transition to raise and return
        insns.addAll(LvarAccessGenerator.checkNull(lvarName, loc, keySup, traceAccum));

        return insns;
    }

    /**
     * Insns to produce the val of a preloaded var.
     */
    private List<Insn> prealodedVarInsns(String lvarName) {
        return List.of(
                new Insn.InvokeDynamic(
                    MethodType.methodType(Val.class),
                    NewVal.BOOTSTRAP_PRELOADED_HANDLE,
                    List.of(lvarName)),
                InsnsGenerator.STORE_CONTPARAM);
    }

    @Override
    public List<Insn> loadLvar(LocalVar lvar, Location loc) {
        Allocation allocation = allocationSet.get(lvar);
        if (allocation instanceof Allocation.Recv) {
            return RECV_INSNS;
        } else if (allocation instanceof Allocation.Arg arg) {
            return argInsns(arg.index());
        } else if (allocation instanceof Allocation.Stack stack) {
            return stack.nonnull()
                ? stackInsnsNoCheck(stack.index())
                : stackInsnsCheckNull(stack.index(), lvar.name(), loc);
        } else if (allocation instanceof Allocation.Field field) {
            return field.nonnull()
                ? fieldInsnsNoCheck(field.index())
                : fieldInsnsCheckNull(field.index(), lvar.name(), loc);
        } else if (allocation instanceof Allocation.Preloaded) {
            return prealodedVarInsns(lvar.name());
        } else {
            throw new IllegalArgumentException(
                    "unsupported storage type: " + allocation
                    + " for " + lvar);
        }
    }

    @Override
    public List<Insn> loadLvarAllowNull(LocalVar lvar) {
        Allocation allocation = allocationSet.get(lvar);
        if (allocation instanceof Allocation.Recv) {
            return RECV_INSNS;
        } else if (allocation instanceof Allocation.Arg arg) {
            return argInsns(arg.index());
        } else if (allocation instanceof Allocation.Stack stack) {
            return stackInsnsNoCheck(stack.index());
        } else if (allocation instanceof Allocation.Field field) {
            return fieldInsnsNoCheck(field.index());
        } else if (allocation instanceof Allocation.Preloaded) {
            return prealodedVarInsns(lvar.name());
        } else {
            throw new IllegalArgumentException(
                    "unsupported storage type: " + allocation
                    + " for " + lvar);
        }
    }

    @Override
    public List<Insn> storeLvar(LocalVar lvar) {
        Allocation allocation = allocationSet.get(lvar);
        Preconds.checkArg(allocation instanceof Allocation.Stack,
                "lvar must be allocated on stack");

        Allocation.Stack stack = (Allocation.Stack) allocation;
        return List.of(
                // => (dataStack)
                InsnsGenerator.LOAD_DATASTACK,

                // => (dataStack #lvar+1)
                new Insn.PushInt(stack.index() + 1),

                // => (dataStack #lvar+1 argCount)
                InsnsGenerator.LOAD_ARGCOUNT,

                // => (dataStack #lvar+1+argCount)
                new Insn.AddInt(),

                // => (dataStack #lvar+1+argCount contParam)
                new Insn.LoadArg(1),

                // => ()
                InsnsGenerator.INVOKE_SET_AT_OFFSET);
    }

    @Override
    public List<Insn> passRecv(LocalVar lvar) {
        Allocation actualAllocation = allocationSet.get(lvar);
        Preconds.checkArg(actualAllocation.equals(new Allocation.Recv()),
                "actualStorage must be recv");
        return List.of();
    }

    @Override
    public List<Insn> passArg(LocalVar lvar, int argIndex) {
        Allocation actualAllocation = allocationSet.get(lvar);
        Allocation expected = new Allocation.Arg(argIndex);
        Preconds.checkArg(actualAllocation.equals(expected), "actualStorage must be arg");
        return List.of();
    }

    @Override
    public boolean isUnused(LocalVar lvar) {
        return allocationSet.get(lvar) instanceof Allocation.Unused;
    }

}

// vim: et sw=4 sts=4 fdm=marker
