package org.kink_lang.kink.internal.mod.java;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.kink_lang.kink.Val;
import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * java_throw val, which is used to throw an exception from the proxy method.
 */
class JavaThrowVal extends Val {

    /** Wrapped throwable. */
    private final Throwable thrown;

    /**
     * Constructs a java_throw val.
     */
    private JavaThrowVal(Vm vm, SharedVars sharedVars, Throwable thrown) {
        super(vm, sharedVars);
        this.thrown = thrown;
    }

    /**
     * Makes a java_throw val.
     */
    static JavaThrowVal of(Vm vm, Throwable thrown) {
        ThrowerHelper helper = ThrowerHelper.of(vm);
        return new JavaThrowVal(vm, helper.sharedVars, thrown);
    }

    /**
     * Returns the wrapped throwable.
     */
    Throwable thrown() {
        return this.thrown;
    }

    @Override
    public String toString() {
        return String.format(Locale.ROOT, "JavaThrowVal(%s)", this.thrown);
    }

    /**
     * Returns the properties of the thrower val.
     */
    private List<Object> getProperties() {
        return List.of(this.vm, this.thrown);
    }

    @Override
    public int hashCode() {
        return getProperties().hashCode();
    }

    @Override
    public boolean equals(Object arg) {
        return arg == this
            || arg instanceof JavaThrowVal
            && getProperties().equals(((JavaThrowVal) arg).getProperties());
    }

    /**
     * Helper of java_throw vals.
     */
    static class ThrowerHelper {

        /** Vm. */
        private final Vm vm;

        /** Shared vars of java_throw vals. */
        final SharedVars sharedVars;

        /**
         * Constructs a helper for the vm.
         */
        private ThrowerHelper(Vm vm) {
            this.vm = vm;
            this.sharedVars = makeSharedVars();
        }

        /**
         * Returns the helper instance of the vm.
         */
        static ThrowerHelper of(Vm vm) {
            return vm.component.getOrRegister(ThrowerHelper.class, ThrowerHelper::new);
        }

        /**
         * Makes the set of shared vars for the helper.
         */
        private SharedVars makeSharedVars() {
            Map<Integer, Val> map = new HashMap<>();
            map.put(vm.sym.handleFor("thrown"),
                    vm.fun.make("Java_throw.thrown").take(0).action(this::thrownMethod));
            map.put(vm.sym.handleFor("repr"),
                    vm.fun.make("Java_throw.repr").take(0).action(this::reprMethod));
            return vm.sharedVars.of(map);
        }

        /**
         * Implementation of Java_throw.thrown.
         */
        private HostResult thrownMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof JavaThrowVal ptr)) {
                return c.call(vm.graph.raiseFormat(
                            "Java_throw.thrown: "
                            + "Java_throw must be a java_throw, but was {}",
                            vm.graph.repr(recv)));
            }
            return vm.java.of(ptr.thrown(), Throwable.class);
        }

        /**
         * Implementation of Java_throw.repr.
         */
        private HostResult reprMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof JavaThrowVal ptr)) {
                return c.call(vm.graph.raiseFormat(
                            "Java_throw.repr: "
                            + "Java_throw must be a java_throw, but was {}",
                            vm.graph.repr(recv)));
            }
            return vm.str.of(String.format(Locale.ROOT, "(java_throw %s)", ptr.thrown()));
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
