package org.kink_lang.kink.internal.mod.random;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Random;

import javax.annotation.Nullable;

import org.kink_lang.kink.BinVal;
import org.kink_lang.kink.NumVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.Vm;

/**
 * Operations to produce random numbers.
 */
final class RandomNumbers {

    /**
     * Should not be instantiated.
     */
    RandomNumbers() {
        throw new UnsupportedOperationException("should not be instantiated");
    }

    // produceIntNum {{{

    /**
     * Produces random num val, or null if upperBound is not valid.
     */
    @Nullable
    static NumVal produceIntNum(Vm vm, Random rng, Val upperBound) {
        if (! (upperBound instanceof NumVal)) {
            return null;
        }

        BigDecimal upperBoundDec = ((NumVal) upperBound).bigDecimal();
        if (upperBoundDec.scale() != 0 || upperBoundDec.signum() <= 0) {
            return null;
        }

        long upperBoundLong = extractLong(upperBoundDec);
        return upperBoundLong >= 2
            ? genLongRandom(vm, rng, upperBoundLong)
            : genBigRandom(vm, rng, upperBoundDec.toBigInteger());
    }

    /**
     * Extracts the non negative long value from dec, or -1L.
     */
    private static long extractLong(BigDecimal dec) {
        try {
            return dec.longValueExact();
        } catch (ArithmeticException ex) {
            return -1L;
        }
    }

    /**
     * Generates long random num val.
     *
     * upperBound : [2, Long.MAX_VALUE].
     */
    private static NumVal genLongRandom(Vm vm, Random rng, long upperBound) {
        long max = upperBound - 1;
        long mask = (Long.highestOneBit(max) << 1) - 1;
        int leadingZeroBits = Long.numberOfLeadingZeros(max);
        int rightShift = (leadingZeroBits / Byte.SIZE) * Byte.SIZE;
        while (true) {
            long candidate = (rng.nextLong() >>> rightShift) & mask;
            if (candidate <= max) {
                return vm.num.of(candidate);
            }
        }
    }

    /**
     * Generates BigInteger randon num val.
     */
    private static NumVal genBigRandom(Vm vm, Random rng, BigInteger upperBound) {
        BigInteger max = upperBound.subtract(BigInteger.ONE);
        int numBits = max.bitLength();
        while (true) {
            BigInteger candidate = new BigInteger(numBits, rng);
            if (candidate.compareTo(max) <= 0) {
                return vm.num.of(candidate);
            }
        }
    }

    // }}}

    // produceBin {{{

    /**
     * Produces bin val.
     */
    static BinVal produceBin(Vm vm, Random rng, Val size) {
        if (! (size instanceof NumVal)) {
            return null;
        }

        BigDecimal sizeDec = ((NumVal) size).bigDecimal();
        if (sizeDec.scale() != 0 || sizeDec.signum() < 0) {
            return null;
        }

        int sizeInt = extractInt(sizeDec);
        if (sizeInt < 0) {
            return null;
        }

        byte[] bytes = new byte[sizeInt];
        rng.nextBytes(bytes);
        return vm.bin.of(bytes);
    }

    /**
     * Extracts the non negative int value from dec, or -1.
     */
    private static int extractInt(BigDecimal dec) {
        try {
            return dec.intValueExact();
        } catch (ArithmeticException ex) {
            return -1;
        }
    }

    // }}}

}

// vim: et sw=4 sts=4 fdm=marker
