package org.kink_lang.kink.internal.mod.random;

import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Random;

import org.kink_lang.kink.BinVal;
import org.kink_lang.kink.NumVal;
import org.kink_lang.kink.Val;
import org.kink_lang.kink.SharedVars;
import org.kink_lang.kink.Vm;
import org.kink_lang.kink.hostfun.CallContext;
import org.kink_lang.kink.hostfun.HostResult;

/**
 * Random number generator val.
 *
 * @param <T> the type of RNG implementation.
 */
public class RngVal<T extends Random> extends Val {

    /** The rng. */
    private final T random;

    /** The type tag such as "csrng". */
    private final String typeTag;

    /**
     * Constructs an rng val.
     */
    RngVal(Vm vm, SharedVars sharedVars, T random, String typeTag) {
        super(vm, sharedVars);
        this.random = random;
        this.typeTag = typeTag;
    }

    /**
     * Makes a random number generator val.
     *
     * @param vm the vm.
     * @param random the RNG.
     * @param typeTag the type tag such as "csrng".
     * @return a random number generator val.
     */
    public static RngVal<Random> of(Vm vm, Random random, String typeTag) {
        RngHelper helper = RngHelper.of(vm);
        return new RngVal<Random>(vm, helper.sharedVars, random, typeTag);
    }

    /**
     * Returns the RNG.
     *
     * @return the RNG.
     */
    public T getRandom() {
        return this.random;
    }

    /**
     * Helper of thrower vals.
     */
    static class RngHelper {

        /** Vm. */
        private final Vm vm;

        /** Map from syms to vars. */
        final Map<Integer, Val> varMap;

        /** Shared vars of thrower vals. */
        final SharedVars sharedVars;

        /**
         * Constructs a helper for the vm.
         */
        private RngHelper(Vm vm) {
            this.vm = vm;
            this.varMap = makeVarMap();
            this.sharedVars = vm.sharedVars.of(this.varMap);
        }

        /**
         * Returns the helper instance of the vm.
         */
        static RngHelper of(Vm vm) {
            return vm.component.getOrRegister(RngHelper.class, RngHelper::new);
        }

        /**
         * Makes the set of shared vars for the helper.
         */
        private Map<Integer, Val> makeVarMap() {
            Map<Integer, Val> map = new HashMap<>();
            map.put(vm.sym.handleFor("gen_int"),
                    vm.fun.make("Rng.gen_int(Upper_bound)").take(1).action(this::genIntMethod));
            map.put(vm.sym.handleFor("gen_bin"),
                    vm.fun.make("Rng.gen_bin(Size)").take(1).action(this::genBinMethod));
            map.put(vm.sym.handleFor("gen_bool"),
                    vm.fun.make("Rng.gen_bool").take(0).action(this::genBoolMethod));
            map.put(vm.sym.handleFor("repr"),
                    vm.fun.make("Rng.repr").take(0).action(this::reprMethod));
            return Collections.unmodifiableMap(map);
        }

        /**
         * Implementation of Rng.gen_int.
         */
        private HostResult genIntMethod(CallContext c) {
            String desc = "Rng.gen_int(Upper_bound)";
            Val recv = c.recv();
            if (! (recv instanceof RngVal)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: Rng must be an rng val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(recv)));
            }
            @SuppressWarnings("unchecked")
            RngVal<Random> rngVal = (RngVal) recv;
            Random random = rngVal.getRandom();
            Val upperBound = c.arg(0);
            NumVal num = RandomNumbers.produceIntNum(vm, random, upperBound);
            return num != null
                ? num
                : c.call(vm.graph.raiseFormat(
                            "{}: Upper_bound must be a positive int num, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(upperBound)));
        }

        /**
         * Implementation of Rng.gen_bin.
         */
        private HostResult genBinMethod(CallContext c) {
            String desc = "Rng.gen_bin(Size)";
            Val recv = c.recv();
            if (! (recv instanceof RngVal)) {
                return c.call(vm.graph.raiseFormat(
                            "{}: Rng must be an rng val, but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(recv)));
            }
            @SuppressWarnings("unchecked")
            RngVal<Random> rngVal = (RngVal) recv;
            Random random = rngVal.getRandom();
            Val size = c.arg(0);
            BinVal bin = RandomNumbers.produceBin(vm, random, size);
            return bin != null
                ? bin
                : c.call(vm.graph.raiseFormat(
                            "{}: Size must be an int num in [0, 0x7fff_ffff], but got {}",
                            vm.graph.of(vm.str.of(desc)),
                            vm.graph.repr(size)));
        }

        /**
         * Implementation of Rng.gen_bool.
         */
        private HostResult genBoolMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof RngVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Rng.gen_bool: Rng must be an rng val, but got {}",
                            vm.graph.repr(recv)));
            }
            @SuppressWarnings("unchecked")
            RngVal<Random> rngVal = (RngVal) recv;
            Random random = rngVal.getRandom();
            return vm.bool.of(random.nextBoolean());
        }

        /**
         * Implementation of Rng.repr.
         */
        private HostResult reprMethod(CallContext c) {
            Val recv = c.recv();
            if (! (recv instanceof RngVal)) {
                return c.call(vm.graph.raiseFormat(
                            "Rng.repr: Rng must be an rng val, but got {}",
                            vm.graph.repr(recv)));
            }
            @SuppressWarnings("unchecked")
            RngVal<Random> rngVal = (RngVal) recv;
            return vm.str.of(String.format(Locale.ROOT, "(%s val_id=%d)",
                        rngVal.typeTag, rngVal.identity()));
        }
    }

}

// vim: et sw=4 sts=4 fdm=marker
