package org.kink_lang.kink.internal.program.itree;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Assignment in the form {@code [:A :B :C.opt :D.opt :R.rest] <- xxx}.
 *
 * @param mandatory the lvars of mandatory params.
 * @param opt       the lvars of optional params.
 * @param rest      local var of the rest param.
 * @param rhs       right-hand-side.
 * @param pos       pos index of the assignment.
 */
public record OptRestVecAssignmentItree(List<LocalVar> mandatory,
                                        List<LocalVar> opt,
                                        LocalVar rest,
                                        Itree rhs,
                                        int pos) implements Itree {

    /**
     * Constructs an assignment itree.
     *
     * @param mandatory the local vars of mandatory params.
     * @param opt       the local vars of optional params.
     * @param rest      the local var of the rest param.
     * @param rhs       the right-hand-side.
     * @param pos       the pos index of the assignment.
     */
    public OptRestVecAssignmentItree {
        mandatory = List.copyOf(mandatory);
        opt = List.copyOf(opt);
    }

    /**
     * Returns the string representation of the left hand side.
     *
     * @return the string representation of the left hand side.
     */
    public String lhsRepr() {
        Stream<String> mandatoryRepr = mandatory().stream().map(LocalVar::lhsRepr);
        Stream<String> optRepr = opt().stream().map(LocalVar::lhsRepr).map(s -> s + ".opt");
        Stream<String> restRepr = Stream.of(rest().lhsRepr() + ".rest");
        return Stream.concat(Stream.concat(mandatoryRepr, optRepr), restRepr)
            .collect(Collectors.joining(" ", "[", "]"));
    }

    /**
     * Returns the minimum arity of the left hand side.
     *
     * @return the minimum arity of the left hand side.
     */
    public int minArity() {
        return mandatory().size();
    }

    /**
     * Returns the count of the optional parameters.
     *
     * @return the count of the optional parameters.
     */
    public int optCount() {
        return opt().size();
    }

    @Override
    public <T> T accept(ItreeVisitor<T> visitor) {
        return visitor.visit(this);
    }

}

// vim: et sw=4 sts=4 fdm=marker
