package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.HashMap;
import java.util.Map;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Propagates the content of local var in its dereference.
 */
public class ContentPropagator extends BaseOptimizer {

    /** Mapping from the local var to the content. */
    private final Map<LocalVar, LocalVarContent> lvarContentMapping;

    /**
     * Constructs an optimizer.
     *
     * @param lvarContentMapping mapping from the local var to the content.
     */
    public ContentPropagator(Map<LocalVar, LocalVarContent> lvarContentMapping) {
        this.lvarContentMapping = lvarContentMapping;
    }

    @Override
    public Itree visit(LderefItree lderef) {
        LocalVarContent content = lvarContentMapping.get(lderef.lvar());
        if (content instanceof LocalVarContent.Alias alias) {
            return new LderefItree(alias.lvar(), lderef.pos());
        } else {
            return lderef;
        }
    }

    /**
     * Returns the factory.
     *
     * @return the factory.
     */
    static Factory factory() {
        return new Factory(Map.of());
    }

    /**
     * Factory of ContentPropagator.
     */
    static class Factory implements OptimizerFactory {

        /** Content mapping. */
        final Map<LocalVar, LocalVarContent> lvarContentMapping;

        /**
         * Constructs a factory.
         */
        Factory(Map<LocalVar, LocalVarContent> lvarContentMapping) {
            this.lvarContentMapping = lvarContentMapping;
        }

        @Override
        public Factory makeFactory(FastFunItree enclosing) {
            Map<LocalVar, LocalVarContent> newMapping = new HashMap<>(this.lvarContentMapping);
            newMapping.putAll(enclosing.lvarContentMapping());
            return new Factory(newMapping);
        }

        @Override
        public ContentPropagator makeOptimizer() {
            return new ContentPropagator(this.lvarContentMapping);
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
