package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

import org.kink_lang.kink.internal.program.itree.Itree;

/**
 * A class to provide the optimizer of itrees.
 */
public final class ItreeOptimizers {

    /** Logger for the class. */
    private static final System.Logger LOGGER = System.getLogger(ItreeOptimizers.class.getName());

    /**
     * This class should not be instantiated.
     */
    ItreeOptimizers() {
        throw new UnsupportedOperationException("should not be instantiated");
    }

    /** Factories of optimizers for every phase. */
    private static final List<OptimizerFactory> MAIN_FACTORIES = List.of(
                new ConstantOptimizerFactory(new ParenContOptimizer()),
                new ConstantOptimizerFactory(new FlattenSeqOptimizer()),

                new ConstantOptimizerFactory(new LderefOptimizer()),
                new ConstantOptimizerFactory(new AssignmentOptimizer()),
                new ConstantOptimizerFactory(new MinusConstantFolder()),
                new ConstantOptimizerFactory(new StoreOptimizer()),
                new ConstantOptimizerFactory(new LstoreOptimizer()),
                new ConstantOptimizerFactory(new NestedVecAssignmentInliner()),
                new ConstantOptimizerFactory(new OptVecAssignmentOptimizer()),
                new ConstantOptimizerFactory(new OptRestVecAssignmentOptimizer()),
                new ConstantOptimizerFactory(new RestVecAssignmentOptimizer()),
                new ConstantOptimizerFactory(new VarrefVecAssignmentInliner()),
                new ConstantOptimizerFactory(new VarrefVecAssignmentToLocalOptimizer()),
                new ConstantOptimizerFactory(new BiArithmeticInliner()),
                new ConstantOptimizerFactory(new ArgsPassingOptimizer()),
                new ConstantOptimizerFactory(new NestedArgsPassingOptimizer()),
                new ConstantOptimizerFactory(new UnnestArgsPassingOptimizer()),
                new ConstantOptimizerFactory(new FunMcallToSymcallOptimizer()),
                new ConstantOptimizerFactory(new LetSymcallInliner()),
                new SlowFunBodyOptimizerFactory(new LetRecProducer()),
                new ConstantOptimizerFactory(new LetRecChainer()),
                new FastFunBodyOptimizerFactory(LetRecReducer::new),

                ContentPropagator.factory(),
                new FastFunBodyOptimizerFactory(DeadLstoreEliminator::new),
                new ConstantOptimizerFactory(new EliminateUnusedOptimizer()),
                new ConstantOptimizerFactory(new DeparenOptimizer()));

    /** Phase after SSA conversion. */
    private static final List<OptimizerFactory> AFTER_SSA_PHASE = List.of(
                // apply them only for lvars which come from top level
                new FastFunBodyOptimizerFactory(fun -> new IfInliner()),
                new FastFunBodyOptimizerFactory(fun -> new BranchInliner()),
                new FastFunBodyOptimizerFactory(fun -> new BranchWithElseInliner()),
                new ConstantOptimizerFactory(new TraitNewValOptimizer()),
                new ConstantOptimizerFactory(new SimpleNewValInliner()));

    /** Make composite optimizer of the phase. */
    private static UnaryOperator<Itree> makePhase(List<OptimizerFactory> phaseFactories) {
        var factories = new ArrayList<>(phaseFactories);
        factories.addAll(MAIN_FACTORIES);
        return new RepetitiveOptimizer(List.of(new RecursiveOptimizer(factories)));
    }

    /** The counter for unique str generation. */
    private static final AtomicLong COUNTER = new AtomicLong(1);

    /** Unique str supplier. */
    private static final Supplier<String> UNIQ_SUPPLIER
        = () -> Long.toString(COUNTER.getAndIncrement());

    /**
     * Makes an optimizer factory of ssafying operation.
     */
    private static OptimizerFactory ssafyFactory(Supplier<String> uniqSupplier) {
        return new ConstantOptimizerFactory(new SsafyOptimizer(uniqSupplier));
    }

    /**
     * Returns the optimizer function of itrees.
     *
     * @return the optimizer function of itrees.
     */
    public static Function<Itree, Itree> getOptimizer() {
        return makeOptimizer(UNIQ_SUPPLIER);
    }

    /**
     * Makes a composite optimizer.
     */
    static Function<Itree, Itree> makeOptimizer(Supplier<String> uniqSupplier) {
        return makePhase(List.of(ssafyFactory(uniqSupplier)))
            .andThen(makePhase(AFTER_SSA_PHASE))
            .andThen(itree -> {
                LOGGER.log(System.Logger.Level.TRACE, "optimized to {0}", itree);
                return itree;
            });
    }

}

// vim: et sw=4 sts=4 fdm=marker
