package org.kink_lang.kink.internal.program.itreeoptimize;

import java.util.ArrayList;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Inlining optimizer of a let clause on the tail context.
 *
 * <p>Example program before inlining:</p>
 *
 * <pre>
 * { do_foo{ ,,, }
 *   :X = produce_arg
 *   do_bar{ ,,, }
 * }
 * </pre>
 *
 * <p>The program above is converted to:</p>
 *
 * <pre>
 * { do_foo{ ,,, }
 *   {(:X)
 *     do_bar{ ,,, }
 *   }!call![()](produce_arg)
 * }
 * </pre>
 *
 * <p>This optimizer converts the program roughly to the following.</p>
 *
 * <pre>
 * { do_foo{ ,,, }
 *   :X &lt;- (* change to the new local binding here *) produce_arg
 *   do_bar{ ,,, }
 * }
 * </pre>
 */
public class LetSymcallInliner extends BaseOptimizer {

    @Override
    public Itree visit(FastFunItree fun) {
        int pos = fun.pos();
        var steps = Operations.toSteps(fun.body());
        var beforeTail = steps.subList(0, steps.size() - 1);
        var tail = steps.get(steps.size() - 1);

        if (! (tail instanceof SymcallItree tailCall)
                || ! (tailCall.recv() instanceof NadaItree)) {
            return fun;
        }

        var args = tailCall.args();
        if (args.size() != 1 || ! (args.get(0) instanceof Itree arg)) {
            return fun;
        }

        if (! (tailCall.fun() instanceof FastFunItree tailFun)) {
            return fun;
        }

        var tailSteps = Operations.toSteps(tailFun.body());
        if (! (tailSteps.get(0) instanceof ArgsPassingItree argsPassing)
                || argsPassing.lvars().size() != 1) {
            return fun;
        }
        LocalVar lvar = argsPassing.lvars().get(0);

        var afterStore = tailSteps.subList(1, tailSteps.size());
        if (afterStore.stream().anyMatch(RecvArgsSearcher::containsRecvOrArgs)) {
            return fun;
        }

        var generatedSteps = new ArrayList<>(beforeTail);
        generatedSteps.add(new LstoreItree(lvar, arg, pos));
        generatedSteps.addAll(afterStore);
        return new FastFunItree(new SeqItree(generatedSteps, pos), pos);
    }

}

// vim: et sw=4 sts=4 fdm=marker
