package org.kink_lang.kink.internal.program.itreeoptimize;

import org.kink_lang.kink.internal.contract.Preconds;
import org.kink_lang.kink.internal.program.itree.BindingItree;
import org.kink_lang.kink.internal.program.itree.Itree;
import org.kink_lang.kink.internal.program.itree.ItreeElem;
import org.kink_lang.kink.internal.program.itree.McallItree;
import org.kink_lang.kink.internal.program.itree.VarrefItree;

/**
 * Utility class for params.
 */
final class Params {

    /**
     * Should not be instantiated.
     */
    Params() {
        throw new UnsupportedOperationException("should not be instantiated");
    }

    /**
     * Whether the itree is a mandatory param.
     */
    static boolean isMandatoryParam(ItreeElem elem) {
        return elem.isSingle()
            && isLocalVarref(elem.expr());
    }

    /**
     * Gets the sym of the mandatory param.
     */
    static String getMandatoryParamSym(ItreeElem elem) {
        Preconds.checkArg(isMandatoryParam(elem), "elem must be a mandatory param");
        VarrefItree varref = (VarrefItree) elem.expr();
        return varref.sym();
    }

    /**
     * Whether the itree is an opt param.
     */
    static boolean isOptParam(ItreeElem elem) {
        if (! elem.isSingle()) {
            return false;
        }

        Itree itree = elem.expr();
        if (! (itree instanceof McallItree)) {
            return false;
        }

        McallItree mcall = (McallItree) itree;
        return isLocalVarref(mcall.ownerRecv())
            && mcall.sym().equals("opt")
            && mcall.args().isEmpty();
    }

    /**
     * Gets the sym of the opt param.
     */
    static String getOptParamSym(ItreeElem elem) {
        Preconds.checkArg(isOptParam(elem), "elem must be an opt param");
        McallItree mcall = (McallItree) elem.expr();
        VarrefItree varref = (VarrefItree) mcall.ownerRecv();
        return varref.sym();
    }

    /**
     * Whether the itree is a rest param.
     */
    static boolean isRestParam(ItreeElem elem) {
        if (! elem.isSingle()) {
            return false;
        }

        Itree itree = elem.expr();
        if (! (itree instanceof McallItree)) {
            return false;
        }

        McallItree mcall = (McallItree) itree;
        return isLocalVarref(mcall.ownerRecv())
            && mcall.sym().equals("rest")
            && mcall.args().isEmpty();
    }

    /**
     * Gets the sym of the rest param.
     */
    static String getRestParamSym(ItreeElem elem) {
        Preconds.checkArg(isRestParam(elem), "elem must be a rest param");
        McallItree mcall = (McallItree) elem.expr();
        VarrefItree varref = (VarrefItree) mcall.ownerRecv();
        return varref.sym();
    }

    /**
     * Whether the itree is a local varref.
     */
    private static boolean isLocalVarref(Itree itree) {
        return itree instanceof VarrefItree
            && ((VarrefItree) itree).owner() instanceof BindingItree;
    }

}

// vim: et sw=4 sts=4 fdm=marker
