package org.kink_lang.kink.internal.program.itreeoptimize.ssafy;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;

import org.kink_lang.kink.internal.program.itree.*;

/**
 * Conversion to static single assignment (SSA) form.
 */
public final class Ssafier {

    /**
     * Cannot instantiate.
     */
    private Ssafier() {
    }

    /**
     * Converts {@code funBody} into SSA form.
     *
     * @param funBody a fun body to convert. It must have been checked by {@link SsaChecker}.
     * @param uniqSupplier supplies unique strings used for LocalVar.Generated.
     * @return ssa-fied fun body.
     */
    public static Itree ssafy(Itree funBody, Supplier<String> uniqSupplier) {
        var callback = new Callback(uniqSupplier, Map.of());
        return DeepTransformer.deepTransform(funBody, callback);
    }

    /**
     * For nested funs, LocalVar.Generated should not be created.
     * Conversion should have been done when they are first ssafied.
     */
    static String noLvarGenerated() {
        throw new IllegalStateException("LocalVar.Generated must not be created");
    }

    /**
     * Callback for conversion.
     *
     * Callback instance is created for each level of {@link FastFunItree}.
     */
    public static class Callback implements DeepTransformer.Callback {

        /** Supplies unique strings. */
        private final Supplier<String> uniqSupplier;

        /** Mapping from original lvars to generated lvars. */
        private final Map<LocalVar.Original, LocalVar> mapping;

        /**
         * Constructs a callback.
         */
        Callback(Supplier<String> uniqSupplier, Map<LocalVar.Original, LocalVar> mapping) {
            this.uniqSupplier = uniqSupplier;
            this.mapping = new HashMap<>(mapping);
        }

        @Override
        public LocalVar derefLvar(LocalVar lvar) {
            return this.mapping.getOrDefault(lvar, lvar);
        }

        @Override
        public LocalVar storeLvar(LocalVar lvar) {
            if (! (lvar instanceof LocalVar.Original olv)) {
                return lvar;
            }
            LocalVar.Generated generated = new LocalVar.Generated(olv.name(), uniqSupplier.get());
            this.mapping.put(olv, generated);
            return generated;
        }

        @Override
        public Itree itree(Itree itree) {
            if (! (itree instanceof FastFunItree fun)) {
                return itree;
            }

            boolean overlaps = fun.freeLvars().stream().anyMatch(this.mapping::containsKey);
            if (! overlaps) {
                return itree;
            }

            var callback = new Callback(Ssafier::noLvarGenerated, this.mapping);
            var body = DeepTransformer.deepTransform(fun.body(), callback);
            return new FastFunItree(body, fun.pos());
        }

    }

}

// vim: et sw=4 sts=4 fdm=marker
