/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.stomp.lite.handler;

import io.vertx.core.Handler;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.ServerWebSocket;
import io.vertx.core.net.SocketAddress;
import io.vertx.ext.stomp.lite.StompServerConnection;
import io.vertx.ext.stomp.lite.StompServerHandler;
import io.vertx.ext.stomp.lite.StompServerHandlerFactory;
import io.vertx.ext.stomp.lite.StompServerOptions;
import io.vertx.ext.stomp.lite.frame.Frame;
import io.vertx.ext.stomp.lite.frame.Frames;
import io.vertx.ext.stomp.lite.frame.Headers;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import javax.security.cert.X509Certificate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DefaultStompServerConnection
implements Handler<Frame>,
StompServerConnection {
    private static final Logger log = LoggerFactory.getLogger(DefaultStompServerConnection.class);
    private final ServerWebSocket serverWebSocket;
    private final Vertx vertx;
    private final StompServerOptions options;
    private final StompServerHandler stompServerHandler;
    private boolean connected = false;
    private boolean closed = false;
    private volatile long lastClientActivity;
    private volatile long lastServerActivity;
    private long serverHeartbeat = -1L;
    private long clientHeartbeat = -1L;

    DefaultStompServerConnection(ServerWebSocket serverWebSocket, Vertx vertx, StompServerOptions options, StompServerHandlerFactory factory) {
        this.serverWebSocket = serverWebSocket;
        this.vertx = vertx;
        this.options = options;
        this.stompServerHandler = factory.create(this);
        if (log.isDebugEnabled()) {
            log.debug("New Stomp Connection. Host: " + serverWebSocket.remoteAddress().host());
        }
    }

    @Override
    public String binaryHandlerID() {
        return this.serverWebSocket.binaryHandlerID();
    }

    @Override
    public String textHandlerID() {
        return this.serverWebSocket.textHandlerID();
    }

    @Override
    public SSLSession sslSession() {
        return this.serverWebSocket.sslSession();
    }

    @Override
    public X509Certificate[] peerCertificateChain() throws SSLPeerUnverifiedException {
        return this.serverWebSocket.peerCertificateChain();
    }

    @Override
    public SocketAddress remoteAddress() {
        return this.serverWebSocket.remoteAddress();
    }

    @Override
    public SocketAddress localAddress() {
        return this.serverWebSocket.localAddress();
    }

    @Override
    public boolean isSsl() {
        return this.serverWebSocket.isSsl();
    }

    @Override
    public Promise<Void> write(Frame frame) {
        return this.write(frame.toBuffer(this.options.isTrailingLine()));
    }

    @Override
    public Promise<Void> write(Buffer buffer) {
        Promise ret = Promise.promise();
        this.onServerActivity();
        try {
            this.serverWebSocket.writeBinaryMessage(buffer, (Handler)ret);
        }
        catch (Exception e) {
            ret.fail((Throwable)e);
        }
        return ret;
    }

    @Override
    public Promise<Void> sendReceiptIfNeeded(Frame frame) {
        Promise ret = Promise.promise();
        String receipt = frame.getReceipt();
        if (receipt != null) {
            this.write(Frames.createReceiptFrame(receipt, Headers.create())).future().onComplete((Handler)ret);
        } else {
            ret.complete();
        }
        return ret;
    }

    @Override
    public Promise<Void> sendError(Throwable throwable) {
        return this.write(Frames.createErrorFrame(throwable, this.options.isDebugEnabled()));
    }

    @Override
    public Promise<Void> sendErrorAndDisconnect(Throwable throwable) {
        if (log.isDebugEnabled()) {
            log.debug("Sending Error and disconnecting client. Host: " + this.serverWebSocket.remoteAddress().host(), throwable);
        }
        Promise ret = Promise.promise();
        this.sendError(throwable).future().onComplete(event -> {
            this.close();
            if (event.succeeded()) {
                ret.complete();
            } else {
                ret.fail(event.cause());
            }
        });
        return ret;
    }

    public void clientCausedException(Throwable t, boolean sendErrorFrame) {
        try {
            this.stompServerHandler.exception(t);
        }
        catch (Exception e) {
            log.error("StompServerHandler.exception handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
        }
        if (sendErrorFrame) {
            this.logIfFailed(this.sendErrorAndDisconnect(t), "Problem sending ERROR frame to client");
        } else {
            this.close();
        }
    }

    @Override
    public void pause() {
        if (!this.closed) {
            this.serverWebSocket.pause();
        }
    }

    @Override
    public void resume() {
        if (!this.closed) {
            this.serverWebSocket.resume();
        }
    }

    @Override
    public void fetch(long amount) {
        if (!this.closed) {
            this.serverWebSocket.fetch(amount);
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            if (log.isDebugEnabled()) {
                log.debug("Closing Stomp Connection. Host: " + this.serverWebSocket.remoteAddress().host());
            }
            this.connected = false;
            try {
                this.cancelHeartbeat();
            }
            catch (Exception e) {
                log.error("StompServerHandler unhandled error on cancelHeartbeat", (Throwable)e);
            }
            try {
                this.stompServerHandler.closed();
            }
            catch (Exception e) {
                log.error("StompServerHandler.disconnected() handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
            }
            try {
                if (!this.serverWebSocket.isClosed()) {
                    this.serverWebSocket.close();
                }
            }
            catch (Exception e) {
                log.warn("Error closing serverWebSocket.", (Throwable)e);
            }
            this.closed = true;
        }
    }

    public void handle(Frame frame) {
        if (!this.closed) {
            try {
                switch (frame.getCommand()) {
                    case CONNECT: {
                        if (this.connected) {
                            this.clientCausedException(new IllegalStateException("CONNECT has already been called."), true);
                        }
                        this.onConnect(frame);
                        break;
                    }
                    case SEND: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.send(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.send handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case SUBSCRIBE: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.subscribe(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.subscribe handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case UNSUBSCRIBE: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.unsubscribe(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.unsubscribe handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case BEGIN: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.begin(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.begin handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case ABORT: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.abort(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.abort handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case COMMIT: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.commit(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.commit handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case ACK: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.ack(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.ack handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case NACK: {
                        this.ensureConnected();
                        this.onClientActivity();
                        try {
                            this.stompServerHandler.nack(frame);
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.nack handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        break;
                    }
                    case DISCONNECT: {
                        this.ensureConnected();
                        this.onClientActivity();
                        this.sendReceiptIfNeeded(frame);
                        try {
                            this.stompServerHandler.disconnected();
                        }
                        catch (Exception e) {
                            log.error("StompServerHandler.disconnected handler threw an exception.. You should fix your handler not to throw exceptions.", (Throwable)e);
                        }
                        this.close();
                        break;
                    }
                    case PING: {
                        this.ensureConnected();
                        this.onClientActivity();
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown command");
                    }
                }
            }
            catch (Exception e) {
                this.clientCausedException(e, false);
            }
        } else {
            log.error("THIS SHOULD NEVER HAPPEN!! Frame Handler called after close.");
        }
    }

    private void ensureConnected() {
        if (!this.connected) {
            throw new IllegalStateException("Client must provide a connect frame before any other frames");
        }
    }

    public boolean isConnected() {
        return this.connected;
    }

    private void onConnect(Frame frame) {
        ArrayList<String> accepted = new ArrayList<String>();
        String accept = frame.getHeader("accept-version");
        if (accept == null) {
            accepted.add("1.2");
        } else {
            accepted.addAll(Arrays.asList(accept.split(",")));
        }
        String version = this.negotiate(accepted);
        if (version == null) {
            throw new IllegalStateException("Client protocol requirement does not mach versions supported by the server.");
        }
        this.stompServerHandler.authenticate(frame.getHeaders()).future().onComplete(authenticatePromise -> {
            if (authenticatePromise.succeeded()) {
                Headers headers = Headers.create((Map)authenticatePromise.result());
                headers.add("version", version);
                headers.add("heart-beat", Frame.Heartbeat.create(this.options.getHeartbeat()).toString());
                this.write(new Frame(Frame.Command.CONNECTED, headers, null)).future().onComplete(writePromise -> {
                    if (writePromise.succeeded()) {
                        Frame.Heartbeat clientHeartbeat = Frame.Heartbeat.parse(frame.getHeader("heart-beat"));
                        Frame.Heartbeat serverHeartbeat = Frame.Heartbeat.create(this.options.getHeartbeat());
                        long clientHeartbeatPeriod = Frame.Heartbeat.computeClientHeartbeatPeriod(clientHeartbeat, serverHeartbeat);
                        long serverHeartbeatPeriod = Frame.Heartbeat.computeServerHeartbeatPeriod(clientHeartbeat, serverHeartbeat);
                        this.onClientActivity();
                        this.configureHeartbeat(clientHeartbeatPeriod, serverHeartbeatPeriod);
                        if (log.isDebugEnabled()) {
                            log.debug("Stomp client authenticated. Host: " + this.serverWebSocket.remoteAddress().host());
                        }
                        this.connected = true;
                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("Could not send CONNECTED frame. Host: " + this.serverWebSocket.remoteAddress().host(), writePromise.cause());
                        }
                        this.close();
                    }
                });
            } else {
                this.logIfFailed(this.sendErrorAndDisconnect(authenticatePromise.cause()), "Problem Sending Authentication Error to client");
            }
        });
    }

    private String negotiate(List<String> accepted) {
        List<String> supported = Collections.singletonList("1.2");
        for (String v : supported) {
            if (!accepted.contains(v)) continue;
            return v;
        }
        return null;
    }

    private void cancelHeartbeat() {
        if (this.serverHeartbeat >= 0L) {
            this.vertx.cancelTimer(this.serverHeartbeat);
            this.serverHeartbeat = -1L;
        }
        if (this.clientHeartbeat >= 0L) {
            this.vertx.cancelTimer(this.clientHeartbeat);
            this.clientHeartbeat = -1L;
        }
    }

    private void onClientActivity() {
        this.lastClientActivity = System.nanoTime();
    }

    private void onServerActivity() {
        this.lastServerActivity = System.nanoTime();
    }

    private void ping() {
        this.serverWebSocket.writeBinaryMessage(Buffer.buffer((String)"\n"));
    }

    private void configureHeartbeat(long clientHeartbeatPeriod, long serverHeartbeatPeriod) {
        if (serverHeartbeatPeriod > 0L) {
            this.serverHeartbeat = this.vertx.setPeriodic(serverHeartbeatPeriod, event -> {
                long delta = System.nanoTime() - this.lastServerActivity;
                long deltaInMs = TimeUnit.MILLISECONDS.convert(delta, TimeUnit.NANOSECONDS);
                if (deltaInMs >= serverHeartbeatPeriod) {
                    this.ping();
                }
            });
        }
        if (clientHeartbeatPeriod > 0L) {
            this.clientHeartbeat = this.vertx.setPeriodic(clientHeartbeatPeriod, l -> {
                long delta = System.nanoTime() - this.lastClientActivity;
                long deltaInMs = TimeUnit.MILLISECONDS.convert(delta, TimeUnit.NANOSECONDS);
                if (deltaInMs > clientHeartbeatPeriod * 2L) {
                    if (log.isDebugEnabled()) {
                        log.debug("Disconnecting client " + this.serverWebSocket.remoteAddress().host() + " - no client activity in the last " + deltaInMs + " ms");
                    }
                    this.close();
                }
            });
        }
    }

    private void logIfFailed(Promise<Void> promise, String message) {
        if (log.isDebugEnabled()) {
            promise.future().onComplete(event -> {
                if (event.failed()) {
                    log.debug(message, event.cause());
                }
            });
        }
    }
}

