/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.sink.kite;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.security.PrivilegedExceptionAction;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericDatumReader;
import org.apache.avro.generic.GenericRecord;
import org.apache.avro.io.BinaryDecoder;
import org.apache.avro.io.DatumReader;
import org.apache.avro.io.Decoder;
import org.apache.avro.io.DecoderFactory;
import org.apache.flume.Channel;
import org.apache.flume.Context;
import org.apache.flume.Event;
import org.apache.flume.EventDeliveryException;
import org.apache.flume.Sink;
import org.apache.flume.Transaction;
import org.apache.flume.conf.Configurable;
import org.apache.flume.instrumentation.SinkCounter;
import org.apache.flume.sink.AbstractSink;
import org.apache.flume.sink.kite.DatasetSinkConstants;
import org.apache.flume.sink.kite.KerberosUtil;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.security.UserGroupInformation;
import org.kitesdk.data.Dataset;
import org.kitesdk.data.DatasetDescriptor;
import org.kitesdk.data.DatasetException;
import org.kitesdk.data.DatasetWriter;
import org.kitesdk.data.Datasets;
import org.kitesdk.data.View;
import org.kitesdk.data.spi.Registration;
import org.kitesdk.data.spi.URIBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatasetSink
extends AbstractSink
implements Configurable {
    private static final Logger LOG = LoggerFactory.getLogger(DatasetSink.class);
    static Configuration conf = new Configuration();
    private String datasetName = null;
    private long batchSize = DatasetSinkConstants.DEFAULT_BATCH_SIZE;
    private URI target = null;
    private Schema targetSchema = null;
    private DatasetWriter<GenericRecord> writer = null;
    private UserGroupInformation login = null;
    private SinkCounter counter = null;
    private int rollIntervalS = DatasetSinkConstants.DEFAULT_ROLL_INTERVAL;
    private long lastRolledMs = 0L;
    private GenericRecord datum = null;
    private boolean reuseDatum = true;
    private BinaryDecoder decoder = null;
    private LoadingCache<Schema, DatumReader<GenericRecord>> readers = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<Schema, DatumReader<GenericRecord>>(){

        public DatumReader<GenericRecord> load(Schema schema) {
            return new GenericDatumReader(schema, DatasetSink.this.targetSchema);
        }
    });
    private static LoadingCache<String, Schema> schemasFromLiteral = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<String, Schema>(){

        public Schema load(String literal) {
            Preconditions.checkNotNull((Object)literal, (Object)"Schema literal cannot be null without a Schema URL");
            return new Schema.Parser().parse(literal);
        }
    });
    private static LoadingCache<String, Schema> schemasFromURL = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<String, Schema>(){

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Schema load(String url) throws IOException {
            Schema.Parser parser = new Schema.Parser();
            InputStream is = null;
            try {
                FileSystem fs = FileSystem.get((URI)URI.create(url), (Configuration)conf);
                is = url.toLowerCase().startsWith("hdfs:/") ? fs.open(new Path(url)) : new URL(url).openStream();
                Schema schema = parser.parse(is);
                return schema;
            }
            finally {
                if (is != null) {
                    is.close();
                }
            }
        }
    });

    protected List<String> allowedFormats() {
        return Lists.newArrayList((Object[])new String[]{"avro", "parquet"});
    }

    public void configure(Context context) {
        String datasetURI;
        this.login = KerberosUtil.login(context.getString("auth.kerberosPrincipal"), context.getString("auth.kerberosKeytab"));
        String effectiveUser = context.getString("auth.proxyUser");
        if (effectiveUser != null) {
            this.login = KerberosUtil.proxyAs(effectiveUser, this.login);
        }
        if ((datasetURI = context.getString("kite.dataset.uri")) != null) {
            this.target = URI.create(datasetURI);
            this.datasetName = DatasetSink.uriToName(this.target);
        } else {
            String repositoryURI = context.getString("kite.repo.uri");
            Preconditions.checkNotNull((Object)repositoryURI, (Object)"Repository URI is missing");
            this.datasetName = context.getString("kite.dataset.name");
            Preconditions.checkNotNull((Object)this.datasetName, (Object)"Dataset name is missing");
            this.target = new URIBuilder(repositoryURI, this.datasetName).build();
        }
        this.setName(this.target.toString());
        this.batchSize = context.getLong("kite.batchSize", Long.valueOf(DatasetSinkConstants.DEFAULT_BATCH_SIZE));
        this.rollIntervalS = context.getInteger("kite.rollInterval", Integer.valueOf(DatasetSinkConstants.DEFAULT_ROLL_INTERVAL));
        this.counter = new SinkCounter(this.datasetName);
    }

    public synchronized void start() {
        this.lastRolledMs = System.currentTimeMillis();
        this.counter.start();
        LOG.info("Started DatasetSink " + this.getName());
        super.start();
    }

    @VisibleForTesting
    public void roll() {
        this.lastRolledMs = 0L;
    }

    public synchronized void stop() {
        this.counter.stop();
        if (this.writer != null) {
            this.writer.close();
            this.writer = null;
            this.lastRolledMs = System.currentTimeMillis();
        }
        LOG.info("Stopped dataset sink: " + this.getName());
        super.stop();
    }

    public Sink.Status process() throws EventDeliveryException {
        if (this.writer == null) {
            try {
                this.writer = this.newWriter(this.login, this.target);
            }
            catch (DatasetException e) {
                throw new EventDeliveryException("Cannot write to " + this.getName(), (Throwable)e);
            }
        }
        if ((System.currentTimeMillis() - this.lastRolledMs) / 1000L > (long)this.rollIntervalS) {
            this.writer.close();
            this.writer = this.newWriter(this.login, this.target);
            this.lastRolledMs = System.currentTimeMillis();
            LOG.info("Rolled writer for " + this.getName());
        }
        Channel channel = this.getChannel();
        Transaction transaction = null;
        try {
            Sink.Status status;
            Event event;
            long processedEvents;
            transaction = channel.getTransaction();
            transaction.begin();
            for (processedEvents = 0L; processedEvents < this.batchSize && (event = channel.take()) != null; ++processedEvents) {
                this.datum = this.deserialize(event, this.reuseDatum ? this.datum : null);
                this.writer.write((Object)this.datum);
            }
            this.writer.flush();
            transaction.commit();
            if (processedEvents == 0L) {
                this.counter.incrementBatchEmptyCount();
                status = Sink.Status.BACKOFF;
                return status;
            }
            if (processedEvents < this.batchSize) {
                this.counter.incrementBatchUnderflowCount();
            } else {
                this.counter.incrementBatchCompleteCount();
            }
            this.counter.addToEventDrainSuccessCount(processedEvents);
            status = Sink.Status.READY;
            return status;
        }
        catch (Throwable th) {
            if (transaction != null) {
                try {
                    transaction.rollback();
                }
                catch (Exception ex) {
                    LOG.error("Transaction rollback failed", (Throwable)ex);
                    throw Throwables.propagate((Throwable)ex);
                }
            }
            this.writer.close();
            this.writer = null;
            this.lastRolledMs = System.currentTimeMillis();
            Throwables.propagateIfInstanceOf((Throwable)th, Error.class);
            Throwables.propagateIfInstanceOf((Throwable)th, EventDeliveryException.class);
            throw new EventDeliveryException(th);
        }
        finally {
            if (transaction != null) {
                transaction.close();
            }
        }
    }

    private DatasetWriter<GenericRecord> newWriter(UserGroupInformation login, final URI uri) {
        View view = (View)KerberosUtil.runPrivileged(login, new PrivilegedExceptionAction<Dataset<GenericRecord>>(){

            @Override
            public Dataset<GenericRecord> run() {
                return (Dataset)Datasets.load((URI)uri);
            }
        });
        DatasetDescriptor descriptor = view.getDataset().getDescriptor();
        String formatName = descriptor.getFormat().getName();
        Preconditions.checkArgument((boolean)this.allowedFormats().contains(formatName), (Object)("Unsupported format: " + formatName));
        Schema newSchema = descriptor.getSchema();
        if (this.targetSchema == null || !newSchema.equals((Object)this.targetSchema)) {
            this.targetSchema = descriptor.getSchema();
            this.readers.invalidateAll();
        }
        this.reuseDatum = !"parquet".equals(formatName);
        this.datasetName = view.getDataset().getName();
        return view.newWriter();
    }

    private GenericRecord deserialize(Event event, GenericRecord reuse) throws EventDeliveryException {
        this.decoder = DecoderFactory.get().binaryDecoder(event.getBody(), this.decoder);
        DatumReader reader = (DatumReader)this.readers.getUnchecked((Object)DatasetSink.schema(event));
        try {
            return (GenericRecord)reader.read((Object)reuse, (Decoder)this.decoder);
        }
        catch (IOException ex) {
            throw new EventDeliveryException("Cannot deserialize event", (Throwable)ex);
        }
    }

    private static Schema schema(Event event) throws EventDeliveryException {
        Map headers = event.getHeaders();
        String schemaURL = (String)headers.get("flume.avro.schema.url");
        try {
            if (headers.get("flume.avro.schema.url") != null) {
                return (Schema)schemasFromURL.get((Object)schemaURL);
            }
            return (Schema)schemasFromLiteral.get(headers.get("flume.avro.schema.literal"));
        }
        catch (ExecutionException ex) {
            throw new EventDeliveryException("Cannot get schema", ex.getCause());
        }
    }

    private static String uriToName(URI uri) {
        return (String)((Map)Registration.lookupDatasetUri((URI)URI.create(uri.getRawSchemeSpecificPart())).second()).get("dataset");
    }
}

