/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flume.channel.file;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.LongBuffer;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.flume.channel.file.BadCheckpointException;
import org.apache.flume.channel.file.EventQueueBackingStore;
import org.apache.flume.channel.file.FlumeEventPointer;
import org.mapdb.DB;
import org.mapdb.DBMaker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class FlumeEventQueue {
    private static final Logger LOG = LoggerFactory.getLogger(FlumeEventQueue.class);
    private static final int EMPTY = 0;
    private final EventQueueBackingStore backingStore;
    private final String channelNameDescriptor;
    private final InflightEventWrapper inflightTakes;
    private final InflightEventWrapper inflightPuts;
    private long searchTime = 0L;
    private long searchCount = 0L;
    private long copyTime = 0L;
    private long copyCount = 0L;
    private DB db;
    private Set<Long> queueSet;

    FlumeEventQueue(EventQueueBackingStore backingStore, File inflightTakesFile, File inflightPutsFile, File queueSetDBDir) throws Exception {
        Preconditions.checkArgument((backingStore.getCapacity() > 0 ? 1 : 0) != 0, (Object)"Capacity must be greater than zero");
        Preconditions.checkNotNull((Object)backingStore, (Object)"backingStore");
        this.channelNameDescriptor = "[channel=" + backingStore.getName() + "]";
        Preconditions.checkNotNull((Object)inflightTakesFile, (Object)"inflightTakesFile");
        Preconditions.checkNotNull((Object)inflightPutsFile, (Object)"inflightPutsFile");
        Preconditions.checkNotNull((Object)queueSetDBDir, (Object)"queueSetDBDir");
        this.backingStore = backingStore;
        try {
            this.inflightPuts = new InflightEventWrapper(inflightPutsFile);
            this.inflightTakes = new InflightEventWrapper(inflightTakesFile);
        }
        catch (Exception e) {
            LOG.error("Could not read checkpoint.", (Throwable)e);
            throw e;
        }
        if (queueSetDBDir.isDirectory()) {
            FileUtils.deleteDirectory((File)queueSetDBDir);
        } else if (queueSetDBDir.isFile() && !queueSetDBDir.delete()) {
            throw new IOException("QueueSetDir " + queueSetDBDir + " is a file and" + " could not be deleted");
        }
        if (!queueSetDBDir.mkdirs()) {
            throw new IllegalStateException("Could not create QueueSet Dir " + queueSetDBDir);
        }
        File dbFile = new File(queueSetDBDir, "db");
        this.db = DBMaker.newFileDB((File)dbFile).closeOnJvmShutdown().transactionDisable().syncOnCommitDisable().deleteFilesAfterClose().cacheDisable().make();
        this.queueSet = this.db.createTreeSet("QueueSet").make();
        long start = System.currentTimeMillis();
        for (int i = 0; i < backingStore.getSize(); ++i) {
            this.queueSet.add(this.get(i));
        }
        LOG.info("QueueSet population inserting " + backingStore.getSize() + " took " + (System.currentTimeMillis() - start));
    }

    SetMultimap<Long, Long> deserializeInflightPuts() throws IOException, BadCheckpointException {
        return this.inflightPuts.deserialize();
    }

    SetMultimap<Long, Long> deserializeInflightTakes() throws IOException, BadCheckpointException {
        return this.inflightTakes.deserialize();
    }

    synchronized long getLogWriteOrderID() {
        return this.backingStore.getLogWriteOrderID();
    }

    synchronized boolean checkpoint(boolean force) throws Exception {
        if (!(this.backingStore.syncRequired() || this.inflightTakes.syncRequired() || force)) {
            LOG.debug("Checkpoint not required");
            return false;
        }
        this.backingStore.beginCheckpoint();
        this.inflightPuts.serializeAndWrite();
        this.inflightTakes.serializeAndWrite();
        this.backingStore.checkpoint();
        return true;
    }

    synchronized FlumeEventPointer removeHead(long transactionID) {
        if (this.backingStore.getSize() == 0) {
            return null;
        }
        long value = this.remove(0, transactionID);
        Preconditions.checkState((value != 0L ? 1 : 0) != 0, (Object)("Empty value " + this.channelNameDescriptor));
        FlumeEventPointer ptr = FlumeEventPointer.fromLong(value);
        this.backingStore.decrementFileID(ptr.getFileID());
        return ptr;
    }

    synchronized boolean addHead(FlumeEventPointer e) {
        if (this.backingStore.getSize() == this.backingStore.getCapacity()) {
            LOG.error("Could not reinsert to queue, events which were taken but not committed. Please report this issue.");
            return false;
        }
        long value = e.toLong();
        Preconditions.checkArgument((value != 0L ? 1 : 0) != 0);
        this.backingStore.incrementFileID(e.getFileID());
        this.add(0, value);
        return true;
    }

    synchronized boolean addTail(FlumeEventPointer e) {
        if (this.getSize() == this.backingStore.getCapacity()) {
            return false;
        }
        long value = e.toLong();
        Preconditions.checkArgument((value != 0L ? 1 : 0) != 0);
        this.backingStore.incrementFileID(e.getFileID());
        this.add(this.backingStore.getSize(), value);
        return true;
    }

    synchronized void addWithoutCommit(FlumeEventPointer e, long transactionID) {
        this.inflightPuts.addEvent(transactionID, e.toLong());
    }

    synchronized boolean remove(FlumeEventPointer e) {
        long value = e.toLong();
        Preconditions.checkArgument((value != 0L ? 1 : 0) != 0);
        if (this.queueSet == null) {
            throw new IllegalStateException("QueueSet is null, thus replayComplete has been called which is illegal");
        }
        if (!this.queueSet.contains(value)) {
            return false;
        }
        ++this.searchCount;
        long start = System.currentTimeMillis();
        for (int i = 0; i < this.backingStore.getSize(); ++i) {
            if (this.get(i) != value) continue;
            this.remove(i, 0L);
            FlumeEventPointer ptr = FlumeEventPointer.fromLong(value);
            this.backingStore.decrementFileID(ptr.getFileID());
            this.searchTime += System.currentTimeMillis() - start;
            return true;
        }
        this.searchTime += System.currentTimeMillis() - start;
        return false;
    }

    synchronized SortedSet<Integer> getFileIDs() {
        TreeSet<Integer> fileIDs = new TreeSet<Integer>((SortedSet<Integer>)this.backingStore.getReferenceCounts());
        fileIDs.addAll(this.inflightPuts.getFileIDs());
        fileIDs.addAll(this.inflightTakes.getFileIDs());
        return fileIDs;
    }

    protected long get(int index) {
        if (index < 0 || index > this.backingStore.getSize() - 1) {
            throw new IndexOutOfBoundsException(String.valueOf(index) + this.channelNameDescriptor);
        }
        return this.backingStore.get(index);
    }

    private void set(int index, long value) {
        if (index < 0 || index > this.backingStore.getSize() - 1) {
            throw new IndexOutOfBoundsException(String.valueOf(index) + this.channelNameDescriptor);
        }
        this.backingStore.put(index, value);
    }

    protected boolean add(int index, long value) {
        if (index < 0 || index > this.backingStore.getSize()) {
            throw new IndexOutOfBoundsException(String.valueOf(index) + this.channelNameDescriptor);
        }
        if (this.backingStore.getSize() == this.backingStore.getCapacity()) {
            return false;
        }
        this.backingStore.setSize(this.backingStore.getSize() + 1);
        if (index <= this.backingStore.getSize() / 2) {
            this.backingStore.setHead(this.backingStore.getHead() - 1);
            if (this.backingStore.getHead() < 0) {
                this.backingStore.setHead(this.backingStore.getCapacity() - 1);
            }
            for (int i = 0; i < index; ++i) {
                this.set(i, this.get(i + 1));
            }
        } else {
            for (int i = this.backingStore.getSize() - 1; i > index; --i) {
                this.set(i, this.get(i - 1));
            }
        }
        this.set(index, value);
        if (this.queueSet != null) {
            this.queueSet.add(value);
        }
        return true;
    }

    synchronized void completeTransaction(long transactionID) {
        if (!this.inflightPuts.completeTransaction(transactionID)) {
            this.inflightTakes.completeTransaction(transactionID);
        }
    }

    protected synchronized long remove(int index, long transactionID) {
        if (index < 0 || index > this.backingStore.getSize() - 1) {
            throw new IndexOutOfBoundsException("index = " + index + ", queueSize " + this.backingStore.getSize() + " " + this.channelNameDescriptor);
        }
        ++this.copyCount;
        long start = System.currentTimeMillis();
        long value = this.get(index);
        if (this.queueSet != null) {
            this.queueSet.remove(value);
        }
        if (transactionID != 0L) {
            this.inflightTakes.addEvent(transactionID, value);
        }
        if (index > this.backingStore.getSize() / 2) {
            for (int i = index; i < this.backingStore.getSize() - 1; ++i) {
                long rightValue = this.get(i + 1);
                this.set(i, rightValue);
            }
            this.set(this.backingStore.getSize() - 1, 0L);
        } else {
            for (int i = index - 1; i >= 0; --i) {
                long leftValue = this.get(i);
                this.set(i + 1, leftValue);
            }
            this.set(0, 0L);
            this.backingStore.setHead(this.backingStore.getHead() + 1);
            if (this.backingStore.getHead() == this.backingStore.getCapacity()) {
                this.backingStore.setHead(0);
            }
        }
        this.backingStore.setSize(this.backingStore.getSize() - 1);
        this.copyTime += System.currentTimeMillis() - start;
        return value;
    }

    protected synchronized int getSize() {
        return this.backingStore.getSize() + this.inflightTakes.getSize();
    }

    public int getCapacity() {
        return this.backingStore.getCapacity();
    }

    synchronized void close() throws IOException {
        try {
            if (this.db != null) {
                this.db.close();
            }
        }
        catch (Exception ex) {
            LOG.warn("Error closing db", (Throwable)ex);
        }
        try {
            this.backingStore.close();
            this.inflightPuts.close();
            this.inflightTakes.close();
        }
        catch (IOException e) {
            LOG.warn("Error closing backing store", (Throwable)e);
        }
    }

    synchronized void replayComplete() {
        String msg = "Search Count = " + this.searchCount + ", Search Time = " + this.searchTime + ", Copy Count = " + this.copyCount + ", Copy Time = " + this.copyTime;
        LOG.info(msg);
        if (this.db != null) {
            this.db.close();
        }
        this.queueSet = null;
        this.db = null;
    }

    @VisibleForTesting
    long getSearchCount() {
        return this.searchCount;
    }

    @VisibleForTesting
    long getCopyCount() {
        return this.copyCount;
    }

    class InflightEventWrapper {
        private SetMultimap<Long, Long> inflightEvents = HashMultimap.create();
        private volatile RandomAccessFile file;
        private volatile FileChannel fileChannel;
        private final MessageDigest digest;
        private final File inflightEventsFile;
        private volatile boolean syncRequired = false;
        private SetMultimap<Long, Integer> inflightFileIDs = HashMultimap.create();

        public InflightEventWrapper(File inflightEventsFile) throws Exception {
            if (!inflightEventsFile.exists()) {
                Preconditions.checkState((boolean)inflightEventsFile.createNewFile(), (Object)("Could notcreate inflight events file: " + inflightEventsFile.getCanonicalPath()));
            }
            this.inflightEventsFile = inflightEventsFile;
            this.file = new RandomAccessFile(inflightEventsFile, "rw");
            this.fileChannel = this.file.getChannel();
            this.digest = MessageDigest.getInstance("MD5");
        }

        public boolean completeTransaction(Long transactionID) {
            if (!this.inflightEvents.containsKey((Object)transactionID)) {
                return false;
            }
            this.inflightEvents.removeAll((Object)transactionID);
            this.inflightFileIDs.removeAll((Object)transactionID);
            this.syncRequired = true;
            return true;
        }

        public void addEvent(Long transactionID, Long pointer) {
            this.inflightEvents.put((Object)transactionID, (Object)pointer);
            this.inflightFileIDs.put((Object)transactionID, (Object)FlumeEventPointer.fromLong(pointer).getFileID());
            this.syncRequired = true;
        }

        public void serializeAndWrite() throws Exception {
            Collection values = this.inflightEvents.values();
            if (!this.fileChannel.isOpen()) {
                this.file = new RandomAccessFile(this.inflightEventsFile, "rw");
                this.fileChannel = this.file.getChannel();
            }
            if (values.isEmpty()) {
                this.file.setLength(0L);
            }
            try {
                int expectedFileSize = (this.inflightEvents.keySet().size() * 2 + values.size()) * 8 + 16;
                this.file.setLength(expectedFileSize);
                Preconditions.checkState((this.file.length() == (long)expectedFileSize ? 1 : 0) != 0, (Object)"Expected File size of inflight events file does not match the current file size. Checkpoint is incomplete.");
                this.file.seek(0L);
                ByteBuffer buffer = ByteBuffer.allocate(expectedFileSize);
                LongBuffer longBuffer = buffer.asLongBuffer();
                for (Long txnID : this.inflightEvents.keySet()) {
                    Set pointers = this.inflightEvents.get((Object)txnID);
                    longBuffer.put(txnID);
                    longBuffer.put(pointers.size());
                    LOG.debug("Number of events inserted into inflights file: " + String.valueOf(pointers.size()) + " file: " + this.inflightEventsFile.getCanonicalPath());
                    long[] written = ArrayUtils.toPrimitive((Long[])pointers.toArray(new Long[0]));
                    longBuffer.put(written);
                }
                byte[] checksum = this.digest.digest(buffer.array());
                this.file.write(checksum);
                buffer.position(0);
                this.fileChannel.write(buffer);
                this.fileChannel.force(true);
                this.syncRequired = false;
            }
            catch (IOException ex) {
                LOG.error("Error while writing checkpoint to disk.", (Throwable)ex);
                throw ex;
            }
        }

        public SetMultimap<Long, Long> deserialize() throws IOException, BadCheckpointException {
            HashMultimap inflights = HashMultimap.create();
            if (!this.fileChannel.isOpen()) {
                this.file = new RandomAccessFile(this.inflightEventsFile, "rw");
                this.fileChannel = this.file.getChannel();
            }
            if (this.file.length() == 0L) {
                return inflights;
            }
            this.file.seek(0L);
            byte[] checksum = new byte[16];
            this.file.read(checksum);
            ByteBuffer buffer = ByteBuffer.allocate((int)(this.file.length() - this.file.getFilePointer()));
            this.fileChannel.read(buffer);
            byte[] fileChecksum = this.digest.digest(buffer.array());
            if (!Arrays.equals(checksum, fileChecksum)) {
                throw new BadCheckpointException("Checksum of inflights file differs from the checksum expected.");
            }
            buffer.position(0);
            LongBuffer longBuffer = buffer.asLongBuffer();
            try {
                block2: while (true) {
                    long txnID = longBuffer.get();
                    int numEvents = (int)longBuffer.get();
                    int i = 0;
                    while (true) {
                        if (i >= numEvents) continue block2;
                        long val = longBuffer.get();
                        inflights.put((Object)txnID, (Object)val);
                        ++i;
                    }
                    break;
                }
            }
            catch (BufferUnderflowException ex) {
                LOG.debug("Reached end of inflights buffer. Long buffer position =" + String.valueOf(longBuffer.position()));
                return inflights;
            }
        }

        public int getSize() {
            return this.inflightEvents.size();
        }

        public boolean syncRequired() {
            return this.syncRequired;
        }

        public Collection<Integer> getFileIDs() {
            return this.inflightFileIDs.values();
        }

        public Collection<Long> getInFlightPointers() {
            return this.inflightEvents.values();
        }

        public void close() throws IOException {
            this.file.close();
        }
    }
}

