/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.SortedSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.KeyValueUtil;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.encoding.DataBlockEncoding;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFilePrettyPrinter;
import org.apache.hadoop.hbase.regionserver.BloomType;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HStore;
import org.apache.hadoop.hbase.regionserver.ScanType;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.regionserver.StoreFileScanner;
import org.apache.hadoop.hbase.regionserver.StoreScanner;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.LoadTestTool;
import org.apache.hadoop.hbase.util.MD5Hash;
import org.apache.hadoop.util.StringUtils;

public class HFileReadWriteTest {
    private static final String TABLE_NAME = "MyTable";
    private static final String OUTPUT_DIR_OPTION = "output_dir";
    private static final String COMPRESSION_OPTION = "compression";
    private static final String BLOOM_FILTER_OPTION = "bloom";
    private static final String BLOCK_SIZE_OPTION = "block_size";
    private static final String DURATION_OPTION = "duration";
    private static final String NUM_THREADS_OPTION = "num_threads";
    private static final Log LOG = LogFactory.getLog(HFileReadWriteTest.class);
    private Workload workload;
    private FileSystem fs;
    private Configuration conf;
    private CacheConfig cacheConf;
    private List<String> inputFileNames;
    private Path outputDir;
    private int numReadThreads;
    private int durationSec;
    private DataBlockEncoding dataBlockEncoding;
    private BloomType bloomType = BloomType.NONE;
    private int blockSize;
    private Compression.Algorithm compression = Compression.Algorithm.NONE;
    private byte[] firstRow;
    private byte[] lastRow;
    private AtomicLong numSeeks = new AtomicLong();
    private AtomicLong numKV = new AtomicLong();
    private AtomicLong totalBytes = new AtomicLong();
    private byte[] family;
    private long endTime = Long.MAX_VALUE;
    private SortedSet<String> keysRead = new ConcurrentSkipListSet<String>();
    private List<StoreFile> inputStoreFiles;

    public HFileReadWriteTest() {
        this.conf = HBaseConfiguration.create();
        this.cacheConf = new CacheConfig(this.conf);
    }

    public boolean parseOptions(String[] args) {
        CommandLine cmdLine;
        Options options = new Options();
        options.addOption(OUTPUT_DIR_OPTION, true, "Output directory" + Workload.MERGE.onlyUsedFor());
        options.addOption(COMPRESSION_OPTION, true, " Compression type, one of " + Arrays.toString(Compression.Algorithm.values()) + Workload.MERGE.onlyUsedFor());
        options.addOption(BLOOM_FILTER_OPTION, true, "Bloom filter type, one of " + Arrays.toString(BloomType.values()) + Workload.MERGE.onlyUsedFor());
        options.addOption(BLOCK_SIZE_OPTION, true, "HFile block size" + Workload.MERGE.onlyUsedFor());
        options.addOption(DURATION_OPTION, true, "The amount of time to run the random read workload for" + Workload.RANDOM_READS.onlyUsedFor());
        options.addOption(NUM_THREADS_OPTION, true, "The number of random reader threads" + Workload.RANDOM_READS.onlyUsedFor());
        options.addOption(NUM_THREADS_OPTION, true, "The number of random reader threads" + Workload.RANDOM_READS.onlyUsedFor());
        options.addOption(LoadTestTool.OPT_DATA_BLOCK_ENCODING, true, LoadTestTool.OPT_DATA_BLOCK_ENCODING_USAGE);
        options.addOptionGroup(Workload.getOptionGroup());
        if (args.length == 0) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(HFileReadWriteTest.class.getSimpleName(), options, true);
            return false;
        }
        PosixParser parser = new PosixParser();
        try {
            cmdLine = parser.parse(options, args);
        }
        catch (ParseException ex) {
            LOG.error((Object)ex);
            return false;
        }
        this.workload = Workload.fromCmdLine(cmdLine);
        if (this.workload == null) {
            return false;
        }
        this.inputFileNames = cmdLine.getArgList();
        if (this.inputFileNames.size() == 0) {
            LOG.error((Object)"No input file names specified");
            return false;
        }
        if (this.inputFileNames.size() < this.workload.minNumInputFiles) {
            LOG.error((Object)("Too few input files: at least " + this.workload.minNumInputFiles + " required"));
            return false;
        }
        if (this.inputFileNames.size() > this.workload.maxNumInputFiles) {
            LOG.error((Object)("Too many input files: at most " + this.workload.minNumInputFiles + " allowed"));
            return false;
        }
        if (cmdLine.hasOption(COMPRESSION_OPTION)) {
            this.compression = Compression.Algorithm.valueOf((String)cmdLine.getOptionValue(COMPRESSION_OPTION));
        }
        if (cmdLine.hasOption(BLOOM_FILTER_OPTION)) {
            this.bloomType = BloomType.valueOf((String)cmdLine.getOptionValue(BLOOM_FILTER_OPTION));
        }
        if (cmdLine.hasOption(LoadTestTool.OPT_DATA_BLOCK_ENCODING)) {
            this.dataBlockEncoding = DataBlockEncoding.valueOf((String)cmdLine.getOptionValue(LoadTestTool.OPT_DATA_BLOCK_ENCODING));
        }
        this.blockSize = this.conf.getInt("hfile.min.blocksize.size", 65536);
        if (cmdLine.hasOption(BLOCK_SIZE_OPTION)) {
            this.blockSize = Integer.valueOf(cmdLine.getOptionValue(BLOCK_SIZE_OPTION));
        }
        if (this.workload == Workload.MERGE) {
            String outputDirStr = cmdLine.getOptionValue(OUTPUT_DIR_OPTION);
            if (outputDirStr == null) {
                LOG.error((Object)"Output directory is not specified");
                return false;
            }
            this.outputDir = new Path(outputDirStr);
        }
        if (this.workload == Workload.RANDOM_READS) {
            if (!this.requireOptions(cmdLine, new String[]{DURATION_OPTION, NUM_THREADS_OPTION})) {
                return false;
            }
            this.durationSec = Integer.parseInt(cmdLine.getOptionValue(DURATION_OPTION));
            this.numReadThreads = Integer.parseInt(cmdLine.getOptionValue(NUM_THREADS_OPTION));
        }
        Collections.sort(this.inputFileNames);
        return true;
    }

    private boolean requireOptions(CommandLine cmdLine, String[] requiredOptions) {
        for (String option : requiredOptions) {
            if (cmdLine.hasOption(option)) continue;
            LOG.error((Object)("Required option -" + option + " not specified"));
            return false;
        }
        return true;
    }

    public boolean validateConfiguration() throws IOException {
        this.fs = FileSystem.get((Configuration)this.conf);
        for (String inputFileName : this.inputFileNames) {
            Path path = new Path(inputFileName);
            if (!this.fs.exists(path)) {
                LOG.error((Object)("File " + inputFileName + " does not exist"));
                return false;
            }
            if (!this.fs.getFileStatus(path).isDir()) continue;
            LOG.error((Object)(inputFileName + " is a directory"));
            return false;
        }
        if (!(this.outputDir == null || this.fs.exists(this.outputDir) && this.fs.getFileStatus(this.outputDir).isDir())) {
            LOG.error((Object)(this.outputDir.toString() + " does not exist or is not a " + "directory"));
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void runMergeWorkload() throws IOException {
        long maxKeyCount = this.prepareForMerge();
        List scanners = StoreFileScanner.getScannersForStoreFiles(this.inputStoreFiles, (boolean)false, (boolean)false);
        HColumnDescriptor columnDescriptor = new HColumnDescriptor(HFileReadWriteTest.class.getSimpleName());
        columnDescriptor.setBlocksize(this.blockSize);
        columnDescriptor.setBloomFilterType(this.bloomType);
        columnDescriptor.setCompressionType(this.compression);
        columnDescriptor.setDataBlockEncoding(this.dataBlockEncoding);
        HRegionInfo regionInfo = new HRegionInfo();
        HTableDescriptor htd = new HTableDescriptor(TableName.valueOf((String)TABLE_NAME));
        HRegion region = new HRegion(this.outputDir, null, this.fs, this.conf, regionInfo, htd, null);
        HStore store = new HStore(region, columnDescriptor, this.conf);
        StoreFile.Writer writer = store.createWriterInTmp(maxKeyCount, this.compression, false, true);
        StatisticsPrinter statsPrinter = new StatisticsPrinter();
        statsPrinter.startThread();
        try {
            this.performMerge(scanners, store, writer);
            writer.close();
        }
        finally {
            statsPrinter.requestStop();
        }
        Path resultPath = writer.getPath();
        resultPath = this.tryUsingSimpleOutputPath(resultPath);
        long fileSize = this.fs.getFileStatus(resultPath).getLen();
        LOG.info((Object)("Created " + resultPath + ", size " + fileSize));
        System.out.println();
        System.out.println("HFile information for " + resultPath);
        System.out.println();
        HFilePrettyPrinter hfpp = new HFilePrettyPrinter();
        hfpp.run(new String[]{"-m", "-f", resultPath.toString()});
    }

    private Path tryUsingSimpleOutputPath(Path resultPath) throws IOException {
        Path inputPath;
        Path betterOutputPath;
        if (this.inputFileNames.size() == 1 && !this.fs.exists(betterOutputPath = new Path(this.outputDir, (inputPath = new Path(this.inputFileNames.get(0))).getName()))) {
            this.fs.rename(resultPath, betterOutputPath);
            resultPath = betterOutputPath;
        }
        return resultPath;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void performMerge(List<StoreFileScanner> scanners, HStore store, StoreFile.Writer writer) throws IOException {
        StoreScanner scanner = null;
        try {
            Scan scan = new Scan();
            scanner = new StoreScanner((Store)store, store.getScanInfo(), scan, scanners, ScanType.COMPACT_DROP_DELETES, Long.MIN_VALUE, Long.MIN_VALUE);
            ArrayList kvs = new ArrayList();
            while (scanner.next(kvs) || kvs.size() != 0) {
                this.numKV.addAndGet(kvs.size());
                for (Cell c : kvs) {
                    KeyValue kv = KeyValueUtil.ensureKeyValue((Cell)c);
                    this.totalBytes.addAndGet(kv.getLength());
                    writer.append(kv);
                }
                kvs.clear();
            }
        }
        finally {
            if (scanner != null) {
                scanner.close();
            }
        }
    }

    private long prepareForMerge() throws IOException {
        LOG.info((Object)("Merging " + this.inputFileNames));
        LOG.info((Object)("Using block size: " + this.blockSize));
        this.inputStoreFiles = new ArrayList<StoreFile>();
        long maxKeyCount = 0L;
        for (String fileName : this.inputFileNames) {
            Path filePath = new Path(fileName);
            StoreFile sf = this.openStoreFile(filePath, false);
            sf.createReader();
            this.inputStoreFiles.add(sf);
            StoreFile.Reader r = sf.getReader();
            if (r == null) continue;
            long keyCount = r.getFilterEntries();
            maxKeyCount += keyCount;
            LOG.info((Object)("Compacting: " + sf + "; keyCount = " + keyCount + "; Bloom Type = " + r.getBloomFilterType().toString() + "; Size = " + StringUtils.humanReadableInt((long)r.length())));
        }
        return maxKeyCount;
    }

    public HFile.Reader[] getHFileReaders() {
        HFile.Reader[] readers = new HFile.Reader[this.inputStoreFiles.size()];
        for (int i = 0; i < this.inputStoreFiles.size(); ++i) {
            readers[i] = this.inputStoreFiles.get(i).getReader().getHFileReader();
        }
        return readers;
    }

    private StoreFile openStoreFile(Path filePath, boolean blockCache) throws IOException {
        return new StoreFile(this.fs, filePath, this.conf, this.cacheConf, BloomType.ROWCOL);
    }

    public static int charToHex(int c) {
        if (48 <= c && c <= 57) {
            return c - 48;
        }
        if (97 <= c && c <= 102) {
            return 10 + c - 97;
        }
        return -1;
    }

    public static int hexToChar(int h) {
        if (0 <= (h &= 0xFF) && h <= 9) {
            return 48 + h;
        }
        if (10 <= h && h <= 15) {
            return 97 + h - 10;
        }
        return -1;
    }

    public static byte[] createRandomRow(Random rand, byte[] first, byte[] last) {
        int resultLen = Math.max(first.length, last.length);
        int minLen = Math.min(first.length, last.length);
        byte[] result = new byte[resultLen];
        boolean greaterThanFirst = false;
        boolean lessThanLast = false;
        for (int i = 0; i < resultLen; ++i) {
            int r;
            int high;
            boolean isHex = i < minLen && HFileReadWriteTest.charToHex(first[i]) != -1 && HFileReadWriteTest.charToHex(last[i]) != -1;
            int low = greaterThanFirst || i >= first.length ? 0 : first[i] & 0xFF;
            int n = high = lessThanLast || i >= last.length ? 255 : last[i] & 0xFF;
            if (isHex) {
                if (low < 48) {
                    low = 48;
                }
                if (high > 102) {
                    high = 102;
                }
                int lowHex = HFileReadWriteTest.charToHex(low);
                int highHex = HFileReadWriteTest.charToHex(high);
                r = HFileReadWriteTest.hexToChar(lowHex + rand.nextInt(highHex - lowHex + 1));
            } else {
                r = low + rand.nextInt(high - low + 1);
            }
            if (r > low) {
                greaterThanFirst = true;
            }
            if (r < high) {
                lessThanLast = true;
            }
            result[i] = (byte)r;
        }
        if (Bytes.compareTo((byte[])result, (byte[])first) < 0) {
            throw new IllegalStateException("Generated key " + Bytes.toStringBinary((byte[])result) + " is less than the first key " + Bytes.toStringBinary((byte[])first));
        }
        if (Bytes.compareTo((byte[])result, (byte[])last) > 0) {
            throw new IllegalStateException("Generated key " + Bytes.toStringBinary((byte[])result) + " is greater than te last key " + Bytes.toStringBinary((byte[])last));
        }
        return result;
    }

    private static byte[] createRandomQualifier(Random rand) {
        byte[] q = new byte[10 + rand.nextInt(30)];
        rand.nextBytes(q);
        return q;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean runRandomReadWorkload() throws IOException {
        int numFailed;
        int numCompleted;
        block16: {
            if (this.inputFileNames.size() != 1) {
                throw new IOException("Need exactly one input file for random reads: " + this.inputFileNames);
            }
            Path inputPath = new Path(this.inputFileNames.get(0));
            StoreFile storeFile = this.openStoreFile(inputPath, true);
            StoreFile.Reader reader = storeFile.createReader();
            LOG.info((Object)("First key: " + Bytes.toStringBinary((byte[])reader.getFirstKey())));
            LOG.info((Object)("Last key: " + Bytes.toStringBinary((byte[])reader.getLastKey())));
            KeyValue firstKV = KeyValue.createKeyValueFromKey((byte[])reader.getFirstKey());
            this.firstRow = firstKV.getRow();
            KeyValue lastKV = KeyValue.createKeyValueFromKey((byte[])reader.getLastKey());
            this.lastRow = lastKV.getRow();
            byte[] family = firstKV.getFamily();
            if (!Bytes.equals((byte[])family, (byte[])lastKV.getFamily())) {
                LOG.error((Object)("First and last key have different families: " + Bytes.toStringBinary((byte[])family) + " and " + Bytes.toStringBinary((byte[])lastKV.getFamily())));
                return false;
            }
            if (Bytes.equals((byte[])this.firstRow, (byte[])this.lastRow)) {
                LOG.error((Object)("First and last row are the same, cannot run read workload: firstRow=" + Bytes.toStringBinary((byte[])this.firstRow) + ", " + "lastRow=" + Bytes.toStringBinary((byte[])this.lastRow)));
                return false;
            }
            ExecutorService exec = Executors.newFixedThreadPool(this.numReadThreads + 1);
            numCompleted = 0;
            numFailed = 0;
            try {
                ExecutorCompletionService<Boolean> ecs = new ExecutorCompletionService<Boolean>(exec);
                this.endTime = System.currentTimeMillis() + (long)(1000 * this.durationSec);
                boolean pread = true;
                for (int i = 0; i < this.numReadThreads; ++i) {
                    ecs.submit(new RandomReader(i, reader, pread));
                }
                ecs.submit(new StatisticsPrinter());
                block8: while (true) {
                    try {
                        while (true) {
                            Future result;
                            if ((result = ecs.poll(this.endTime + 1000L - System.currentTimeMillis(), TimeUnit.MILLISECONDS)) == null) {
                                break block16;
                            }
                            try {
                                if (((Boolean)result.get()).booleanValue()) {
                                    ++numCompleted;
                                    continue block8;
                                }
                                ++numFailed;
                                continue block8;
                            }
                            catch (ExecutionException e) {
                                LOG.error((Object)"Worker thread failure", e.getCause());
                                ++numFailed;
                                continue;
                            }
                            break;
                        }
                    }
                    catch (InterruptedException ex) {
                        LOG.error((Object)("Interrupted after " + numCompleted + " workers completed"));
                        Thread.currentThread().interrupt();
                        continue;
                    }
                    break;
                }
            }
            finally {
                storeFile.closeReader(true);
                exec.shutdown();
                BlockCache c = this.cacheConf.getBlockCache();
                if (c != null) {
                    c.shutdown();
                }
            }
        }
        LOG.info((Object)("Worker threads completed: " + numCompleted));
        LOG.info((Object)("Worker threads failed: " + numFailed));
        return true;
    }

    public boolean run() throws IOException {
        LOG.info((Object)("Workload: " + (Object)((Object)this.workload)));
        switch (this.workload) {
            case MERGE: {
                this.runMergeWorkload();
                break;
            }
            case RANDOM_READS: {
                return this.runRandomReadWorkload();
            }
            default: {
                LOG.error((Object)("Unknown workload: " + (Object)((Object)this.workload)));
                return false;
            }
        }
        return true;
    }

    private static void failure() {
        System.exit(1);
    }

    public static void main(String[] args) {
        HFileReadWriteTest app = new HFileReadWriteTest();
        if (!app.parseOptions(args)) {
            HFileReadWriteTest.failure();
        }
        try {
            if (!app.validateConfiguration() || !app.run()) {
                HFileReadWriteTest.failure();
            }
        }
        catch (IOException ex) {
            LOG.error((Object)ex);
            HFileReadWriteTest.failure();
        }
    }

    private class StatisticsPrinter
    implements Callable<Boolean> {
        private volatile boolean stopRequested;
        private volatile Thread thread;
        private long totalSeekAndReads;
        private long totalPositionalReads;

        private StatisticsPrinter() {
        }

        public void startThread() {
            new Thread(){

                @Override
                public void run() {
                    try {
                        StatisticsPrinter.this.call();
                    }
                    catch (Exception e) {
                        LOG.error((Object)e);
                    }
                }
            }.start();
        }

        @Override
        public Boolean call() throws Exception {
            long curTime;
            LOG.info((Object)"Starting statistics printer");
            this.thread = Thread.currentThread();
            this.thread.setName(StatisticsPrinter.class.getSimpleName());
            long startTime = System.currentTimeMillis();
            while ((curTime = System.currentTimeMillis()) < HFileReadWriteTest.this.endTime && !this.stopRequested) {
                long elapsedTime = curTime - startTime;
                this.printStats(elapsedTime);
                try {
                    Thread.sleep(1000L - elapsedTime % 1000L);
                }
                catch (InterruptedException iex) {
                    Thread.currentThread().interrupt();
                    if (!this.stopRequested) continue;
                    break;
                }
            }
            this.printStats(curTime - startTime);
            LOG.info((Object)"Stopping statistics printer");
            return true;
        }

        private void printStats(long elapsedTime) {
            long numSeeksL = HFileReadWriteTest.this.numSeeks.get();
            double timeSec = (double)elapsedTime / 1000.0;
            double seekPerSec = (double)numSeeksL / timeSec;
            long kvCount = HFileReadWriteTest.this.numKV.get();
            double kvPerSec = (double)kvCount / timeSec;
            long bytes = HFileReadWriteTest.this.totalBytes.get();
            double bytesPerSec = (double)bytes / timeSec;
            this.totalSeekAndReads += (long)HFile.getReadOps();
            this.totalPositionalReads += (long)HFile.getPreadOps();
            long totalBlocksRead = this.totalSeekAndReads + this.totalPositionalReads;
            double blkReadPerSec = (double)totalBlocksRead / timeSec;
            double seekReadPerSec = (double)this.totalSeekAndReads / timeSec;
            double preadPerSec = (double)this.totalPositionalReads / timeSec;
            boolean isRead = HFileReadWriteTest.this.workload == Workload.RANDOM_READS;
            StringBuilder sb = new StringBuilder();
            sb.append("Time: " + (long)timeSec + " sec");
            if (isRead) {
                sb.append(", seek/sec: " + (long)seekPerSec);
            }
            sb.append(", kv/sec: " + (long)kvPerSec);
            sb.append(", bytes/sec: " + (long)bytesPerSec);
            sb.append(", blk/sec: " + (long)blkReadPerSec);
            sb.append(", total KV: " + HFileReadWriteTest.this.numKV);
            sb.append(", total bytes: " + HFileReadWriteTest.this.totalBytes);
            sb.append(", total blk: " + totalBlocksRead);
            sb.append(", seekRead/sec: " + (long)seekReadPerSec);
            sb.append(", pread/sec: " + (long)preadPerSec);
            if (isRead) {
                sb.append(", unique keys: " + (long)HFileReadWriteTest.this.keysRead.size());
            }
            LOG.info((Object)sb.toString());
        }

        public void requestStop() {
            this.stopRequested = true;
            if (this.thread != null) {
                this.thread.interrupt();
            }
        }
    }

    private class RandomReader
    implements Callable<Boolean> {
        private int readerId;
        private StoreFile.Reader reader;
        private boolean pread;

        public RandomReader(int readerId, StoreFile.Reader reader, boolean pread) {
            this.readerId = readerId;
            this.reader = reader;
            this.pread = pread;
        }

        @Override
        public Boolean call() throws Exception {
            Thread.currentThread().setName("reader " + this.readerId);
            Random rand = new Random();
            StoreFileScanner scanner = this.reader.getStoreFileScanner(true, this.pread);
            block2: while (System.currentTimeMillis() < HFileReadWriteTest.this.endTime) {
                boolean seekResult;
                byte[] row = HFileReadWriteTest.createRandomRow(rand, HFileReadWriteTest.this.firstRow, HFileReadWriteTest.this.lastRow);
                KeyValue kvToSeek = new KeyValue(row, HFileReadWriteTest.this.family, HFileReadWriteTest.createRandomQualifier(rand));
                if (rand.nextDouble() < 1.0E-4) {
                    LOG.info((Object)("kvToSeek=" + kvToSeek));
                }
                try {
                    seekResult = scanner.seek(kvToSeek);
                }
                catch (IOException ex) {
                    throw new IOException("Seek failed for key " + kvToSeek + ", pread=" + this.pread, ex);
                }
                HFileReadWriteTest.this.numSeeks.incrementAndGet();
                if (!seekResult) {
                    this.error("Seek returned false for row " + Bytes.toStringBinary((byte[])row));
                    return false;
                }
                for (int i = 0; i < rand.nextInt(10) + 1; ++i) {
                    KeyValue kv = scanner.next();
                    HFileReadWriteTest.this.numKV.incrementAndGet();
                    if (i == 0 && kv == null) {
                        this.error("scanner.next() returned null at the first iteration for row " + Bytes.toStringBinary((byte[])row));
                        return false;
                    }
                    if (kv == null) continue block2;
                    String keyHashStr = MD5Hash.getMD5AsHex((byte[])kv.getKey());
                    HFileReadWriteTest.this.keysRead.add(keyHashStr);
                    HFileReadWriteTest.this.totalBytes.addAndGet(kv.getLength());
                }
            }
            return true;
        }

        private void error(String msg) {
            LOG.error((Object)("error in reader " + this.readerId + " (pread=" + this.pread + "): " + msg));
        }
    }

    private static enum Workload {
        MERGE("merge", "Merge the specified HFiles", 1, Integer.MAX_VALUE),
        RANDOM_READS("read", "Perform a random read benchmark on the given HFile", 1, 1);

        private String option;
        private String description;
        public final int minNumInputFiles;
        public final int maxNumInputFiles;

        private Workload(String option, String description, int minNumInputFiles, int maxNumInputFiles) {
            this.option = option;
            this.description = description;
            this.minNumInputFiles = minNumInputFiles;
            this.maxNumInputFiles = maxNumInputFiles;
        }

        static OptionGroup getOptionGroup() {
            OptionGroup optionGroup = new OptionGroup();
            for (Workload w : Workload.values()) {
                optionGroup.addOption(new Option(w.option, w.description));
            }
            return optionGroup;
        }

        private static String getOptionListStr() {
            StringBuilder sb = new StringBuilder();
            for (Workload w : Workload.values()) {
                if (sb.length() > 0) {
                    sb.append(", ");
                }
                sb.append("-" + w.option);
            }
            return sb.toString();
        }

        static Workload fromCmdLine(CommandLine cmdLine) {
            for (Workload w : Workload.values()) {
                if (!cmdLine.hasOption(w.option)) continue;
                return w;
            }
            LOG.error((Object)("No workload specified. Specify one of the options: " + Workload.getOptionListStr()));
            return null;
        }

        public String onlyUsedFor() {
            return ". Only used for the " + (Object)((Object)this) + " workload.";
        }
    }
}

