/*
 * Decompiled with CFR 0.152.
 */
package org.kitesdk.data.spi.filesystem;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.Flushable;
import java.io.IOException;
import java.util.UUID;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.kitesdk.data.DatasetDescriptor;
import org.kitesdk.data.DatasetIOException;
import org.kitesdk.data.DatasetWriterException;
import org.kitesdk.data.Format;
import org.kitesdk.data.Formats;
import org.kitesdk.data.spi.AbstractDatasetWriter;
import org.kitesdk.data.spi.DescriptorUtil;
import org.kitesdk.data.spi.ReaderWriterState;
import org.kitesdk.data.spi.filesystem.AvroAppender;
import org.kitesdk.data.spi.filesystem.CSVAppender;
import org.kitesdk.data.spi.filesystem.DurableParquetAppender;
import org.kitesdk.data.spi.filesystem.ParquetAppender;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class FileSystemWriter<E>
extends AbstractDatasetWriter<E> {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemWriter.class);
    protected final FileSystem fs;
    private final Path directory;
    private final DatasetDescriptor descriptor;
    private FileAppender<E> appender;
    private Path tempPath;
    private Path finalPath;
    private ReaderWriterState state;
    private int count = 0;
    @VisibleForTesting
    final Configuration conf;

    public FileSystemWriter(FileSystem fs, Path path, DatasetDescriptor descriptor) {
        Preconditions.checkNotNull((Object)fs, (Object)"File system is not defined");
        Preconditions.checkNotNull((Object)path, (Object)"Destination directory is not defined");
        Preconditions.checkNotNull((Object)descriptor, (Object)"Descriptor is not defined");
        this.fs = fs;
        this.directory = path;
        this.descriptor = descriptor;
        this.conf = new Configuration(fs.getConf());
        this.state = ReaderWriterState.NEW;
        for (String prop : descriptor.listProperties()) {
            this.conf.set(prop, descriptor.getProperty(prop));
        }
    }

    @Override
    public final void initialize() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.NEW), (String)"Unable to open a writer from state:%s", (Object[])new Object[]{this.state});
        try {
            this.fs.mkdirs(this.directory);
        }
        catch (IOException ex) {
            this.state = ReaderWriterState.ERROR;
            throw new DatasetIOException("Failed to create path " + this.directory, ex);
        }
        this.finalPath = new Path(this.directory, FileSystemWriter.uniqueFilename(this.descriptor.getFormat()));
        this.tempPath = FileSystemWriter.tempFilename(this.finalPath);
        this.appender = this.newAppender(this.tempPath);
        try {
            this.appender.open();
        }
        catch (IOException e) {
            this.state = ReaderWriterState.ERROR;
            throw new DatasetIOException("Failed to open appender " + this.appender, e);
        }
        this.count = 0;
        LOG.debug("Opened appender {} for {}", this.appender, (Object)this.finalPath);
        this.state = ReaderWriterState.OPEN;
    }

    @Override
    public final void write(E entity) {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to write to a writer in state:%s", (Object[])new Object[]{this.state});
        try {
            this.appender.append(entity);
            ++this.count;
        }
        catch (IOException e) {
            this.state = ReaderWriterState.ERROR;
            throw new DatasetIOException("Failed to append " + entity + " to " + this.appender, e);
        }
    }

    @Override
    public void flush() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to flush a writer in state:%s", (Object[])new Object[]{this.state});
        try {
            this.appender.flush();
        }
        catch (IOException e) {
            this.state = ReaderWriterState.ERROR;
            throw new DatasetWriterException("Failed to flush appender " + this.appender);
        }
    }

    @Override
    public void sync() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to sync a writer in state:%s", (Object[])new Object[]{this.state});
        try {
            this.appender.sync();
        }
        catch (IOException e) {
            this.state = ReaderWriterState.ERROR;
            throw new DatasetIOException("Failed to sync appender " + this.appender, e);
        }
    }

    @Override
    public final void close() {
        if (this.state.equals((Object)ReaderWriterState.OPEN)) {
            try {
                this.appender.close();
            }
            catch (IOException e) {
                this.state = ReaderWriterState.ERROR;
                throw new DatasetIOException("Failed to close appender " + this.appender, e);
            }
            if (this.count > 0) {
                try {
                    if (!this.fs.rename(this.tempPath, this.finalPath)) {
                        this.state = ReaderWriterState.ERROR;
                        throw new DatasetWriterException("Failed to move " + this.tempPath + " to " + this.finalPath);
                    }
                }
                catch (IOException e) {
                    this.state = ReaderWriterState.ERROR;
                    throw new DatasetIOException("Failed to commit " + this.finalPath, e);
                }
                LOG.debug("Committed {} for appender {} ({} entities)", new Object[]{this.finalPath, this.appender, this.count});
            } else {
                try {
                    if (!this.fs.delete(this.tempPath, true)) {
                        this.state = ReaderWriterState.ERROR;
                        throw new DatasetWriterException("Failed to delete " + this.tempPath);
                    }
                }
                catch (IOException e) {
                    this.state = ReaderWriterState.ERROR;
                    throw new DatasetIOException("Failed to remove temporary file " + this.tempPath, e);
                }
                LOG.debug("Discarded {} ({} entities)", (Object)this.tempPath, (Object)this.count);
            }
            try {
                this.appender.cleanup();
            }
            catch (IOException e) {
                throw new DatasetIOException("Failed to clean up " + this.appender, e);
            }
            this.state = ReaderWriterState.CLOSED;
        } else if (this.state.equals((Object)ReaderWriterState.ERROR)) {
            this.state = ReaderWriterState.CLOSED;
        }
    }

    @Override
    public final boolean isOpen() {
        return this.state.equals((Object)ReaderWriterState.OPEN);
    }

    private static String uniqueFilename(Format format) {
        return UUID.randomUUID() + "." + format.getExtension();
    }

    private static Path tempFilename(Path location) {
        return new Path(location.getParent(), "." + location.getName() + ".tmp");
    }

    @VisibleForTesting
    <E> FileAppender<E> newAppender(Path temp) {
        Format format = this.descriptor.getFormat();
        if (Formats.PARQUET.equals(format)) {
            if (DescriptorUtil.isDisabled("kite.parquet.non-durable-writes", this.descriptor)) {
                return new DurableParquetAppender(this.fs, temp, this.descriptor.getSchema(), this.conf, this.descriptor.getCompressionType());
            }
            return new ParquetAppender(this.fs, temp, this.descriptor.getSchema(), this.conf, this.descriptor.getCompressionType());
        }
        if (Formats.AVRO.equals(format)) {
            return new AvroAppender(this.fs, temp, this.descriptor.getSchema(), this.descriptor.getCompressionType());
        }
        if (Formats.CSV.equals(format) && DescriptorUtil.isEnabled("kite.allow.csv", this.descriptor)) {
            return new CSVAppender(this.fs, temp, this.descriptor);
        }
        this.state = ReaderWriterState.ERROR;
        throw new DatasetWriterException("Unknown format " + this.descriptor);
    }

    public RecordWriter<E, Void> asRecordWriter() {
        return new KiteRecordWriter();
    }

    private class KiteRecordWriter
    extends RecordWriter<E, Void> {
        private KiteRecordWriter() {
        }

        public void write(E e, Void aVoid) throws IOException, InterruptedException {
            FileSystemWriter.this.write(e);
        }

        public void close(TaskAttemptContext context) throws IOException, InterruptedException {
            FileSystemWriter.this.close();
        }
    }

    static interface FileAppender<E>
    extends Flushable,
    Closeable {
        public void open() throws IOException;

        public void append(E var1) throws IOException;

        public void sync() throws IOException;

        public void cleanup() throws IOException;
    }
}

