/*
 * Decompiled with CFR 0.152.
 */
package org.kitesdk.data.spi.filesystem;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.util.Map;
import org.apache.hadoop.fs.Path;
import org.kitesdk.data.DatasetDescriptor;
import org.kitesdk.data.DatasetWriter;
import org.kitesdk.data.PartitionStrategy;
import org.kitesdk.data.spi.AbstractDatasetWriter;
import org.kitesdk.data.spi.EntityAccessor;
import org.kitesdk.data.spi.PartitionListener;
import org.kitesdk.data.spi.ReaderWriterState;
import org.kitesdk.data.spi.StorageKey;
import org.kitesdk.data.spi.filesystem.FileSystemDataset;
import org.kitesdk.data.spi.filesystem.FileSystemView;
import org.kitesdk.data.spi.filesystem.FileSystemWriter;
import org.kitesdk.data.spi.filesystem.PathConversion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class PartitionedDatasetWriter<E>
extends AbstractDatasetWriter<E> {
    private static final Logger LOG = LoggerFactory.getLogger(PartitionedDatasetWriter.class);
    private static final int DEFAULT_WRITER_CACHE_SIZE = 10;
    private FileSystemView<E> view;
    private final int maxWriters;
    private final PartitionStrategy partitionStrategy;
    private LoadingCache<StorageKey, DatasetWriter<E>> cachedWriters;
    private final StorageKey reusedKey;
    private final EntityAccessor<E> accessor;
    private final Map<String, Object> provided;
    private ReaderWriterState state;

    public PartitionedDatasetWriter(FileSystemView<E> view) {
        DatasetDescriptor descriptor = view.getDataset().getDescriptor();
        Preconditions.checkArgument((boolean)descriptor.isPartitioned(), (Object)("Dataset " + view.getDataset() + " is not partitioned"));
        this.view = view;
        this.partitionStrategy = descriptor.getPartitionStrategy();
        int maxWriters = 10;
        if (descriptor.hasProperty("kite.writer.cache-size")) {
            try {
                maxWriters = Integer.parseInt(descriptor.getProperty("kite.writer.cache-size"));
            }
            catch (NumberFormatException e) {
                LOG.warn("Not an integer: kite.writer.cache-size=" + descriptor.getProperty("kite.writer.cache-size"));
            }
        } else if (this.partitionStrategy.getCardinality() != -1) {
            maxWriters = Math.min(maxWriters, this.partitionStrategy.getCardinality());
        }
        this.maxWriters = maxWriters;
        this.state = ReaderWriterState.NEW;
        this.reusedKey = new StorageKey(this.partitionStrategy);
        this.accessor = view.getAccessor();
        this.provided = view.getProvidedValues();
    }

    @Override
    public void initialize() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.NEW), (String)"Unable to open a writer from state:%s", (Object[])new Object[]{this.state});
        LOG.debug("Opening partitioned dataset writer w/strategy:{}", (Object)this.partitionStrategy);
        this.cachedWriters = CacheBuilder.newBuilder().maximumSize((long)this.maxWriters).removalListener(new DatasetWriterCloser()).build(new DatasetWriterCacheLoader<E>(this.view));
        this.state = ReaderWriterState.OPEN;
    }

    @Override
    public void write(E entity) {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to write to a writer in state:%s", (Object[])new Object[]{this.state});
        this.accessor.keyFor(entity, this.provided, this.reusedKey);
        DatasetWriter writer = (DatasetWriter)this.cachedWriters.getIfPresent((Object)this.reusedKey);
        if (writer == null) {
            Preconditions.checkArgument((boolean)this.view.includes(entity), (String)"View %s does not include entity %s", (Object[])new Object[]{this.view, entity});
            StorageKey key = StorageKey.copy(this.reusedKey);
            try {
                writer = (DatasetWriter)this.cachedWriters.getUnchecked((Object)key);
            }
            catch (UncheckedExecutionException ex) {
                throw new IllegalArgumentException("Problem creating view for entity: " + entity, ex.getCause());
            }
        }
        writer.write(entity);
    }

    @Override
    public void flush() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to flush a writer in state:%s", (Object[])new Object[]{this.state});
        LOG.debug("Flushing all cached writers for view:{}", this.view);
        for (DatasetWriter writer : this.cachedWriters.asMap().values()) {
            LOG.debug("Flushing partition writer:{}", (Object)writer);
            writer.flush();
        }
    }

    @Override
    public void sync() {
        Preconditions.checkState((boolean)this.state.equals((Object)ReaderWriterState.OPEN), (String)"Attempt to sync a writer in state:%s", (Object[])new Object[]{this.state});
        LOG.debug("Syncing all cached writers for view:{}", this.view);
        for (DatasetWriter writer : this.cachedWriters.asMap().values()) {
            LOG.debug("Syncing partition writer:{}", (Object)writer);
            writer.sync();
        }
    }

    @Override
    public void close() {
        if (this.state.equals((Object)ReaderWriterState.OPEN)) {
            LOG.debug("Closing all cached writers for view:{}", this.view);
            for (DatasetWriter writer : this.cachedWriters.asMap().values()) {
                LOG.debug("Closing partition writer:{}", (Object)writer);
                writer.close();
            }
            this.state = ReaderWriterState.CLOSED;
        }
    }

    @Override
    public boolean isOpen() {
        return this.state.equals((Object)ReaderWriterState.OPEN);
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add("partitionStrategy", (Object)this.partitionStrategy).add("maxWriters", this.maxWriters).add("view", this.view).add("cachedWriters", this.cachedWriters).toString();
    }

    private static class DatasetWriterCloser<E>
    implements RemovalListener<StorageKey, DatasetWriter<E>> {
        private DatasetWriterCloser() {
        }

        public void onRemoval(RemovalNotification<StorageKey, DatasetWriter<E>> notification) {
            DatasetWriter writer = (DatasetWriter)notification.getValue();
            LOG.debug("Closing writer:{} for partition:{}", (Object)writer, notification.getKey());
            writer.close();
        }
    }

    private static class DatasetWriterCacheLoader<E>
    extends CacheLoader<StorageKey, DatasetWriter<E>> {
        private final FileSystemView<E> view;
        private final PathConversion convert;

        public DatasetWriterCacheLoader(FileSystemView<E> view) {
            this.view = view;
            this.convert = new PathConversion(view.getDataset().getDescriptor().getSchema());
        }

        public DatasetWriter<E> load(StorageKey key) throws Exception {
            Preconditions.checkState((boolean)(this.view.getDataset() instanceof FileSystemDataset), (Object)("FileSystemWriters cannot create writer for " + this.view.getDataset()));
            FileSystemDataset dataset = (FileSystemDataset)this.view.getDataset();
            Path partition = this.convert.fromKey(key);
            FileSystemWriter writer = new FileSystemWriter(dataset.getFileSystem(), new Path(dataset.getDirectory(), partition), dataset.getDescriptor());
            PartitionListener listener = dataset.getPartitionListener();
            if (listener != null) {
                listener.partitionAdded(dataset.getNamespace(), dataset.getName(), partition.toString());
            }
            writer.initialize();
            return writer;
        }
    }
}

