/*
 * Decompiled with CFR 0.152.
 */
package org.kiwiproject.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.errorprone.annotations.Immutable;
import java.beans.ConstructorProperties;
import java.io.File;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import lombok.Generated;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.kiwiproject.base.KiwiPreconditions;
import org.kiwiproject.base.KiwiStrings;
import org.kiwiproject.collect.KiwiEvictingQueues;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class TimeBasedDirectoryCleaner
implements Runnable {
    @Generated
    private static final Logger LOG = LoggerFactory.getLogger(TimeBasedDirectoryCleaner.class);
    private static final int MAX_RECENT_DELETE_ERRORS = 500;
    private static final boolean SUPPRESS_LEADING_ZERO_ELEMENTS = true;
    private static final boolean SUPPRESS_TRAILING_ZERO_ELEMENTS = true;
    private static final File[] EMPTY_FILE_ARRAY = new File[0];
    private final Queue<DeleteError> recentDeleteErrors = KiwiEvictingQueues.synchronizedEvictingQueue(500);
    private final AtomicLong deleteCount = new AtomicLong();
    private final AtomicInteger deleteErrorCount = new AtomicInteger();
    private final File directory;
    private final long retentionThresholdInMillis;
    @VisibleForTesting
    final Level deleteErrorLogLevel;
    private final String retentionThresholdDescription;

    public TimeBasedDirectoryCleaner(String directoryPath, Duration retentionThreshold, String deleteErrorLogLevel) {
        KiwiPreconditions.checkArgumentNotNull(directoryPath, "directoryPath is required");
        KiwiPreconditions.checkArgumentNotNull(retentionThreshold, "retentionThreshold is required");
        this.directory = new File(directoryPath);
        this.retentionThresholdInMillis = retentionThreshold.toMillis();
        Preconditions.checkArgument((this.retentionThresholdInMillis > 0L ? 1 : 0) != 0, (Object)"retentionThreshold cannot be negative");
        this.retentionThresholdDescription = TimeBasedDirectoryCleaner.durationDescription(this.retentionThresholdInMillis);
        this.deleteErrorLogLevel = Objects.isNull(deleteErrorLogLevel) ? Level.WARN : TimeBasedDirectoryCleaner.resolveLevelOrDefaultToWarn(deleteErrorLogLevel);
    }

    private static Level resolveLevelOrDefaultToWarn(String deleteErrorLogLevel) {
        try {
            return Level.valueOf((String)deleteErrorLogLevel);
        }
        catch (IllegalArgumentException e) {
            LOG.warn("Level {} is not a valid SLF4J level, defaulting to WARN. Valid levels: {}", (Object)deleteErrorLogLevel, (Object)Arrays.toString(Level.values()));
            LOG.trace("Actual exception:", (Throwable)e);
            return Level.WARN;
        }
    }

    private static String durationDescription(long milliseconds) {
        if (milliseconds < 1000L) {
            return milliseconds + " milliseconds";
        }
        return DurationFormatUtils.formatDurationWords((long)milliseconds, (boolean)true, (boolean)true);
    }

    public String getDirectoryPath() {
        return this.directory.getAbsolutePath();
    }

    public Duration getRetentionThreshold() {
        return Duration.ofMillis(this.retentionThresholdInMillis);
    }

    public long getDeleteCount() {
        return this.deleteCount.get();
    }

    public int getDeleteErrorCount() {
        return this.deleteErrorCount.get();
    }

    public int getNumberOfRecentDeleteErrors() {
        return this.recentDeleteErrors.size();
    }

    public static int capacityOfRecentDeleteErrors() {
        return 500;
    }

    public void clearRecentDeleteErrors() {
        this.recentDeleteErrors.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<DeleteError> getRecentDeleteErrors() {
        Queue<DeleteError> queue = this.recentDeleteErrors;
        synchronized (queue) {
            return new ArrayList<DeleteError>(this.recentDeleteErrors);
        }
    }

    @Override
    public void run() {
        try {
            this.cleanDirectory();
        }
        catch (Exception e) {
            this.deleteErrorCount.incrementAndGet();
            this.recentDeleteErrors.add(DeleteError.of(e));
            LOG.error("Error cleaning directory [{}] with retention threshold {}", new Object[]{this.directory.getAbsolutePath(), this.retentionThresholdDescription, e});
        }
    }

    @VisibleForTesting
    void cleanDirectory() {
        LOG.debug("Cleaning directory [{}] with retention threshold {}", (Object)this.directory.getAbsolutePath(), (Object)this.retentionThresholdDescription);
        long now = System.currentTimeMillis();
        LOG.trace("Reference current time for directory cleanup: {}", (Object)now);
        File[] filesToClean = Optional.ofNullable(this.directory.listFiles(file -> this.olderThanRetentionThreshold(file, now))).orElse(EMPTY_FILE_ARRAY);
        LOG.debug("Found {} files to clean (that are older than retention threshold)", (Object)filesToClean.length);
        List<FileDeleteResult> attemptedDeletes = Arrays.stream(filesToClean).map(TimeBasedDirectoryCleaner::tryDeleteIfExists).filter(result -> result.deleteWasAttempted).toList();
        int numExpectedDeletes = attemptedDeletes.size();
        List<FileDeleteResult> failedDeletes = attemptedDeletes.stream().filter(FileDeleteResult::deleteAttemptedAndFailed).peek(this::logUnableToDelete).toList();
        this.updateFileDeletionMetadata(numExpectedDeletes, failedDeletes);
    }

    @VisibleForTesting
    static FileDeleteResult tryDeleteIfExists(File file) {
        String absolutePath = file.getAbsolutePath();
        LOG.trace("Attempting to delete file {}", (Object)absolutePath);
        if (file.exists()) {
            LOG.trace("File {} exists", (Object)absolutePath);
            boolean wasDeleted = FileUtils.deleteQuietly((File)file);
            LOG.trace("Attempt to delete existing file {} was successful? {}", (Object)absolutePath, (Object)wasDeleted);
            return FileDeleteResult.attempted(absolutePath, wasDeleted);
        }
        LOG.trace("Skipped delete attempt as file did not exist: {}", (Object)absolutePath);
        return FileDeleteResult.skipped(absolutePath);
    }

    @VisibleForTesting
    void logUnableToDelete(FileDeleteResult deleteResult) {
        this.logDeleteError("Unable to delete " + deleteResult.absolutePath);
    }

    @VisibleForTesting
    void updateFileDeletionMetadata(int expectedDeleteCount, List<FileDeleteResult> failedDeleteResults) {
        if (!failedDeleteResults.isEmpty()) {
            int newDeleteErrorCount = this.deleteErrorCount.addAndGet(failedDeleteResults.size());
            this.logDeleteError(KiwiStrings.f("There are now {} total file delete errors", newDeleteErrorCount));
            failedDeleteResults.stream().map(DeleteError::of).forEach(this.recentDeleteErrors::add);
        }
        int actualDeleteCount = expectedDeleteCount - failedDeleteResults.size();
        long newCumulativeDeleteCount = this.deleteCount.addAndGet(actualDeleteCount);
        LOG.debug("Deleted {} files; new cumulative delete count: {}", (Object)actualDeleteCount, (Object)newCumulativeDeleteCount);
    }

    private void logDeleteError(String message) {
        switch (this.deleteErrorLogLevel) {
            case TRACE: {
                LOG.trace(message);
                break;
            }
            case DEBUG: {
                LOG.debug(message);
                break;
            }
            case INFO: {
                LOG.info(message);
                break;
            }
            case WARN: 
            case ERROR: {
                LOG.error(message);
                break;
            }
            default: {
                LOG.warn(message);
            }
        }
    }

    private boolean olderThanRetentionThreshold(File file, long now) {
        long ageInMillis = now - file.lastModified();
        boolean shouldDelete = ageInMillis > this.retentionThresholdInMillis;
        LOG.trace("Age of file {}: {} ms (retention threshold: {} ms); should delete? {}", new Object[]{file.getAbsolutePath(), ageInMillis, this.retentionThresholdInMillis, shouldDelete});
        return shouldDelete;
    }

    @Generated
    public static TimeBasedDirectoryCleanerBuilder builder() {
        return new TimeBasedDirectoryCleanerBuilder();
    }

    @Generated
    public String getRetentionThresholdDescription() {
        return this.retentionThresholdDescription;
    }

    @Immutable
    public static class DeleteError {
        private final long timestamp;
        private final String fileName;
        private final String exceptionType;
        private final String exceptionMessage;

        public static DeleteError of(String fileName) {
            KiwiPreconditions.checkArgumentNotNull(fileName, "fileName is required");
            return new DeleteError(System.currentTimeMillis(), fileName, null, null);
        }

        public static DeleteError of(FileDeleteResult deleteResult) {
            KiwiPreconditions.checkArgumentNotNull(deleteResult, "deleteResult is required");
            Preconditions.checkState((boolean)deleteResult.deleteAttemptedAndFailed(), (Object)"must be an attempted delete that failed");
            return new DeleteError(System.currentTimeMillis(), deleteResult.absolutePath, null, null);
        }

        public static DeleteError of(Exception ex) {
            KiwiPreconditions.checkArgumentNotNull(ex, "exception is required");
            return new DeleteError(System.currentTimeMillis(), null, ex.getClass().getName(), ex.getMessage());
        }

        public boolean isExceptionError() {
            return Objects.nonNull(this.exceptionType);
        }

        public boolean isFileDeleteError() {
            return Objects.nonNull(this.fileName);
        }

        @Generated
        public long getTimestamp() {
            return this.timestamp;
        }

        @Generated
        public String getFileName() {
            return this.fileName;
        }

        @Generated
        public String getExceptionType() {
            return this.exceptionType;
        }

        @Generated
        public String getExceptionMessage() {
            return this.exceptionMessage;
        }

        @ConstructorProperties(value={"timestamp", "fileName", "exceptionType", "exceptionMessage"})
        @Generated
        private DeleteError(long timestamp, String fileName, String exceptionType, String exceptionMessage) {
            this.timestamp = timestamp;
            this.fileName = fileName;
            this.exceptionType = exceptionType;
            this.exceptionMessage = exceptionMessage;
        }

        @Generated
        public DeleteError withTimestamp(long timestamp) {
            return this.timestamp == timestamp ? this : new DeleteError(timestamp, this.fileName, this.exceptionType, this.exceptionMessage);
        }
    }

    public static class FileDeleteResult {
        private final String absolutePath;
        private final boolean deleteWasAttempted;
        private final boolean deleteWasSuccessful;

        static FileDeleteResult attempted(String absolutePath, boolean deleteWasSuccessful) {
            return new FileDeleteResult(Objects.requireNonNull(absolutePath), true, deleteWasSuccessful);
        }

        static FileDeleteResult skipped(String absolutePath) {
            return new FileDeleteResult(Objects.requireNonNull(absolutePath), false, false);
        }

        boolean deleteAttemptedAndFailed() {
            return this.deleteWasAttempted && !this.deleteWasSuccessful;
        }

        @ConstructorProperties(value={"absolutePath", "deleteWasAttempted", "deleteWasSuccessful"})
        @Generated
        public FileDeleteResult(String absolutePath, boolean deleteWasAttempted, boolean deleteWasSuccessful) {
            this.absolutePath = absolutePath;
            this.deleteWasAttempted = deleteWasAttempted;
            this.deleteWasSuccessful = deleteWasSuccessful;
        }

        @Generated
        public String getAbsolutePath() {
            return this.absolutePath;
        }

        @Generated
        public boolean isDeleteWasAttempted() {
            return this.deleteWasAttempted;
        }

        @Generated
        public boolean isDeleteWasSuccessful() {
            return this.deleteWasSuccessful;
        }
    }

    @Generated
    public static class TimeBasedDirectoryCleanerBuilder {
        @Generated
        private String directoryPath;
        @Generated
        private Duration retentionThreshold;
        @Generated
        private String deleteErrorLogLevel;

        @Generated
        TimeBasedDirectoryCleanerBuilder() {
        }

        @Generated
        public TimeBasedDirectoryCleanerBuilder directoryPath(String directoryPath) {
            this.directoryPath = directoryPath;
            return this;
        }

        @Generated
        public TimeBasedDirectoryCleanerBuilder retentionThreshold(Duration retentionThreshold) {
            this.retentionThreshold = retentionThreshold;
            return this;
        }

        @Generated
        public TimeBasedDirectoryCleanerBuilder deleteErrorLogLevel(String deleteErrorLogLevel) {
            this.deleteErrorLogLevel = deleteErrorLogLevel;
            return this;
        }

        @Generated
        public TimeBasedDirectoryCleaner build() {
            return new TimeBasedDirectoryCleaner(this.directoryPath, this.retentionThreshold, this.deleteErrorLogLevel);
        }

        @Generated
        public String toString() {
            return "TimeBasedDirectoryCleaner.TimeBasedDirectoryCleanerBuilder(directoryPath=" + this.directoryPath + ", retentionThreshold=" + this.retentionThreshold + ", deleteErrorLogLevel=" + this.deleteErrorLogLevel + ")";
        }
    }
}

