/*
 * Decompiled with CFR 0.152.
 */
package org.komamitsu.failuredetector;

import java.util.LinkedList;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

public class PhiAccrualFailureDetector {
    private final double threshold;
    private final double minStdDeviationMillis;
    private final long acceptableHeartbeatPauseMillis;
    private final HeartbeatHistory heartbeatHistory;
    private final AtomicReference<Long> lastTimestampMillis = new AtomicReference();

    private PhiAccrualFailureDetector(double threshold, int maxSampleSize, double minStdDeviationMillis, long acceptableHeartbeatPauseMillis, long firstHeartbeatEstimateMillis) {
        if (threshold <= 0.0) {
            throw new IllegalArgumentException("Threshold must be > 0: " + threshold);
        }
        if (maxSampleSize <= 0) {
            throw new IllegalArgumentException("Sample size must be > 0: " + maxSampleSize);
        }
        if (minStdDeviationMillis <= 0.0) {
            throw new IllegalArgumentException("Minimum standard deviation must be > 0: " + minStdDeviationMillis);
        }
        if (acceptableHeartbeatPauseMillis < 0L) {
            throw new IllegalArgumentException("Acceptable heartbeat pause millis must be >= 0: " + acceptableHeartbeatPauseMillis);
        }
        if (firstHeartbeatEstimateMillis <= 0L) {
            throw new IllegalArgumentException("First heartbeat value must be > 0: " + firstHeartbeatEstimateMillis);
        }
        this.threshold = threshold;
        this.minStdDeviationMillis = minStdDeviationMillis;
        this.acceptableHeartbeatPauseMillis = acceptableHeartbeatPauseMillis;
        long stdDeviationMillis = firstHeartbeatEstimateMillis / 4L;
        this.heartbeatHistory = new HeartbeatHistory(maxSampleSize);
        this.heartbeatHistory.add(firstHeartbeatEstimateMillis - stdDeviationMillis).add(firstHeartbeatEstimateMillis + stdDeviationMillis);
    }

    private double ensureValidStdDeviation(double stdDeviationMillis) {
        return Math.max(stdDeviationMillis, this.minStdDeviationMillis);
    }

    public synchronized double phi(long timestampMillis) {
        Long lastTimestampMillis = this.lastTimestampMillis.get();
        if (lastTimestampMillis == null) {
            return 0.0;
        }
        long timeDiffMillis = timestampMillis - lastTimestampMillis;
        double meanMillis = this.heartbeatHistory.mean() + (double)this.acceptableHeartbeatPauseMillis;
        double stdDeviationMillis = this.ensureValidStdDeviation(this.heartbeatHistory.stdDeviation());
        double y = ((double)timeDiffMillis - meanMillis) / stdDeviationMillis;
        double e = Math.exp(-y * (1.5976 + 0.070566 * y * y));
        if ((double)timeDiffMillis > meanMillis) {
            return -Math.log10(e / (1.0 + e));
        }
        return -Math.log10(1.0 - 1.0 / (1.0 + e));
    }

    public synchronized double phi() {
        return this.phi(System.currentTimeMillis());
    }

    public boolean isAvailable(long timestampMillis) {
        return this.phi(timestampMillis) < this.threshold;
    }

    public boolean isAvailable() {
        return this.phi(System.currentTimeMillis()) < this.threshold;
    }

    public synchronized void heartbeat(long timestampMillis) {
        Long lastTimestampMillis = this.lastTimestampMillis.getAndSet(timestampMillis);
        if (lastTimestampMillis != null) {
            long interval = timestampMillis - lastTimestampMillis;
            if (this.isAvailable(timestampMillis)) {
                this.heartbeatHistory.add(interval);
            }
        }
    }

    public void heartbeat() {
        this.heartbeat(System.currentTimeMillis());
    }

    private static class HeartbeatHistory {
        private final int maxSampleSize;
        private final LinkedList<Long> intervals = new LinkedList();
        private final AtomicLong intervalSum = new AtomicLong();
        private final AtomicLong squaredIntervalSum = new AtomicLong();

        public HeartbeatHistory(int maxSampleSize) {
            if (maxSampleSize < 1) {
                throw new IllegalArgumentException("maxSampleSize must be >= 1, got " + maxSampleSize);
            }
            this.maxSampleSize = maxSampleSize;
        }

        public double mean() {
            return (double)this.intervalSum.get() / (double)this.intervals.size();
        }

        public double variance() {
            return (double)this.squaredIntervalSum.get() / (double)this.intervals.size() - this.mean() * this.mean();
        }

        public double stdDeviation() {
            return Math.sqrt(this.variance());
        }

        public HeartbeatHistory add(long interval) {
            if (this.intervals.size() >= this.maxSampleSize) {
                Long dropped = this.intervals.pollFirst();
                this.intervalSum.addAndGet(-dropped.longValue());
                this.squaredIntervalSum.addAndGet(-this.pow2(dropped));
            }
            this.intervals.add(interval);
            this.intervalSum.addAndGet(interval);
            this.squaredIntervalSum.addAndGet(this.pow2(interval));
            return this;
        }

        private long pow2(long x) {
            return x * x;
        }
    }

    public static class Builder {
        private double threshold = 16.0;
        private int maxSampleSize = 200;
        private double minStdDeviationMillis = 500.0;
        private long acceptableHeartbeatPauseMillis = 0L;
        private long firstHeartbeatEstimateMillis = 500L;

        public Builder setThreshold(double threshold) {
            this.threshold = threshold;
            return this;
        }

        public Builder setMaxSampleSize(int maxSampleSize) {
            this.maxSampleSize = maxSampleSize;
            return this;
        }

        public Builder setMinStdDeviationMillis(double minStdDeviationMillis) {
            this.minStdDeviationMillis = minStdDeviationMillis;
            return this;
        }

        public Builder setAcceptableHeartbeatPauseMillis(long acceptableHeartbeatPauseMillis) {
            this.acceptableHeartbeatPauseMillis = acceptableHeartbeatPauseMillis;
            return this;
        }

        public Builder setFirstHeartbeatEstimateMillis(long firstHeartbeatEstimateMillis) {
            this.firstHeartbeatEstimateMillis = firstHeartbeatEstimateMillis;
            return this;
        }

        public PhiAccrualFailureDetector build() {
            return new PhiAccrualFailureDetector(this.threshold, this.maxSampleSize, this.minStdDeviationMillis, this.acceptableHeartbeatPauseMillis, this.firstHeartbeatEstimateMillis);
        }
    }
}

