package org.krosai.core.chat.client

import kotlinx.coroutines.flow.Flow
import org.krosai.core.chat.enhancer.Enhancer
import org.krosai.core.chat.function.Func1
import org.krosai.core.chat.function.FunctionCall
import org.krosai.core.chat.function.FunctionCallBuilder
import org.krosai.core.chat.model.ChatResponse
import kotlin.reflect.KFunction

/**
 * The chat client that interacts with an AI model to send and receive chat messages.
 *
 * @author KAMOsama
 */
interface ChatClient {

    /**
     * Executes a chat request and returns the chat response.
     *
     * @param requestScopeSpec the specification for the chat request
     * @return the chat response generated by the chat client
     */
    suspend fun call(requestScopeSpec: ChatClientRequestDefinition): ChatResponse

    /**
     * Executes a chat request and returns the chat response as a flow of `ChatResponse` objects.
     *
     * @param requestScopeSpec the specification for the chat request
     * @return the flow of `ChatResponse` objects
     */
    suspend fun stream(requestScopeSpec: ChatClientRequestDefinition): Flow<ChatResponse>

}

/**
 * A function that defines the chat request.
 */
typealias ChatClientRequestDefinition = (ChatClientRequestScope.() -> Unit)?

/**
 * This interface defines the operations related to request parameters when the chat client sends a request.
 */
interface ChatClientRequestScope {

    fun userText(block: Map<String, Any>.() -> String?)

    fun systemText(block: Map<String, Any>.() -> String?)

    fun user(userScope: PromptUserScope.() -> Unit)

    fun system(systemScope: PromptSystemScope.() -> Unit)

    fun enhancers(enhancersScope: EnhancersScope.() -> Unit)

    fun functions(functionCallScope: FunctionCallScope.() -> Unit)

}

/**
 * This interface defines the operations related to prompting the user for input.
 */
interface PromptUserScope {

    fun text(block: Map<String, Any>.() -> String?)

    infix fun String.to(value: Any)

}

/**
 *
 */
interface PromptSystemScope {

    fun text(block: Map<String, Any>.() -> String?)

    infix fun String.to(value: Any)

}

/**
 * This interface defines the operations related to enhancing the chat request.
 */
interface EnhancersScope {

    infix fun String.to(value: Any)

    operator fun Enhancer.unaryPlus()

    operator fun List<Enhancer>.unaryPlus()

}

/**
 * This interface defines the operations related to calling functions in the chat request.
 */
interface FunctionCallScope {

    operator fun FunctionCall.unaryPlus()

    operator fun List<FunctionCall>.unaryPlus()

    operator fun String.unaryPlus()

    operator fun KFunction<*>.unaryPlus()

    fun function(vararg functions: KFunction<*>) =
        functions.map { it.name }.toTypedArray().let { function(*it) }

    fun function(vararg functionNames: String)

}

inline fun <reified I : Any, O : Any> FunctionCallScope.function(
    name: String,
    description: String,
    noinline call: Func1<I, O>,
) {
    with(FunctionCallBuilder(name, description)) {
        withCall<I>(call)
        build()
    }.also { +it }
}
