package org.krproject.ocean.skeletons.fish.batch.endpoint;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.krproject.ocean.skeletons.fish.batch.api.FishBatchRequest;
import org.krproject.ocean.skeletons.fish.batch.api.FishBatchResponse;
import org.krproject.ocean.skeletons.fish.batch.exception.FishSkeletonBatchBadRequestException;
import org.krproject.ocean.skeletons.fish.batch.exception.FishSkeletonBatchException;
import org.krproject.ocean.vitamins.batch.exception.BatchException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.ClassUtils;

import lombok.extern.slf4j.Slf4j;

/**
 * fish批量服务端点抽象类.
 * 
 * @param <REQ> 请求泛型
 * @param <RESP> 响应泛型
 * @author Tiger
 *
 */
@Slf4j
public abstract class AbstractFishBatchActivator<REQ extends FishBatchRequest<RESP>, RESP extends FishBatchResponse> {

	@Autowired
	private Validator validator;

	@Autowired(required = false)
	private List<AbstractFishBatchHandler<?, ?>> handlers;

	private static boolean initialized = false;

	private static final Map<String, AbstractFishBatchHandler<?, ?>> HANDLER_MAP = new ConcurrentHashMap<String, AbstractFishBatchHandler<?, ?>>();
	
	@PostConstruct
	public void init() {
		if (!initialized) {
			if (this.handlers != null) {
				for (AbstractFishBatchHandler<?, ?> handler : this.handlers) {
					AbstractFishBatchHandler<?, ?> handlerFound = HANDLER_MAP.get(handler.getRequestClass().getName());
					if (handlerFound != null) {
						log.error("handler {} and {} got same request name：{}", 
								handlerFound, handler, handler.getRequestClass().getName());
						throw new RuntimeException("duplicate handler for request " + handler.getRequestClass().getName());
					}
					HANDLER_MAP.put(handler.getRequestClass().getName(), handler);
					log.debug("Loaded Handler:{} for Request:{}", ClassUtils.getUserClass(handler).getName(), handler.getRequestClass().getName());
				}
			}
			initialized = true;
		}
	}

	/**
	 * 根据异常获取响应.
	 * 
	 * @param request 请求
	 * @param exception 异常
	 * @return 响应
	 */
	public abstract RESP responseWithException(REQ request, Exception exception);

	
	/**
	 * 日志表插入.
	 * 
	 * @param request 请求
	 * 
	 * @return 日志对象
	 */
	public abstract Object insertJournal(REQ request);
	
	
	/**
	 * 日志表更新.
	 * 
	 * @param journal 日志
	 * @param handler 处理器
	 * @param response 响应
	 */
	public abstract void updateJournal(Object journal, AbstractFishBatchHandler<REQ, RESP> handler, RESP response);

	
	@SuppressWarnings("unchecked")
	protected RESP doActivate(REQ request) throws BatchException {
		
		// 请求不能为空
		if (request == null) {
			log.error("request cant't be null");
			throw new FishSkeletonBatchBadRequestException("request cant't be null");
		}
		
		// 输入请求的合法性校验.
		Set<ConstraintViolation<REQ>> constraintViolations = this.validator.validate(request);
		for (ConstraintViolation<REQ> cv: constraintViolations) {
			log.error("错误描述:{}, {}", cv.getPropertyPath(), cv.getMessage());
			throw new FishSkeletonBatchBadRequestException(cv.getMessage());
		}
		
		// 记录Journal日志
		Object journal = insertJournal(request);
		
		RESP response = null;
		AbstractFishBatchHandler<REQ, RESP> handler = null;
		try {
			// 根据请求类获取对应的handler
			String requestClassName = ClassUtils.getUserClass(request).getName();
			handler = (AbstractFishBatchHandler<REQ, RESP>) HANDLER_MAP.get(requestClassName);
			if (handler == null) {
				log.warn("Fail to get handler for request:{}", requestClassName);
				throw new FishSkeletonBatchException("No handler for " + requestClassName);
			}
			log.debug("Found Handler:{} for request:{}", handler.getClass(), request.getClass());

			// 调用处理器方法处理
			response = handler.handle(request);			
		} catch (Exception exception) {
			response = responseWithException(request, exception);
		}

		// 更新journal日志
		updateJournal(journal, handler, response);
		
		return response;
	}
}
