package org.krproject.ocean.skeletons.octopus.batch.systems;

import java.util.List;
import java.util.Set;

import org.krproject.ocean.vitamins.batch.service.model.BatchJobExecution;
import org.krproject.ocean.vitamins.batch.service.model.BatchJobExecutions;
import org.krproject.ocean.vitamins.batch.service.model.BatchJobInfo;
import org.krproject.ocean.vitamins.batch.service.model.BatchJobInstance;
import org.krproject.ocean.vitamins.batch.service.model.BatchJobInstances;
import org.krproject.ocean.vitamins.batch.service.model.BatchStepExecution;

/**
 * 远程批量系统客户端抽象.
 * <p>
 * 远程独立部署批量系统使用
 * @author zhongyang
 *
 */
public abstract class AbstractBatchClient {
	
	/**
	 * 获取批量标识码.
	 * @return String 批量标识码.
	 */
	public abstract String getSystemId();
	
	/**
	 * 获取所有批量作业.
	 * @return 批量作业集合.
	 */
	public abstract List<BatchJobInfo> getJobInfos();
	
	/**
	 * 根据批量名获取所有批量实例.
	 * @param jobName 批量名.
	 * @param start 起始值.
	 * @param count 数据条数.
	 * @return 批量实例集合.
	 */
	public abstract BatchJobInstances getJobInstances(String jobName, int start, int count);
	
	/**
	 * 根据执行Id获取执行信息.
	 * @param executionId 执行Id.
	 * @return Job执行信息.
	 */
	public abstract BatchJobExecution getJobExecution(Long executionId);
	
	/**
	 * 获取Step执行信息.
	 * @param jobExecutionId job执行Id.
	 * @param stepExecutionId step执行Id.
	 * @return Step执行信息.
	 */
	public abstract BatchStepExecution getStepExecution(Long jobExecutionId, Long stepExecutionId);
	
	/**
	 * 根据实例Id获取实例信息.
	 * @param instanceId 实例Id.
	 * @return 实例信息.
	 */
	public abstract BatchJobInstance getJobInstance(Long instanceId);
	
	/**
	 * 根据实例ID获取执行信息.
	 * @param instanceId 实例Id.
	 * @return Job执行信息集合.
	 */
	public abstract List<BatchJobExecution> getJobExecutions(Long instanceId);
	
	/**
	 * 根据JobName获取执行信息.
	 * @param jobName 批量名.
	 * @param start 起始值.
	 * @param count 个数.
	 * @return Job执行信息集合.
	 */
	public abstract BatchJobExecutions getJobExecutionsByJobName(String jobName, int start, int count);
	
	/**
	 * 根据批量名获取正在执行的执行信息.
	 * @param jobName 批量名.
	 * @return Job执行信息集合.
	 */
	public abstract Set<BatchJobExecution> findRunningJobExecutions(String jobName);
	
	/**
	 * 启动批量.
	 * @param operateUser 操作员.
	 * @param jobName 批量名.
	 * @param parameters 批量参数.
	 * @param checkLastJob 是否检查上次批量.
	 * @return Job执行Id.
	 */
	public abstract Long start(String operateUser, String jobName, String parameters, Boolean checkLastJob);
	
	/**
	 * 重启批量.
	 * @param operateUser 操作员.
	 * @param executionId Job执行Id.
	 * @return Job执行Id.
	 */
	public abstract Long restart(String operateUser, long executionId);
	
	/**
	 * 根据批量名重启批量.
	 * @param operateUser 操作员.
	 * @param jobName 批量名.
	 * @return Job执行Id.
	 */
	public abstract Long restartByJobName(String operateUser, String jobName);
	
	/**
	 * 停止批量.
	 * @param operateUser 操作员.
	 * @param executionId Job执行Id.
	 * @return true-成功,false-失败.
	 */
	public abstract boolean stop(String operateUser, long executionId);
	
	/**
	 * 废弃批量.
	 * @param operateUser 操作员.
	 * @param jobExecutionId Job执行Id.
	 * @return Job执行信息.
	 */
	public abstract BatchJobExecution abandon(String operateUser, long jobExecutionId);
	
	/**
	 * 启动下一个批量实例.
	 * @param operateUser 操作员.
	 * @param jobName 批量名.
	 * @param checkLastJob 是否检查上次批量.
	 * @return Job执行Id.
	 */
	public abstract Long startNextInstance(String operateUser, String jobName, Boolean checkLastJob);
	
}
