package org.krproject.ocean.skeletons.octopus.online.inbound;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.krproject.ocean.skeletons.octopus.online.api.OctopusSkeletonOnlineRequest;
import org.krproject.ocean.skeletons.octopus.online.api.OctopusSkeletonOnlineResponse;
import org.krproject.ocean.skeletons.octopus.online.context.OctopusOnlineContextHolder;
import org.krproject.ocean.skeletons.octopus.online.event.HandleCompletedEvent;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusBadRequestException;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusDuplicatedInboundException;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusNotPermittedException;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusSkeletonOnlineException;
import org.krproject.ocean.skeletons.octopus.param.InboundChannelParam;
import org.krproject.ocean.vitamins.param.service.ParamService;
import org.krproject.ocean.vitamins.service.config.ServiceProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import lombok.extern.slf4j.Slf4j;


/**
 * 上游渠道服务端点抽象类.
 * 
 * @param <REQ> 请求泛型
 * @param <RESP> 响应泛型
 * @author Tiger
 *
 */
@Slf4j
public abstract class AbstractInboundActivator<REQ extends OctopusSkeletonOnlineRequest<RESP>, RESP extends OctopusSkeletonOnlineResponse> {
	
	@Resource
	private Validator validator;

	@Resource
	private ParamService paramService;
	
	@Resource
	private ServiceProperties serviceProperties;
	
	@Resource
	private ApplicationContext applicationContext;

	@Autowired(required = false)
	private List<AbstractHandler<?, ?>> handlers;
	
	private static boolean initialized = false;
	
	private static final Map<String, AbstractHandler<?, ?>> HANDLER_MAP = new ConcurrentHashMap<String, AbstractHandler<?, ?>>();

	@PostConstruct
	public void init() {
		if (!initialized) {
			// 加载上游渠道交易处理器
			if (this.handlers != null) {
				for (AbstractHandler<?, ?> handler : this.handlers) {
					AbstractHandler<?, ?> handlerFound = HANDLER_MAP.get(handler.getRequestName());
					if (handlerFound != null) {
						log.error("handler {} and {} got same request name：{}", 
								handlerFound, handler, handler.getRequestName());
						throw new RuntimeException("duplicate handler for request " + handler.getRequestName());
					}
					HANDLER_MAP.put(handler.getRequestName(), handler);
					log.debug("Loaded Handler:{} for Request:{}", ClassUtils.getUserClass(handler).getName(), handler.getRequestName());
				}
			}
			
			// 置为已加载
			initialized = true;
		}
	}
	

	/**
	 * 根据异常获取响应.
	 * 
	 * @param request 请求
	 * @param exception 异常
	 * @return 响应
	 */
	public abstract RESP responseWithException(REQ request, Exception exception);

	
	/**
	 * 插入日志.
	 * 
	 * @param channelId 上游渠道号
	 * @param channelTxnId 上游渠道流水号
	 * @param request 请求对象
	 * 
	 * @return 日志实体
	 */
	public abstract Object insertInbound(String channelId, String channelTxnId, REQ request);
	
	
	/**
	 * 更新日志.
	 * 
	 * @param inboundEntity 日志实体
	 * @param handler 处理器
	 * @param response 响应对象
	 */
	public abstract void updateInbound(Object inboundEntity, AbstractHandler<REQ, RESP> handler, RESP response);
	
	
	/**
	 * 主服务流程.
	 * 
	 * @param channelId 上游渠道号
	 * @param channelTxnId 上游渠道流水号
	 * @param request 请求
	 * @return response 响应
	 * @throws OctopusBadRequestException 请求校验异常
	 * @throws OctopusDuplicatedInboundException 数据插入异常
	 */
	@SuppressWarnings("unchecked")
	protected RESP doActivate(String channelId, String channelTxnId, REQ request) throws OctopusBadRequestException, OctopusDuplicatedInboundException {
		
		// 上游渠道号不能为空
		if (!StringUtils.hasLength(channelId)) {
			log.error("channelId cant't be empty");
			throw new OctopusBadRequestException("channelId cant't be empty");
		}
		
		// 上游渠道交易流水号不能为空
		if (!StringUtils.hasLength(channelTxnId)) {
			log.error("channelTxnId cant't be empty");
			throw new OctopusBadRequestException("channelTxnId cant't be empty");
		}
		
		// 上游渠道请求不能为空
		if (request == null) {
			log.error("request cant't be null");
			throw new OctopusBadRequestException("request cant't be null");
		}
				
		// 上游渠道请求实例类型判断
		if (!(request instanceof OctopusSkeletonOnlineRequest)) {
			log.error("request:{} not instanceof {}", request, OctopusSkeletonOnlineRequest.class);
			throw new OctopusBadRequestException("request instance illegal");
		}

		// 获取请求时间
		long requestTime = System.currentTimeMillis(); 
		
		// 上游渠道请求字段的合法性校验
		Set<ConstraintViolation<REQ>> constraintViolations = this.validator.validate(request);
		for (ConstraintViolation<REQ> cv: constraintViolations) {
			log.error("request validation failed:{}, {}", cv.getPropertyPath(), cv.getMessage());
			throw new OctopusBadRequestException(cv.getMessage());
		}

		// 插入日志，交易重复会抛出异常
		Object inbound = null;
		try {
			inbound = insertInbound(channelId, channelTxnId, request);
		} catch (Exception e) {
			throw new OctopusDuplicatedInboundException("duplicated trans, channelId:" + channelId + " channelTxnId:" + channelTxnId, e);
		}

		// 内部处理
		RESP response = null;
		AbstractHandler<REQ, RESP> handler = null;
		try {
			// 根据报文渠道字段ChannelId获取上游渠道参数
			InboundChannelParam inboundChannel = this.paramService.getParameter(InboundChannelParam.class, channelId);
			if (inboundChannel == null) {
				log.error("unsupported channel:{}", channelId);
				throw new OctopusNotPermittedException("unsupported channel:" + channelId);
			}
			log.debug("found parameter of inboundChannel:{}", inboundChannel);
			
			// 获取请求类名
			String requestClassName = ClassUtils.getUserClass(request).getName(); 
			
			// 判断交易是否被渠道允许
			if (inboundChannel.getTransControlEnabled()) {
				if (inboundChannel.getTransControlPermittedRequests() == null ||
						!inboundChannel.getTransControlPermittedRequests().contains(requestClassName)) {
					log.error("request:{} is not permitted by channel:{}", requestClassName, channelId);
					throw new OctopusNotPermittedException("request:" + requestClassName + " is not permitted by channel:" + channelId);
				}
			}
			
			// 根据请求类与降级场景获取对应的handler
			if (inboundChannel.getFallbackEnabled()) {
				// 激活挡板采用挡板场景中对应的handler，没有则采用默认handler
				handler = (AbstractHandler<REQ, RESP>) HANDLER_MAP.get(requestClassName + inboundChannel.getFallbackScenario());
				if (handler == null) {
					log.debug("no mock scenario:{} handler for request:{}, use default!", inboundChannel.getFallbackScenario(), requestClassName);
					handler = (AbstractHandler<REQ, RESP>) HANDLER_MAP.get(requestClassName);
					if (handler == null) {
						log.error("fail to get default handler for request:{}", requestClassName);
						throw new OctopusSkeletonOnlineException("no default handler for " + requestClassName);
					}
				}
			} else {
				handler = (AbstractHandler<REQ, RESP>) HANDLER_MAP.get(requestClassName);
				if (handler == null) {
					log.error("fail to get handler for request:{}", requestClassName);
					throw new OctopusSkeletonOnlineException("no handler for " + requestClassName);
				}
			}
			log.debug("found handler:{} for request:{}", handler, request);
			
			// 清理并设置联机上下文上游渠道相关信息
			OctopusOnlineContextHolder.clearContext();
			OctopusOnlineContextHolder.getContext().setInboundRequest(request);
			OctopusOnlineContextHolder.getContext().setInboundEntity(inbound);
			OctopusOnlineContextHolder.getContext().setInboundHandler(handler);
			
			// 调用handler处理
			response = handler.handle(request);
		} catch (Exception exception) {
			// 异常情况下设置响应
			response = responseWithException(request, exception);
		}
		
		// 更新日志
		updateInbound(inbound, handler, response); 
		
		// 获取响应时间
		long responseTime = System.currentTimeMillis(); 

		// 广播交易处理完成事件
		this.applicationContext.publishEvent(new HandleCompletedEvent(this, this.serviceProperties.getInstanceId(), 
				request, new Date(requestTime), handler, response, new Date(responseTime)));

		// 正常返回
		return response;
	}	
}
