package org.krproject.ocean.skeletons.octopus.online.internal;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.validation.ConstraintViolation;
import javax.validation.Validator;

import org.krproject.ocean.skeletons.octopus.online.api.constants.OctopusSkeletonOnlineApiConstants;
import org.krproject.ocean.skeletons.octopus.online.api.internal.OctopusInternalRequest;
import org.krproject.ocean.skeletons.octopus.online.api.internal.OctopusInternalResponse;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.integration.annotation.MessageEndpoint;
import org.springframework.integration.annotation.ServiceActivator;
import org.springframework.util.ClassUtils;

import lombok.extern.slf4j.Slf4j;


/**
 * 内部交易服务端点.
 *
 * @author Tiger
 *
 */
@Slf4j
@MessageEndpoint
public class OctopusInternalActivator {

	@Resource
	private Validator validator;

	@Autowired(required = false)
	private List<AbstractInternalHandler<?, ?>> internalHandlers;
	
	private static boolean initialized = false;
	
	private static final Map<String, AbstractInternalHandler<?, ?>> INTERNAL_HANDLER_MAP = new ConcurrentHashMap<String, AbstractInternalHandler<?, ?>>();

	
	@PostConstruct
	public void init() {
		if (!initialized) {
			// 加载上游渠道交易处理器
			if (this.internalHandlers != null) {
				for (AbstractInternalHandler<?, ?> handler : this.internalHandlers) {
					AbstractInternalHandler<?, ?> handlerFound = INTERNAL_HANDLER_MAP.get(handler.getRequestName());
					if (handlerFound != null) {
						log.error("internal handler {} and {} got same request name：{}", 
								handlerFound, handler, handler.getRequestName());
						throw new RuntimeException("duplicate internal handler for request " + handler.getRequestName());
					}
					INTERNAL_HANDLER_MAP.put(handler.getRequestName(), handler);
					log.debug("Loaded Internal Handler:{} for Request:{}", ClassUtils.getUserClass(handler).getName(), handler.getRequestName());
				}
			}
			
			// 置为已加载
			initialized = true;
		}
	}
	
	
	@SuppressWarnings({ "rawtypes", "unchecked" })
	@ServiceActivator(inputChannel = OctopusSkeletonOnlineApiConstants.INTERNAL_CHANNEL_NAME)
	public OctopusInternalResponse activate(OctopusInternalRequest<OctopusInternalResponse> request) {

		// 创建默认响应
		OctopusInternalResponse response = request.createResponse();

		// 请求字段的合法性校验
		Set<ConstraintViolation<OctopusInternalRequest>> constraintViolations = this.validator.validate(request);
		for (ConstraintViolation<OctopusInternalRequest> cv: constraintViolations) {
			log.error("constraint voilations:{}, {}", cv.getPropertyPath(), cv.getMessage());
			return response;
		}
		
		// 根据请求类名获取对应处理器
		String requestClassName = ClassUtils.getUserClass(request).getName();
		AbstractInternalHandler handler = INTERNAL_HANDLER_MAP.get(requestClassName);
		if (handler == null) {
			log.error("fail to get handler for request:{}", requestClassName);
			return response;
		}
		
		// 调用处理器处理
		try {
			response = (OctopusInternalResponse) handler.handle(request);
		} catch (Exception e) {
			log.error("handle internal request:{} failed with exception", request, e);
			return response;
		}
		
		return response;
	}
}
