package org.krproject.ocean.skeletons.octopus.online.outbound;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.krproject.ocean.skeletons.octopus.online.context.OctopusOnlineContextHolder;
import org.krproject.ocean.skeletons.octopus.online.event.ProcessCompletedEvent;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusDuplicatedOutboundException;
import org.krproject.ocean.skeletons.octopus.online.exception.OctopusSkeletonOnlineException;
import org.krproject.ocean.skeletons.octopus.param.OutboundSystemParam;
import org.krproject.ocean.vitamins.param.service.ParamService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import lombok.extern.slf4j.Slf4j;



/**
 * 下游系统调度器抽象类.
 *
 * @param <REQ> request
 * @param <RESP>  response
 * @author Tiger
 */
@Slf4j
public abstract class AbstractOutboundDispatcher<REQ, RESP> {

	@Resource
	private ParamService paramService;

	@Resource
	private ApplicationContext applicationContext;
	
	@Autowired(required = false)
	private List<AbstractProcessor<?, ?>> processors;

	private static boolean initialized = false;
	
	private static final Map<String, AbstractProcessor<?, ?>> PROCESSOR_MAP = new ConcurrentHashMap<String, AbstractProcessor<?, ?>>();

	@PostConstruct
	public void init() {
		if (!initialized) {
			// 加载下游系统交易处理器
			if (this.processors != null) {
				for (AbstractProcessor<?, ?> processor : this.processors) {
					AbstractProcessor<?, ?> processorFound = PROCESSOR_MAP.get(processor.getRequestName());
					if (processorFound != null) {
						log.error("processors {} and {} got same request name：{}", processorFound, processor, processor.getRequestName());
						throw new RuntimeException("Duplicate processors for request " + processor.getRequestName());
					}
					PROCESSOR_MAP.put(processor.getRequestName(), processor);
					log.debug("Loaded Processor:{} for Request:{}", ClassUtils.getUserClass(processor).getName(), processor.getRequestName());
				}
			}
			
			// 置为已加载
			initialized = true;
		}
	}
	
	
	/**
	 * 根据异常获取响应.
	 * 
	 * @param request 请求
	 * @param exception 异常
	 * @return RESP 响应 
	 */
	public abstract RESP responseWithException(REQ request, Exception exception);
	
	
	/**
	 * 插入日志.
	 * @param systemId 下游系统号
	 * @param systemTxnId 下游系统流水号
	 * @param request 请求
	 * @return 日志实体
	 */
	public abstract Object insertOutbound(String systemId, String systemTxnId, REQ request);

	
	/**
	 * 更新日志.
	 * 
	 * @param outboundEntity 日志实体
	 * @param processor 处理器
	 * @param response 响应信息
	 */
	public abstract void updateOutbound(Object outboundEntity, AbstractProcessor<REQ, RESP> processor, RESP response);

	
	/**
	 * 下游系统接口调用.
	 * 
	 * @param systemId 下游系统号
	 * @param systemTxnId 下游系统流水号	 
	 * @param request 下游系统请求
	 * @return 下游系统响应
	 * @throws OctopusDuplicatedOutboundException 数据插入异常
	 */
	@SuppressWarnings("unchecked")
	protected RESP doDispatch(String systemId, String systemTxnId, REQ request) throws OctopusDuplicatedOutboundException {

		// 获取下游系统号
		if (!StringUtils.hasLength(systemId)) {
			log.error("systemId cant't be empty");
			throw new OctopusSkeletonOnlineException("systemId cant't be empty");
		}
		
		// 获取下游系统交易流水号
		if (!StringUtils.hasLength(systemTxnId)) {
			log.error("systemTxnId cant't be empty");
			throw new OctopusSkeletonOnlineException("systemTxnId cant't be empty");
		}
		
		// 请求不能为空
		if (request == null) {
			log.error("request cant't be null");
			throw new OctopusSkeletonOnlineException("request cant't be null");
		}
		
		// 设置请求时间
		long requestTime = System.currentTimeMillis(); 
		
		// 插入日志，交易重复会抛出异常
		Object outbound = null;
		try {
			outbound = insertOutbound(systemId, systemTxnId, request);
		} catch (Exception e) {
			throw new OctopusDuplicatedOutboundException("duplicated trans, systemId:" + systemId + " systemTxnId:" + systemTxnId, e);
		}

		// 内部处理
		RESP response = null;
		AbstractProcessor<REQ, RESP> processor = null;
		try {
			// 根据报文渠道字段获取下游系统参数，频率及次数调用控制
			OutboundSystemParam outboundSystem = this.paramService.getParameter(OutboundSystemParam.class, systemId);
			if (outboundSystem == null) {
				log.error("unsupported system:{}", systemId);
				throw new OctopusSkeletonOnlineException("unsupported system: " + systemId);
			}
			log.debug("found parameter for outboundSystem :{}", outboundSystem);
			
			// 根据请求类与挡板场景获取对应的processor
			String requestClassName = ClassUtils.getUserClass(request).getName();
			if (outboundSystem.getMockEnabled()) {
				// 激活挡板采用挡板场景中对应的processor，没有则采用默认processor
				processor = (AbstractProcessor<REQ, RESP>) PROCESSOR_MAP.get(requestClassName + outboundSystem.getMockScenario());
				if (processor == null) {
					log.debug("no mock scenario:{} processor for request:{}, use default!", outboundSystem.getMockScenario(), requestClassName);
					processor = (AbstractProcessor<REQ, RESP>) PROCESSOR_MAP.get(requestClassName);
					if (processor == null) {
						log.error("no default processor for request {}", requestClassName);
						throw new OctopusSkeletonOnlineException("no default processor for request " + requestClassName);
					}
				}
			} else {
				processor = (AbstractProcessor<REQ, RESP>) PROCESSOR_MAP.get(requestClassName);
				if (processor == null) {
					log.error("no processor for request {}", requestClassName);
					throw new OctopusSkeletonOnlineException("no processor for request " + requestClassName);
				}
			}
			log.debug("found processor:{} for request:{}", processor, request);
			
			// 设置联机上下文下游系统相关信息，由 AbstractInboundActivator 统一清理
			OctopusOnlineContextHolder.getContext().setOutboundRequest(request);
			OctopusOnlineContextHolder.getContext().setOutboundEntity(outbound);
			OctopusOnlineContextHolder.getContext().setOutboundProcessor(processor);

			// 调用processor处理
			response = processor.process(request);
		} catch (Exception exception) {
			// 异常情况下设置响应
			response = responseWithException(request, exception);
		}

		// 更新日志
		updateOutbound(outbound, processor, response);
		
		// 获取响应时间
		long responseTime = System.currentTimeMillis(); 
		
		// 广播事件
		this.applicationContext.publishEvent(new ProcessCompletedEvent(this, request, new Date(requestTime), processor, response, new Date(responseTime)));

		// 返回响应
		return response;
	}
}
