package org.krproject.ocean.skeletons.octopus.online.transaction;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.krproject.ocean.skeletons.octopus.online.constants.OctopusSkeletonOnlineConstants;
import org.krproject.ocean.skeletons.octopus.online.transaction.event.InquireCompletedEvent;
import org.krproject.ocean.skeletons.octopus.online.transaction.model.OnlineOutboundModel;
import org.krproject.ocean.skeletons.octopus.online.transaction.service.OctopusOnlineTransactionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.integration.annotation.MessageEndpoint;
import org.springframework.integration.annotation.ServiceActivator;

import lombok.extern.slf4j.Slf4j;


/**
 * 质询类交易服务端点.
 * 
 * @author Tiger
 *
 */
@Slf4j
@MessageEndpoint
public class InquireServiceActivator {
	
	@Resource
	private ApplicationContext applicationContext;

	@Resource
	private OctopusOnlineTransactionService octopusOnlineTransactionService;

	@Autowired(required = false)
	private List<AbstractInquirer> inquirers;
	
	private Map<String, AbstractInquirer> inquirerMap = new ConcurrentHashMap<String, AbstractInquirer>();
	
	@PostConstruct
	public void init() {
		if (this.inquirers != null) {
			for (AbstractInquirer inquirer : this.inquirers) {
				AbstractInquirer inquirerFound = this.inquirerMap.get(inquirer.getProcessorName());
				if (inquirerFound != null) {
					log.error("inquirer {} and {} got same processor name：{}", inquirerFound, inquirer, inquirer.getProcessorName());
					throw new RuntimeException("duplicate inquirer for processor " + inquirer.getProcessorName());
				}
				this.inquirerMap.put(inquirer.getProcessorName(), inquirer);
			}
		}
	}
	
	
	@ServiceActivator(inputChannel = OctopusSkeletonOnlineConstants.INQUIRE_CHANNEL_NAME)
	private void activate(String inboundId) throws Exception {
		log.info("inquire inboundId:{}", inboundId);

		Boolean succeed = true; // 本交易是否成功
		// 根据inbound id顺序遍历所有的outbound流水，并质询"处理中"和“异常”的交易结果
		for (OnlineOutboundModel outbound : this.octopusOnlineTransactionService.listOutboundForInquire(inboundId)) {
			log.debug("inquire outboundId:{} of inboundId:{}!", outbound.getOutboundId(), inboundId);
			
			// 根据处理器找到对应质询器
			AbstractInquirer inquirer = this.inquirerMap.get(outbound.getProcessorName());
			if (inquirer != null) {
				// 使用质询器对原Outbound交易质询
				Boolean result = inquireOutbound(outbound.getOutboundId(), inquirer);
				if (result == null) {
					// 交易未明
					succeed = null;
					break;
				} else if (!result) {
					// 交易失败
					succeed = false;
					break; 
				}
			}
		}
		log.info("inquire inboundId:{} result:{}", inboundId, succeed);
		
		// 事务处理
		if (succeed != null) {
			if (succeed) {
				// 交易成功，发起重做
				this.octopusOnlineTransactionService.replay(inboundId);
			} else {
				// 交易失败，发起冲正
				this.octopusOnlineTransactionService.reverse(inboundId);
			}
		}
		
		// 广播质询完成事件
		this.applicationContext.publishEvent(new InquireCompletedEvent(this, inboundId, succeed));
	}
	
	
	/**
	 * 质询下游系统交易状态.
	 * @param outboundId 下游系统交易编号
	 * @param inquirer 质询器
	 * @return 交易是否成功，null-未明，true-成功，false-失败
	 */
	private Boolean inquireOutbound(String outboundId, AbstractInquirer inquirer) {
		Boolean result = null; // 本交易是否成功
		
		// 对原交易冲正，有间隔与次数限制
		for (int retry = 1; retry <= inquirer.getInquireTimes(); retry++) {
			log.debug("inquire outboundId:{} with times:{}!", outboundId, retry);
							
			// 调用冲正器进行冲正处理
			try {
				if (inquirer.inquire(outboundId)) {
					log.debug("inquire outboundId:{} success!", outboundId);
					result = true;
				} else {
					log.debug("inquire outboundId:{} failure!", outboundId);
					result = false;
				}
				break;  // 有明确结果，不再冲正
			} catch (Exception e) {
				log.debug("inquire outboundId:{} exception:{}!, retry later!", outboundId, e);
				try {
					// 睡眠质询间隔后，继续重试
					Thread.sleep(inquirer.getInquireInterval());
				} catch (InterruptedException e1) {
					// ignore
				}
			}
		}
		return result;
	}
}
