package org.krproject.ocean.vitamins.online.api;

import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.util.Assert;

import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * 分页响应对象，支持序列化.
 * 
 * @param <T> T
 * 
 * @author zhongyang
 * 
 */
@SuppressWarnings("serial")
public class OnlinePageResponse<T> implements Serializable, Page<T> {
	
	private long total;
	
	private OnlinePageRequest pageable;
	
	private List<T> content;
	
	public OnlinePageResponse() {
	}
	
	public OnlinePageResponse(List<T> content) {
		this(content, null, content.size());
	}
	
	public OnlinePageResponse(List<T> content, OnlinePageRequest pageable) {
		this(content, pageable, content.size());
	}
	
	public OnlinePageResponse(List<T> content, OnlinePageRequest pageable, long total) {
		this.content = content;
		this.pageable = pageable;
		this.total = total;
	}

	@Override
	public List<T> getContent() {
		return Collections.unmodifiableList(this.content);
	}

	@JsonIgnore
	@Override
	public int getNumber() {
		return ((this.pageable == null) ? 0 : this.pageable.getPageNumber());
	}

	@JsonIgnore
	@Override
	public int getNumberOfElements() {
		return this.content.size();
	}

	@JsonIgnore
	@Override
	public int getSize() {
		return ((this.pageable == null) ? 0 : this.pageable.getPageSize());
	}

	@JsonIgnore
	@Override
	public Sort getSort() {
		return ((this.pageable == null) ? null : this.pageable.getSort());
	}

	@JsonIgnore
	@Override
	public boolean hasContent() {
		return (!(this.content.isEmpty()));
	}

	@JsonIgnore
	@Override
	public boolean hasNext() {
		return (getNumber() + 1 < getTotalPages());
	}

	@JsonIgnore
	@Override
	public boolean hasPrevious() {
		return (getNumber() > 0);
	}

	@JsonIgnore
	@Override
	public boolean isFirst() {
		return (!(hasPrevious()));
	}

	@JsonIgnore
	@Override
	public boolean isLast() {
		return (!(hasNext()));
	}
	
	@JsonIgnore
	@Override
	public boolean isEmpty() {
		return Page.super.isEmpty();
	}

	@Override
	public Pageable nextPageable() {
		return ((hasNext()) ? this.pageable.next() : null);
	}

	@Override
	public Pageable previousPageable() {
		if (hasPrevious()) {
			return this.pageable.previousOrFirst();
		}

		return null;
	}

	@Override
	public Iterator<T> iterator() {
		return this.content.iterator();
	}

	@JsonIgnore
	@Override
	public int getTotalPages() {
		return ((getSize() == 0) ? 1 : (int) Math.ceil(this.total / getSize()));
	}

	@JsonIgnore
	@Override
	public long getTotalElements() {
		return this.total;
	}
	

	protected <U> List<U> getConvertedContent(Function<? super T, ? extends U> converter) {

		Assert.notNull(converter, "Function must not be null!");

		return this.stream().map(converter::apply).collect(Collectors.toList());
	}
	

	@SuppressWarnings({ "rawtypes", "unchecked" })
	@Override
	public <S> Page<S> map(Function<? super T, ? extends S> paramConverter) {
		return new OnlinePageResponse(getConvertedContent(paramConverter), this.pageable);
	}

	public long getTotal() {
		return this.total;
	}

	public void setTotal(long total) {
		this.total = total;
	}

	public OnlinePageRequest getPageable() {
		return this.pageable;
	}

	public void setPageable(OnlinePageRequest pageable) {
		this.pageable = pageable;
	}

	public void setContent(List<T> content) {
		this.content = content;
	}
}
