package org.krproject.schedule.service.impl;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.krproject.schedule.domain.ScheduleTaskDefine;
import org.krproject.schedule.domain.ScheduleTaskDefineRepository;
import org.krproject.schedule.domain.ScheduleTaskJournal;
import org.krproject.schedule.domain.ScheduleTaskJournalRepository;
import org.krproject.schedule.exception.ScheduleException;
import org.krproject.schedule.service.AbstractScheduleTask;
import org.krproject.schedule.service.ScheduleService;
import org.krproject.schedule.service.ScheduledTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Service;
import org.springframework.util.ClassUtils;


/**
 * 定时任务服务实现.
 * 
 * @author Tiger
 */
@Service
public class ScheduleServiceImpl implements ScheduleService {

	private final Logger logger = LoggerFactory.getLogger(getClass());
	
	@Resource
	private ScheduleTaskDefineRepository scheduleTaskDefineRepository;
	
	@Resource
	private ScheduleTaskJournalRepository scheduleTaskJournalRepository;
	
	@Resource
	private TaskScheduler taskScheduler;
	
	@Autowired(required = false)
	private List<AbstractScheduleTask> taskClasses;
	
	//系统当前提供的任务类
	private Map<String, AbstractScheduleTask> taskClassMap = new ConcurrentHashMap<String, AbstractScheduleTask>();
	
	//参数配置的调度任务
	private Map<String, ScheduledTask> scheduledTaskMap = new ConcurrentHashMap<String, ScheduledTask>();
	
	@PostConstruct
	public void init() {
		if (this.taskClasses != null) {
			for (AbstractScheduleTask taskClass:this.taskClasses) {
				String taskClassName = ClassUtils.getUserClass(taskClass).getName();
				AbstractScheduleTask taskFound = this.taskClassMap.get(taskClassName);
				if (taskFound != null) {
					this.logger.error("task {} and {} got same processor name：{}", taskFound, taskClass, taskClassName);
					throw new RuntimeException("duplicate inquirer for processor " + taskClassName);
				}
				this.taskClassMap.put(taskClassName, taskClass);
			}
		}
	}

	
	@Override
	public void start() {
		this.logger.debug("schedule starting");
		
		//先停止所有当前任务
		for (String taskName:this.scheduledTaskMap.keySet()) {
			taskCancel(taskName);
		}
				
		//再获取任务配置列表，加入任务调度
		for (ScheduleTaskDefine std:taskDefineList(new PageRequest(0, Integer.MAX_VALUE))) {
			taskSchedule(std.getTaskName());
		}
	}
	
	@Override
	public void stop() {
		this.logger.debug("schedule stoping");
		
		//停止并清空所有当前任务
		for (String taskName:this.scheduledTaskMap.keySet()) {
			taskCancel(taskName);
		}
	}


	@Override
	public Map<String, ScheduledTask> taskMap() {
		return this.scheduledTaskMap;
	}


	@Override
	public ScheduledTask taskGet(String taskName) {
		return this.scheduledTaskMap.get(taskName);
	}


	@Override
	public ScheduledTask taskSchedule(String taskName) throws ScheduleException {
		this.logger.debug("schedule task:{}", taskName);
		
		ScheduledTask stFound = taskGet(taskName);
		if (stFound != null) {
			this.logger.error("already scheduled task:{} ", taskName);
			throw new ScheduleException("already scheduled task " + taskName);
		}
		
		ScheduleTaskDefine std = taskDefineGet(taskName);
		if (std == null) {
			this.logger.error("no task define named {}", taskName);
			throw new ScheduleException("no task define named " + taskName);
		}
		//未启用的不加入调度
		if (!std.getEnabled()) {
			this.logger.debug("skip not enabled task:{}", std);
			return null;
		}
		
		AbstractScheduleTask ast = taskClassGet(std.getTaskClassName());
		if (ast == null) {
			this.logger.error("no task class for {}", std.getTaskClassName());
			throw new ScheduleException("no task class for " + std.getTaskClassName());
		}
		
		ScheduledTask st = new ScheduledTask(std, ast);
		ScheduledFuture<?> sf = this.taskScheduler.schedule(st, new CronTrigger(std.getTaskCron()));
		st.setSf(sf);
		
		this.scheduledTaskMap.put(taskName, st);
		
		return st;
	}

	
	@Override
	public void taskCancel(String taskName) throws ScheduleException {
		ScheduledTask st = taskGet(taskName);
		if (st == null) {
			this.logger.error("task:{} not scheduled ", taskName);
			throw new ScheduleException("not scheduled task " + taskName);
		}
		st.cancel(false);
		this.scheduledTaskMap.remove(taskName);
	}


	@Override
	public List<AbstractScheduleTask> taskClassList() {
		return this.taskClasses;
	}


	@Override
	public AbstractScheduleTask taskClassGet(String taskClassName) {
		return this.taskClassMap.get(taskClassName);
	}


	@Override
	public Page<ScheduleTaskDefine> taskDefineList(Pageable pageable) {
		return this.scheduleTaskDefineRepository.findAll(pageable);
	}


	@Override
	public Page<ScheduleTaskDefine> taskDefineListByTaskClassName(String taskClassName, Pageable pageable) {
		return this.scheduleTaskDefineRepository.findByTaskClassName(taskClassName, pageable);
	}


	@Override
	public ScheduleTaskDefine taskDefineGet(String taskName) {
		return this.scheduleTaskDefineRepository.findOne(taskName);
	}


	@Override
	public ScheduleTaskDefine taskDefineAdd(ScheduleTaskDefine std) {
		return this.scheduleTaskDefineRepository.save(std);
	}


	@Override
	public ScheduleTaskDefine taskDefineUpdate(String taskName, ScheduleTaskDefine newStd) {
		//主键检查
		if (!newStd.getTaskName().equals(taskName)) {
			this.logger.error("task name not match, {}, {}", taskName, newStd);
			return null;
		}
		
		//找到原实体记录
		ScheduleTaskDefine oldStd = this.scheduleTaskDefineRepository.findOne(taskName);
		if (oldStd == null) {
			this.logger.error("task name not found, {}", taskName);
			return null;
		}
		
		//更新原实体记录
		BeanUtils.copyProperties(newStd, oldStd);
		
		return this.scheduleTaskDefineRepository.save(oldStd);
	}


	@Override
	public void taskDefineRemove(String taskName) {
		this.scheduleTaskDefineRepository.delete(taskName);
	}


	@Override
	public ScheduleTaskDefine taskDefineEnable(String taskName) {
		ScheduleTaskDefine std = this.scheduleTaskDefineRepository.findOne(taskName);
		if (std != null) {
			if (std.getEnabled()) {
				this.logger.debug("task:{} disabled", taskName);
				std.setEnabled(false);
			}
			return std;
		}
		return null;
	}


	@Override
	public ScheduleTaskDefine taskDefineDisable(String taskName) {
		ScheduleTaskDefine std = this.scheduleTaskDefineRepository.findOne(taskName);
		if (std != null) {
			if (!std.getEnabled()) {
				this.logger.debug("task:{} enabled", taskName);
				std.setEnabled(true);
			}
			return std;
		}
		return null;
	}


	@Override
	public Page<ScheduleTaskJournal> taskJournalList(Pageable pageable) {
		return this.scheduleTaskJournalRepository.findAll(pageable);
	}


	@Override
	public Page<ScheduleTaskJournal> taskJournalListByTaskName(String taskName, Pageable pageable) {
		return this.scheduleTaskJournalRepository.findByTaskName(taskName, pageable);
	}


	@Override
	public Page<ScheduleTaskJournal> taskJournalListByTaskClassName(String taskClassName, Pageable pageable) {
		return this.scheduleTaskJournalRepository.findByTaskClassName(taskClassName, pageable);
	}


	@Override
	public ScheduleTaskJournal taskJournalGet(BigDecimal journalId) {
		return this.scheduleTaskJournalRepository.findOne(journalId);
	}

}
