/*
 * Copyright 2005-2013 The Kuali Foundation
 * 
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.opensource.org/licenses/ecl1.php
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.kra.institutionalproposal.contacts;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.kuali.kra.bo.Unit;
import org.kuali.kra.institutionalproposal.document.InstitutionalProposalDocument;
import org.kuali.kra.institutionalproposal.home.InstitutionalProposal;
import org.kuali.rice.krad.util.GlobalVariables;

/**
 * This class...
 */
public class InstitutionalProposalProjectPersonAddRuleImpl extends BaseInstitutionalProposalContactAddRule implements
        InstitutionalProposalProjectPersonAddRule {

    /**
     * @see org.kuali.kra.institutionalproposal.contacts.InstitutionalProposalProjectPersonAddRule
     */
    public boolean processAddInstitutionalProposalProjectPersonBusinessRules(InstitutionalProposalProjectPersonRuleAddEvent event) {
        InstitutionalProposalPerson newProjectPerson = event.getNewProjectPerson();
        InstitutionalProposal institutionalProposal = ((InstitutionalProposalDocument) event.getDocument()).getInstitutionalProposal();
        
        return checkForSelectedContactAndRole(newProjectPerson) && (checkForExistingPrincipalInvestigators(institutionalProposal, newProjectPerson) 
                                                                    & checkForDuplicatePerson(institutionalProposal, newProjectPerson));
    }
    
    boolean checkForSelectedContactAndRole(InstitutionalProposalContact newContact) {
        return super.checkForSelectedContactAndRole(newContact, PROPOSAL_PROJECT_PERSON_LIST_ERROR_KEY);
    }
    
    /**
     * Verify a PI exists
     * @param award
     * @param newProjectPerson
     * @return
     */
    boolean checkForExistingPrincipalInvestigators(InstitutionalProposal institutionalProposal, InstitutionalProposalPerson newProjectPerson) {
        boolean valid = true;
        if(newProjectPerson.isPrincipalInvestigator()) {
            for(InstitutionalProposalPerson p: institutionalProposal.getProjectPersons()) {
                if(p.isPrincipalInvestigator()) {
                    valid = false;
                    break;
                }
            }
        }
        
        if(!valid) {
            GlobalVariables.getMessageMap().putError(PROPOSAL_PROJECT_PERSON_LIST_ERROR_KEY, ERROR_PROPOSAL_PROJECT_PERSON_PI_EXISTS);
        }
        
        return valid;
    }

    /**
     * Verify no duplicate person
     * @param award
     * @param newProjectPerson
     * @return
     */
    boolean checkForDuplicatePerson(InstitutionalProposal institutionalProposal, InstitutionalProposalPerson newProjectPerson) {
        boolean valid = true;
        for(InstitutionalProposalPerson p: institutionalProposal.getProjectPersons()) {
            if (p.getClass().equals(newProjectPerson.getClass()) 
                    && p.getContact().getIdentifier().equals(newProjectPerson.getContact().getIdentifier())
                    && isProjectPersonInvestigator(p) && isProjectPersonInvestigator(newProjectPerson)) {
                valid = false;
                break;
            }
        }
        
        if(!valid) {
            GlobalVariables.getMessageMap().putError(PROPOSAL_PROJECT_PERSON_LIST_ERROR_KEY, ERROR_PROPOSAL_PROJECT_PERSON_EXISTS, 
                                                                                newProjectPerson.getContact().getFullName());
        }
        
        return valid;
    }
    
    private boolean isProjectPersonInvestigator(InstitutionalProposalPerson projectPerson) {
        return projectPerson.isCoInvestigator() || projectPerson.isPrincipalInvestigator();
    }
    /**
     * verify no duplicate units for a person
     * @param projectPersons
     * @return
     */
    boolean checkForDuplicateUnits(List<InstitutionalProposalPerson> projectPersons) {
        boolean valid = true;
        for(InstitutionalProposalPerson p: projectPersons) {
            Set<Unit> uniqueUnits = new HashSet<Unit>();
            List<Unit> tempUnits = new ArrayList<Unit>();
            for(InstitutionalProposalPersonUnit apu: p.getUnits()) {
                uniqueUnits.add(apu.getUnit());
                tempUnits.add(apu.getUnit());
            }
            
            valid &= tempUnits.size() == uniqueUnits.size();
            if(!valid) {
                removeUniqueUnitsFromAllUnits(uniqueUnits, tempUnits);
                reportDuplicateUnits(p, tempUnits);
            }
        }
        
        return valid;
    }

    private void reportDuplicateUnits(InstitutionalProposalPerson p, List<Unit> tempUnits) {
        Set<Unit> duplicateUnits = new HashSet<Unit>(tempUnits);
        for(Unit dupeUnit: duplicateUnits) {
            GlobalVariables.getMessageMap().putError(PROPOSAL_PROJECT_PERSON_LIST_ERROR_KEY, 
                                                    ERROR_PROPOSAL_PROJECT_PERSON_DUPLICATE_UNITS, 
                                                    dupeUnit.getUnitName(), dupeUnit.getUnitNumber(),
                                                    p.getFullName());
        }
    }

    private void removeUniqueUnitsFromAllUnits(Set<Unit> uniqueUnits, List<Unit> tempUnits) {
        for(Unit u: uniqueUnits) {
            tempUnits.remove(u);
        }
    }
}
