/*
 * Copyright 2005-2013 The Kuali Foundation
 * 
 * Licensed under the Educational Community License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.opensource.org/licenses/ecl1.php
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kuali.kra.institutionalproposal.service.impl;

import java.io.IOException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.kuali.kra.award.home.fundingproposal.AwardFundingProposal;
import org.kuali.kra.bo.CustomAttribute;
import org.kuali.kra.bo.CustomAttributeDocument;
import org.kuali.kra.bo.versioning.VersionStatus;
import org.kuali.kra.budget.BudgetDecimal;
import org.kuali.kra.budget.core.Budget;
import org.kuali.kra.budget.distributionincome.BudgetCostShare;
import org.kuali.kra.budget.distributionincome.BudgetUnrecoveredFandA;
import org.kuali.kra.budget.parameters.BudgetPeriod;
import org.kuali.kra.infrastructure.Constants;
import org.kuali.kra.institutionalproposal.ProposalStatus;
import org.kuali.kra.institutionalproposal.contacts.InstitutionalProposalPerson;
import org.kuali.kra.institutionalproposal.contacts.InstitutionalProposalPersonCreditSplit;
import org.kuali.kra.institutionalproposal.contacts.InstitutionalProposalPersonUnit;
import org.kuali.kra.institutionalproposal.contacts.InstitutionalProposalPersonUnitCreditSplit;
import org.kuali.kra.institutionalproposal.customdata.InstitutionalProposalCustomData;
import org.kuali.kra.institutionalproposal.document.InstitutionalProposalDocument;
import org.kuali.kra.institutionalproposal.exception.InstitutionalProposalCreationException;
import org.kuali.kra.institutionalproposal.home.InstitutionalProposal;
import org.kuali.kra.institutionalproposal.home.InstitutionalProposalCostShare;
import org.kuali.kra.institutionalproposal.home.InstitutionalProposalUnrecoveredFandA;
import org.kuali.kra.institutionalproposal.proposaladmindetails.ProposalAdminDetails;
import org.kuali.kra.institutionalproposal.service.InstitutionalProposalService;
import org.kuali.kra.institutionalproposal.service.InstitutionalProposalVersioningService;
import org.kuali.kra.institutionalproposal.specialreview.InstitutionalProposalSpecialReview;
import org.kuali.kra.institutionalproposal.specialreview.InstitutionalProposalSpecialReviewExemption;
import org.kuali.kra.proposaldevelopment.bo.DevelopmentProposal;
import org.kuali.kra.proposaldevelopment.bo.PropScienceKeyword;
import org.kuali.kra.proposaldevelopment.bo.ProposalPerson;
import org.kuali.kra.proposaldevelopment.bo.ProposalPersonCreditSplit;
import org.kuali.kra.proposaldevelopment.bo.ProposalPersonUnit;
import org.kuali.kra.proposaldevelopment.bo.ProposalUnitCreditSplit;
import org.kuali.kra.proposaldevelopment.specialreview.ProposalSpecialReview;
import org.kuali.kra.service.ServiceHelper;
import org.kuali.kra.service.VersionException;
import org.kuali.kra.service.VersioningService;
import org.kuali.rice.core.api.util.type.KualiDecimal;
import org.kuali.rice.coreservice.framework.parameter.ParameterService;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.krad.bo.AdHocRouteRecipient;
import org.kuali.rice.krad.service.BusinessObjectService;
import org.kuali.rice.krad.service.DocumentService;
import org.kuali.rice.krad.service.SequenceAccessorService;
import org.kuali.rice.krad.util.ObjectUtils;
import org.springframework.transaction.annotation.Transactional;

/**
 * This class provides the default implementation of the InstitutionalProposalService.
 */
@Transactional
public class InstitutionalProposalServiceImpl implements InstitutionalProposalService {
    private static final Log LOG = LogFactory.getLog(InstitutionalProposalServiceImpl.class);
    
    private static final String WORKFLOW_EXCEPTION_MESSAGE = "Caught workflow exception creating new Institutional Proposal";
    private static final String VERSION_EXCEPTION_MESSAGE = "Caught version exception creating new Institutional Proposal";
    private static final String ROUTE_MESSAGE = "Autogenerated Institutional Proposal from Development Proposal ";
    private static final String NO_PRIOR_VERSION_MESSAGE = "Tried to version an InstitutionalProposal where no prior version exists.";
    private static final String NEW_DOCUMENT_DESCRIPTION = "Generated by Dev Proposal ";
    private static final String DECIMAL_FORMAT = "00000000";
    private static final String PROPOSAL_NUMBER = "proposalNumber";
    private static final String SEQUENCE_NUMBER = "sequenceNumber";
    
    private BusinessObjectService businessObjectService;
    private DocumentService documentService;
    private VersioningService versioningService;
    private InstitutionalProposalVersioningService institutionalProposalVersioningService;
    private SequenceAccessorService sequenceAccessorService;
    private ParameterService parameterService;
    
    /**
     * Creates a new pending Institutional Proposal based on given development proposal and budget.
     * 
     * @param developmentProposal DevelopmentProposal
     * @param budget Budget
     * @return String The new proposal number
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#createInstitutionalProposal(DevelopmentProposal, Budget)
     */
    public String createInstitutionalProposal(DevelopmentProposal developmentProposal, Budget budget) {
        
        try {
            InstitutionalProposal institutionalProposal = new InstitutionalProposal();
            
            // Set proposal number on new Institutional Proposal so that it will be propagated to all created child BO's before initial save.

            institutionalProposal.setProposalNumber(getNextInstitutionalProposalNumber());
            
            InstitutionalProposalDocument institutionalProposalDocument = mergeProposals(institutionalProposal, developmentProposal, budget);
            documentService.routeDocument(institutionalProposalDocument, 
                    ROUTE_MESSAGE + developmentProposal.getProposalNumber(), 
                    new ArrayList<AdHocRouteRecipient>());
            return institutionalProposalDocument.getInstitutionalProposal().getProposalNumber();
        } catch (WorkflowException ex) {
            throw new InstitutionalProposalCreationException(WORKFLOW_EXCEPTION_MESSAGE, ex);
        } 
    }
    
    /**
     * Creates a new active version of the Institutional Proposal corresponding to the given proposal number, 
     * with data copied from the given development proposal and budget.
     * 
     * @param proposalNumber String
     * @param developmentProposal DevelopmentProposal
     * @param budget Budget
     * @return String The new version number
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#createInstitutionalProposalVersion(String, DevelopmentProposal, Budget)
     */
    public String createInstitutionalProposalVersion(String proposalNumber, DevelopmentProposal developmentProposal, Budget budget) {
        
        try {
            InstitutionalProposalDocument newInstitutionalProposalDocument = versionProposal(proposalNumber, developmentProposal, budget);
            documentService.routeDocument(newInstitutionalProposalDocument, 
                    ROUTE_MESSAGE + developmentProposal.getProposalNumber(), 
                    new ArrayList<AdHocRouteRecipient>());
            institutionalProposalVersioningService.updateInstitutionalProposalVersionStatus(
                    newInstitutionalProposalDocument.getInstitutionalProposal(), VersionStatus.ACTIVE);
            return newInstitutionalProposalDocument.getInstitutionalProposal().getSequenceNumber().toString();
        } catch (WorkflowException we) {
            throw new InstitutionalProposalCreationException(WORKFLOW_EXCEPTION_MESSAGE, we);
        } catch (VersionException ve) {
            throw new InstitutionalProposalCreationException(VERSION_EXCEPTION_MESSAGE, ve);
        } 
    }
    
    /**
     * Return an Institutional Proposal, if one exists.
     * 
     * @param proposalId String
     * @return InstitutionalProposal, or null if none is found.
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#getInstitutionalProposal(String)
     */
    public InstitutionalProposal getInstitutionalProposal(String proposalId) {
        Map<String, String> criteria = new HashMap<String, String>();
        criteria.put(InstitutionalProposal.PROPOSAL_ID_PROPERTY_STRING, proposalId);
        return (InstitutionalProposal) businessObjectService.findByPrimaryKey(InstitutionalProposal.class, criteria);
    }
    
    /**
     * Return the PENDING version of an Institutional Proposal, if one exists.
     * Note, PENDING here refers to the Version Status, NOT the Proposal Status of the Institutional Proposal.
     * 
     * This is just a pass-through to InstitutionalProposalVersioningService, but we needed this method to be part of 
     * the module API.
     * 
     * @param proposalNumber String
     * @return InstitutionalProposal, or null if a PENDING version is not found.
     * @see org.kuali.kra.bo.versioning.VersionStatus
     */
    public InstitutionalProposal getPendingInstitutionalProposalVersion(String proposalNumber) {
        return institutionalProposalVersioningService.getPendingInstitutionalProposalVersion(proposalNumber);
    }
    
    /**
     * Return the ACTIVE version of an Institutional Proposal, if one exists.
     * Note, ACTIVE here refers to the Version Status, NOT the Proposal Status of the Institutional Proposal.
     * 
     * @param proposalNumber String
     * @return InstitutionalProposal, or null if a ACTIVE version is not found.
     * @see org.kuali.kra.bo.versioning.VersionStatus
     */
    public InstitutionalProposal getActiveInstitutionalProposalVersion(String proposalNumber) {
        return institutionalProposalVersioningService.getActiveInstitutionalProposalVersion(proposalNumber);
    }
    
    /**
     * Designate one or more Institutional Proposals as Funded by an Award.
     * 
     * If the given Proposal has a Proposal Status of Pending, a new Final version of the Proposal
     * will be created with a Proposal Status of Funded.
     * 
     * If the current Active version is already Funded, it will be left alone.
     * 
     * @param proposalNumbers The proposals to update.
     * @return List<InstitutionalProposal> The new Funded versions.
     */
    public List<InstitutionalProposal> fundInstitutionalProposals(Set<String> proposalNumbers) {

        List<InstitutionalProposal> updatedProposals = new ArrayList<InstitutionalProposal>();
        
        try {
            for (String proposalNumber : proposalNumbers) {
                InstitutionalProposal activeVersion = getActiveInstitutionalProposal(proposalNumber);
                
                if (activeVersion != null && !ProposalStatus.FUNDED.equals(activeVersion.getStatusCode())) {
                    
                    LOG.info("Creating a new version of proposal " + proposalNumber + ".");
                    
                    InstitutionalProposal newVersion = versioningService.createNewVersion(activeVersion);
                    newVersion.setStatusCode(ProposalStatus.FUNDED);
                    newVersion.setAwardFundingProposals(transferFundingProposals(activeVersion, newVersion));
                    
                    InstitutionalProposalDocument institutionalProposalDocument = 
                        (InstitutionalProposalDocument) documentService.getNewDocument(InstitutionalProposalDocument.class);
                    
                    institutionalProposalDocument.getDocumentHeader().setDocumentDescription(
                            activeVersion.getInstitutionalProposalDocument().getDocumentHeader().getDocumentDescription());
                    
                    institutionalProposalDocument.setInstitutionalProposal(newVersion);
                    
                    documentService.routeDocument(institutionalProposalDocument, 
                            "Update Proposal Status to Funded", new ArrayList<AdHocRouteRecipient>());
                    
                    updatedProposals.add(newVersion);
                    
                } else if (activeVersion != null && ProposalStatus.FUNDED.equals(activeVersion.getStatusCode())) {
                    LOG.info("Skipped creating a new version of proposal " + proposalNumber + " - proposal is already Funded.");
                } else if (activeVersion == null) {
                    LOG.warn("Could not designate proposal " + proposalNumber + " as Funded: no Active version found.");
                }
            }
            
            return updatedProposals;
            
        } catch (WorkflowException we) {
            throw new InstitutionalProposalCreationException(WORKFLOW_EXCEPTION_MESSAGE, we);
        } catch (VersionException ve) {
            throw new InstitutionalProposalCreationException(VERSION_EXCEPTION_MESSAGE, ve);
        }
    }
    
    /**
     * Designate the given Proposals as no longer funded by the given Award.
     * 
     * If the given Award was the only funding Award for a Proposal, a new Final version of the Proposal
     * will be created with a Proposal Status of Pending.
     * 
     * If the Proposal has other funding Awards, it will be left alone.  It will also be left alone
     * if it is funded by the active version of the given award number (this is a functional requirement).
     * 
     * @param proposalNumbers The proposals to update.
     * @param awardNumber The Award that is de-funding the proposal.
     * @param awardSequence The sequence number of the Award.
     * @return List<InstitutionalProposal> The new Pending versions.
     */
    public List<InstitutionalProposal> defundInstitutionalProposals(Set<String> proposalNumbers, String awardNumber, Integer awardSequence) {

        List<InstitutionalProposal> updatedProposals = new ArrayList<InstitutionalProposal>();
        
        try {
            for (String proposalNumber : proposalNumbers) {
                InstitutionalProposal activeVersion = getActiveInstitutionalProposal(proposalNumber);
                
                if (activeVersion != null && activeVersion.isFundedByAward(awardNumber, awardSequence)
                        && activeVersion.getActiveAwardFundingProposals().size() == 1) {
                    LOG.info("Creating a new version of proposal " + proposalNumber + ".");
                    
                    InstitutionalProposal newVersion = versioningService.createNewVersion(activeVersion);
                    newVersion.getAwardFundingProposals().clear();
                    newVersion.setStatusCode(ProposalStatus.PENDING);
                    
                    InstitutionalProposalDocument institutionalProposalDocument = 
                        (InstitutionalProposalDocument) documentService.getNewDocument(InstitutionalProposalDocument.class);
                    
                    institutionalProposalDocument.getDocumentHeader().setDocumentDescription(
                            activeVersion.getInstitutionalProposalDocument().getDocumentHeader().getDocumentDescription());
                    
                    institutionalProposalDocument.setInstitutionalProposal(newVersion);
                    
                    documentService.routeDocument(institutionalProposalDocument, 
                            "Update Proposal Status to Pending", new ArrayList<AdHocRouteRecipient>());
                    
                    updatedProposals.add(newVersion);
                    
                } else {
                    LOG.info("Skipped setting proposal " + proposalNumber + " to Pending. It is either funded by another Award, " 
                            + "another version of Award " + awardNumber + ", or no active version found.");
                }
            }
            
            return updatedProposals;
            
        } catch (WorkflowException we) {
            throw new InstitutionalProposalCreationException(WORKFLOW_EXCEPTION_MESSAGE, we);
        } catch (VersionException ve) {
            throw new InstitutionalProposalCreationException(VERSION_EXCEPTION_MESSAGE, ve);
        }
    }
    
    /**
     * 
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#getProposalForProposalNumber(java.lang.String)
     */
    public List<InstitutionalProposal> getProposalsForProposalNumber(String proposalNumber) {
        List<InstitutionalProposal> results = new ArrayList<InstitutionalProposal>(businessObjectService.findMatchingOrderBy(InstitutionalProposal.class, 
                                                                ServiceHelper.getInstance().buildCriteriaMap(PROPOSAL_NUMBER, proposalNumber),
                                                                SEQUENCE_NUMBER,
                                                                true));
        return results;    
    }
    
    /**
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#getAllLinkedDevelopmentProposals(java.lang.String)
     */
    public List<DevelopmentProposal> getAllLinkedDevelopmentProposals(String proposalNumber) {
        List<DevelopmentProposal> result = new ArrayList<DevelopmentProposal>();
        List<InstitutionalProposal> proposals = getProposalsForProposalNumber(proposalNumber);
        for (InstitutionalProposal curProposal : proposals) {
            List<ProposalAdminDetails> details = new ArrayList<ProposalAdminDetails>(businessObjectService.findMatching(ProposalAdminDetails.class,
                    ServiceHelper.getInstance().buildCriteriaMap("instProposalId", curProposal.getProposalId())));
            for (ProposalAdminDetails detail : details) {
                result.add(detail.getDevelopmentProposal());
            }
        }
        return result;
    }
    
    public String getNextInstitutionalProposalNumber() {
        Long nextProposalNumber = sequenceAccessorService.getNextAvailableSequenceNumber(Constants.INSTITUTIONAL_PROPSAL_PROPSAL_NUMBER_SEQUENCE);
        DecimalFormat formatter = new DecimalFormat(DECIMAL_FORMAT);
        String nextProposalNumberAsString = formatter.format(nextProposalNumber);
        return nextProposalNumberAsString;
    }
    
    /* Local helper methods */
    
    /**
     * Queries the persistence layer to find the InstitutionalProposal record for the given proposalNumber.
     * 
     * @param proposalNumber String
     * @return InstitutionalProposal
     */
    @SuppressWarnings("unchecked")
    protected InstitutionalProposal getActiveInstitutionalProposal(String proposalNumber) {
        Map<String, String> criteria = new HashMap<String, String>();
        criteria.put(InstitutionalProposal.PROPOSAL_NUMBER_PROPERTY_STRING, proposalNumber);
        criteria.put(InstitutionalProposal.PROPOSAL_SEQUENCE_STATUS_PROPERTY_STRING, VersionStatus.ACTIVE.toString());
        Collection results = businessObjectService.findMatching(InstitutionalProposal.class, criteria);
        if (results.isEmpty()) {
            return null;
        }
        
        return (InstitutionalProposal) results.toArray()[0];
    }
    
    protected InstitutionalProposalDocument versionProposal(String proposalNumber, DevelopmentProposal developmentProposal, Budget budget)
        throws VersionException, WorkflowException {
        
        InstitutionalProposal currentVersion = getActiveInstitutionalProposal(proposalNumber);
        if (currentVersion == null) {
            throw new RuntimeException(NO_PRIOR_VERSION_MESSAGE);
        }
        ObjectUtils.materializeObjects(currentVersion.getInstitutionalProposalScienceKeywords());
        InstitutionalProposal newVersion = versioningService.createNewVersion(currentVersion);
        InstitutionalProposalDocument newInstitutionalProposalDocument = mergeProposals(newVersion, developmentProposal, budget);
        return newInstitutionalProposalDocument;
    }
    
    protected InstitutionalProposalDocument mergeProposals(InstitutionalProposal institutionalProposal, DevelopmentProposal developmentProposal, Budget budget)
        throws WorkflowException {
        
        InstitutionalProposalDocument institutionalProposalDocument = 
            (InstitutionalProposalDocument) documentService.getNewDocument(InstitutionalProposalDocument.class);
        
        institutionalProposalDocument.getDocumentHeader().setDocumentDescription(
                NEW_DOCUMENT_DESCRIPTION + developmentProposal.getProposalNumber());
        
        institutionalProposalDocument.setInstitutionalProposal(institutionalProposal);

        doBaseFieldsDataFeed(institutionalProposal, developmentProposal);
        doCustomAttributeDataFeed(institutionalProposalDocument, developmentProposal);
        
        institutionalProposal.getProjectPersons().clear();
        for (ProposalPerson pdPerson : developmentProposal.getProposalPersons()) {
            institutionalProposal.add(generateInstitutionalProposalPerson(pdPerson));
        }
        
        institutionalProposal.getSpecialReviews().clear();
        for (ProposalSpecialReview dpSpecialReview : developmentProposal.getPropSpecialReviews()) {
            institutionalProposal.addSpecialReview(generateIpSpecialReview(dpSpecialReview));
        }
        if (!institutionalProposal.getSpecialReviews().isEmpty()) {
            institutionalProposal.setSpecialReviewIndicator("1");
        }
        
        institutionalProposal.getInstitutionalProposalScienceKeywords().clear();
        for (PropScienceKeyword dpKeyword : developmentProposal.getPropScienceKeywords()) {
            institutionalProposal.addKeyword(dpKeyword.getScienceKeyword());
        }
        if (!institutionalProposal.getInstitutionalProposalScienceKeywords().isEmpty()) {
            institutionalProposal.setScienceCodeIndicator("1");
        }
        
        if (budget != null) {
            doBudgetDataFeed(institutionalProposal, budget);
        }
        institutionalProposal.refreshNonUpdateableReferences();
        
        return institutionalProposalDocument;
    }
    
    protected void doBaseFieldsDataFeed(InstitutionalProposal institutionalProposal, DevelopmentProposal developmentProposal) {
        institutionalProposal.setProposalTypeCode(Integer.parseInt(developmentProposal.getProposalTypeCode()));
        institutionalProposal.setActivityTypeCode(developmentProposal.getActivityTypeCode());
        if (developmentProposal.getProposalDocument().getDocumentHeader().getWorkflowDocument().isDisapproved()) {
            //if rejected set status code to WITHDRAWN
            institutionalProposal.setStatusCode(getWithdrawnStatusCode());
        } else {
            institutionalProposal.setStatusCode(getDefaultStatusCode());
        }
        institutionalProposal.setSponsorCode(developmentProposal.getSponsorCode());
        institutionalProposal.setTitle(developmentProposal.getTitle());
        institutionalProposal.setSubcontractFlag(developmentProposal.getSubcontracts());
        institutionalProposal.setRequestedStartDateTotal(developmentProposal.getRequestedStartDateInitial());
        institutionalProposal.setRequestedEndDateTotal(developmentProposal.getRequestedEndDateInitial());
        institutionalProposal.setDeadlineDate(developmentProposal.getDeadlineDate());
        institutionalProposal.setDeadlineTime(developmentProposal.getDeadlineTime());
        institutionalProposal.setNoticeOfOpportunityCode(developmentProposal.getNoticeOfOpportunityCode());
        institutionalProposal.setNumberOfCopies(developmentProposal.getNumberOfCopies());
        institutionalProposal.setDeadlineType(developmentProposal.getDeadlineType());
        institutionalProposal.setMailBy(developmentProposal.getMailBy());
        institutionalProposal.setMailType(developmentProposal.getMailType());
        institutionalProposal.setMailAccountNumber(developmentProposal.getMailAccountNumber());
        institutionalProposal.setMailDescription(developmentProposal.getMailDescription());
        institutionalProposal.setPrimeSponsorCode(developmentProposal.getPrimeSponsorCode());
        institutionalProposal.setCurrentAwardNumber(developmentProposal.getCurrentAwardNumber());
        institutionalProposal.setCfdaNumber(developmentProposal.getCfdaNumber());
        institutionalProposal.setNewDescription(developmentProposal.getNewDescription());
        institutionalProposal.setNoticeOfOpportunityCode(developmentProposal.getNoticeOfOpportunityCode());
        institutionalProposal.setNsfCode(developmentProposal.getNsfCode());
        institutionalProposal.setSponsorProposalNumber(developmentProposal.getSponsorProposalNumber());
        institutionalProposal.setOpportunity(developmentProposal.getProgramAnnouncementNumber());
        institutionalProposal.setCfdaNumber(developmentProposal.getCfdaNumber());
        institutionalProposal.setLeadUnitNumber(developmentProposal.getUnitNumber());
        institutionalProposal.setDefaultInitialContractAdmin();
        if (developmentProposal.getRolodex() != null) {
            institutionalProposal.setRolodexId(developmentProposal.getRolodex().getRolodexId());
        }
    }
    
    protected void doCustomAttributeDataFeed(InstitutionalProposalDocument institutionalProposalDocument, DevelopmentProposal developmentProposal) throws WorkflowException {
        Map<String, CustomAttributeDocument> dpCustomAttributes = developmentProposal.getProposalDocument().getCustomAttributeDocuments();
        Map<String, CustomAttributeDocument> ipCustomAttributes = institutionalProposalDocument.getCustomAttributeDocuments();
        List<InstitutionalProposalCustomData> ipCustomDataList = institutionalProposalDocument.getInstitutionalProposal().getInstitutionalProposalCustomDataList();
        InstitutionalProposalCustomData ipCustomData;
        CustomAttributeDocument dpCustomAttributeDocument;
        for (String key : dpCustomAttributes.keySet()) {
            if (ipCustomAttributes.containsKey(key)) {
                dpCustomAttributeDocument = dpCustomAttributes.get(key);
                ipCustomAttributes.put(key, dpCustomAttributeDocument);
                ipCustomData = new InstitutionalProposalCustomData();
                ipCustomData.setCustomAttribute(new CustomAttribute());
                ipCustomData.getCustomAttribute().setId(dpCustomAttributeDocument.getCustomAttributeId());
                ipCustomData.setCustomAttributeId((long) dpCustomAttributeDocument.getCustomAttributeId());
                ipCustomData.setInstitutionalProposal(institutionalProposalDocument.getInstitutionalProposal());
                ipCustomData.setValue(dpCustomAttributeDocument.getCustomAttribute().getValue());
                ipCustomDataList.add(ipCustomData);
            }
        }
    }

    
    protected InstitutionalProposalPerson generateInstitutionalProposalPerson(ProposalPerson pdPerson) {
        InstitutionalProposalPerson ipPerson = new InstitutionalProposalPerson();
        if (ObjectUtils.isNotNull(pdPerson.getPersonId())) {
            ipPerson.setPersonId(pdPerson.getPersonId());
        }
        if (ObjectUtils.isNotNull(pdPerson.getRolodexId())) {
            ipPerson.setRolodexId(pdPerson.getRolodexId());
        }
        ipPerson.setContactRoleCode(pdPerson.getRole().getRoleCode());
        for (ProposalPersonCreditSplit pdPersonCreditSplit : pdPerson.getCreditSplits()) {
            InstitutionalProposalPersonCreditSplit ipPersonCreditSplit = new InstitutionalProposalPersonCreditSplit();
            ipPersonCreditSplit.setCredit(pdPersonCreditSplit.getCredit());
            ipPersonCreditSplit.setInvCreditTypeCode(pdPersonCreditSplit.getInvCreditTypeCode());
            ipPersonCreditSplit.setNewCollectionRecord(pdPersonCreditSplit.isNewCollectionRecord());
            ipPerson.add(ipPersonCreditSplit);
        }
        //ipPerson.setEmailAddress(pdPerson.getEmailAddress());
        ipPerson.setFaculty(pdPerson.getFacultyFlag());
        ipPerson.setFullName(pdPerson.getFullName());
        ipPerson.setKeyPersonRole(pdPerson.getProjectRole());
        ipPerson.setNewCollectionRecord(pdPerson.isNewCollectionRecord());
        //ipPerson.setPerson(pdPerson.getPerson());
        //ipPerson.setPhoneNumber(pdPerson.getPhoneNumber());
        ipPerson.setRoleCode(pdPerson.getRole().getRoleCode());
        ipPerson.setTotalEffort(pdPerson.getPercentageEffort());
        ipPerson.setMultiplePi(pdPerson.isMultiplePi());
        for (ProposalPersonUnit pdPersonUnit : pdPerson.getUnits()) {
            InstitutionalProposalPersonUnit ipPersonUnit = new InstitutionalProposalPersonUnit();
            ipPersonUnit.setLeadUnit(pdPersonUnit.isLeadUnit());
            ipPersonUnit.setNewCollectionRecord(pdPersonUnit.isNewCollectionRecord());
            ipPersonUnit.setUnitNumber(pdPersonUnit.getUnitNumber());
            for (ProposalUnitCreditSplit pdPersonCreditSplit : pdPersonUnit.getCreditSplits()) {
                InstitutionalProposalPersonUnitCreditSplit ipPersonUnitCreditSplit = new InstitutionalProposalPersonUnitCreditSplit();
                ipPersonUnitCreditSplit.setCredit(pdPersonCreditSplit.getCredit());
                ipPersonUnitCreditSplit.setInvCreditTypeCode(pdPersonCreditSplit.getInvCreditTypeCode());
                ipPersonUnitCreditSplit.setNewCollectionRecord(pdPersonCreditSplit.isNewCollectionRecord());
                ipPersonUnit.add(ipPersonUnitCreditSplit);
            }
            ipPerson.add(ipPersonUnit);
        }
        
        return ipPerson;
    }
    
    protected InstitutionalProposalSpecialReview generateIpSpecialReview(ProposalSpecialReview dpSpecialReview) {
        InstitutionalProposalSpecialReview ipSpecialReview = new InstitutionalProposalSpecialReview();
        ipSpecialReview.setApplicationDate(dpSpecialReview.getApplicationDate());
        ipSpecialReview.setApprovalDate(dpSpecialReview.getApprovalDate());
        ipSpecialReview.setApprovalTypeCode(dpSpecialReview.getApprovalTypeCode());
        ipSpecialReview.setComments(dpSpecialReview.getComments());
        ipSpecialReview.setExpirationDate(dpSpecialReview.getExpirationDate());
        ipSpecialReview.setProtocolNumber(dpSpecialReview.getProtocolNumber());
        ipSpecialReview.setSpecialReviewType(dpSpecialReview.getSpecialReviewType());
        ipSpecialReview.setApprovalType(dpSpecialReview.getApprovalType());
        ipSpecialReview.setSpecialReviewTypeCode(dpSpecialReview.getSpecialReviewTypeCode());
        ipSpecialReview.setSpecialReviewNumber(dpSpecialReview.getSpecialReviewNumber());
        for (String dpExempt : dpSpecialReview.getExemptionTypeCodes()) {
            InstitutionalProposalSpecialReviewExemption newIpSpecialReviewExemption = ipSpecialReview.createSpecialReviewExemption(dpExempt);
            ipSpecialReview.getSpecialReviewExemptions().add(newIpSpecialReviewExemption);
            ipSpecialReview.getExemptionTypeCodes().add(dpExempt);
        }
        return ipSpecialReview;
    }
    
    protected void doBudgetDataFeed(InstitutionalProposal institutionalProposal, Budget budget) {
     // Base fields from Budget
        institutionalProposal.setRequestedStartDateInitial(budget.getBudgetPeriods().get(0).getStartDate());
        institutionalProposal.setRequestedEndDateInitial(budget.getBudgetPeriods().get(0).getEndDate());
        
        if (budget.getModularBudgetFlag()) {
            if (budget.getBudgetPeriod(0).getBudgetModular().getTotalDirectCost() != null) {
                institutionalProposal.setTotalDirectCostInitial(
                    new KualiDecimal(budget.getBudgetPeriod(0).getBudgetModular().getTotalDirectCost().bigDecimalValue()));
            }
            budget.getBudgetPeriod(0).getBudgetModular().calculateTotalFnaRequested();
            if (budget.getBudgetPeriod(0).getBudgetModular().getTotalFnaRequested() != null) {
                institutionalProposal.setTotalIndirectCostInitial(
                    new KualiDecimal(budget.getBudgetPeriod(0).getBudgetModular().getTotalFnaRequested().bigDecimalValue()));
            }
            
            BudgetDecimal totalDirect = new BudgetDecimal(0);
            BudgetDecimal totalIndirect = new BudgetDecimal(0);
            for (BudgetPeriod bp : budget.getBudgetPeriods()) {
                if (bp.getBudgetModular() != null) {
                    if (bp.getBudgetModular().getTotalDirectCost() != null) {
                        totalDirect = totalDirect.add(bp.getBudgetModular().getTotalDirectCost());
                    }
                    bp.getBudgetModular().calculateTotalFnaRequested();
                    if (bp.getBudgetModular().getTotalFnaRequested() != null) {
                        totalIndirect = totalIndirect.add(bp.getBudgetModular().getTotalFnaRequested());
                    }
                }
            }
            institutionalProposal.setTotalDirectCostTotal(new KualiDecimal(totalDirect.bigDecimalValue()));
            institutionalProposal.setTotalIndirectCostTotal(new KualiDecimal(totalIndirect.bigDecimalValue()));
            
        } else { 
            institutionalProposal.setTotalDirectCostInitial(new KualiDecimal(budget.getBudgetPeriod(0).getTotalDirectCost().bigDecimalValue()));
            institutionalProposal.setTotalIndirectCostInitial(new KualiDecimal(budget.getBudgetPeriod(0).getTotalIndirectCost().bigDecimalValue()));
            institutionalProposal.setTotalDirectCostTotal(new KualiDecimal(budget.getTotalDirectCost().bigDecimalValue()));
            institutionalProposal.setTotalIndirectCostTotal(new KualiDecimal(budget.getTotalIndirectCost().bigDecimalValue()));
        }
        
        // Cost Shares (from Budget)
        institutionalProposal.getInstitutionalProposalCostShares().clear();
        for (BudgetCostShare budgetCostShare : budget.getBudgetCostShares()) {
            InstitutionalProposalCostShare ipCostShare = new InstitutionalProposalCostShare();
            ipCostShare.setCostShareTypeCode(getDefaultCostShareTypeCode());
            ipCostShare.setAmount(new KualiDecimal(budgetCostShare.getShareAmount().bigDecimalValue()));
            ipCostShare.setCostSharePercentage(new KualiDecimal(budgetCostShare.getSharePercentage().bigDecimalValue()));
            ipCostShare.setProjectPeriod(budgetCostShare.getProjectPeriod().toString());
            ipCostShare.setSourceAccount(budgetCostShare.getSourceAccount());
            institutionalProposal.add(ipCostShare);
        }
        if (!institutionalProposal.getInstitutionalProposalCostShares().isEmpty()) {
            institutionalProposal.setCostSharingIndicator("1");
        }
        
        // Unrecovered F and As (from Budget)
        institutionalProposal.getInstitutionalProposalUnrecoveredFandAs().clear();
        for (BudgetUnrecoveredFandA budgetUfa : budget.getBudgetUnrecoveredFandAs()) {
            InstitutionalProposalUnrecoveredFandA ipUfa = new InstitutionalProposalUnrecoveredFandA();
            ipUfa.setApplicableIndirectcostRate(new KualiDecimal(budgetUfa.getApplicableRate().bigDecimalValue()));
            ipUfa.setFiscalYear(budgetUfa.getFiscalYear().toString());
            ipUfa.setOnCampusFlag("Y".equals(budgetUfa.getOnCampusFlag()) ? true : false);
            ipUfa.setSourceAccount(budgetUfa.getSourceAccount());
            ipUfa.setIndirectcostRateTypeCode(Integer.parseInt(budget.getOhRateClassCode()));
            ipUfa.setUnderrecoveryOfIndirectcost(new KualiDecimal(budgetUfa.getAmount().bigDecimalValue()));
            institutionalProposal.add(ipUfa);
        }
        if (!institutionalProposal.getInstitutionalProposalUnrecoveredFandAs().isEmpty()) {
            institutionalProposal.setIdcRateIndicator("1");
        }
    }
    
    protected Integer getDefaultStatusCode() {
        return 1;
    }
    
    protected Integer getWithdrawnStatusCode() {
        return 5;
    }
    
    protected Integer getDefaultCostShareTypeCode() {
        return 1;
    }
    
    
    /**
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#createAndSaveNewVersion(org.kuali.kra.institutionalproposal.home.InstitutionalProposal, org.kuali.kra.institutionalproposal.document.InstitutionalProposalDocument)
     */
    public InstitutionalProposalDocument createAndSaveNewVersion(InstitutionalProposal currentInstitutionalProposal, 
            InstitutionalProposalDocument currentInstitutionalProposalDocument) throws VersionException, 
            WorkflowException, IOException{
        InstitutionalProposal newVersion = getVersioningService().createNewVersion(currentInstitutionalProposal);
        
        synchNewCustomAttributes(newVersion, currentInstitutionalProposal);
        
        newVersion.setProposalSequenceStatus(VersionStatus.PENDING.toString());
        newVersion.setAwardFundingProposals(transferFundingProposals(currentInstitutionalProposal, newVersion));
        InstitutionalProposalDocument newInstitutionalProposalDocument = 
            (InstitutionalProposalDocument) getDocumentService().getNewDocument(InstitutionalProposalDocument.class);
        newInstitutionalProposalDocument.getDocumentHeader().setDocumentDescription(currentInstitutionalProposalDocument.getDocumentHeader().getDocumentDescription());
        newInstitutionalProposalDocument.setInstitutionalProposal(newVersion);
        getDocumentService().saveDocument(newInstitutionalProposalDocument);
        return newInstitutionalProposalDocument;
    }
    
    /**
     * This method is to synch custom attributes. During version process only existing custom attributes
     * available in the old document is copied. We need to make sure we have all the latest custom attributes
     * tied to the new document.
     * @param newInstitutionalProposal
     * @param oldInstitutionalProposal
     */
    protected void synchNewCustomAttributes(InstitutionalProposal newInstitutionalProposal, InstitutionalProposal oldInstitutionalProposal) {
        Set<Integer> availableCustomAttributes = new HashSet<Integer>();
        for(InstitutionalProposalCustomData customData : newInstitutionalProposal.getInstitutionalProposalCustomDataList()) {
            availableCustomAttributes.add(customData.getCustomAttributeId().intValue());
        }
        
        if(oldInstitutionalProposal.getInstitutionalProposalDocument() != null) {
            Map<String, CustomAttributeDocument> customAttributeDocuments = oldInstitutionalProposal.getInstitutionalProposalDocument().getCustomAttributeDocuments();
            for (Map.Entry<String, CustomAttributeDocument> entry : customAttributeDocuments.entrySet()) {
                CustomAttributeDocument customAttributeDocument = entry.getValue();
                if(!availableCustomAttributes.contains(customAttributeDocument.getCustomAttributeId())) {
                    InstitutionalProposalCustomData customData = new InstitutionalProposalCustomData();
                    customData.setCustomAttributeId((long) customAttributeDocument.getCustomAttributeId());
                    customData.setCustomAttribute(customAttributeDocument.getCustomAttribute());
                    customData.setValue("");
                    customData.setInstitutionalProposal(newInstitutionalProposal);
                    newInstitutionalProposal.getInstitutionalProposalCustomDataList().add(customData);
                }
            }
        }
    }
    
    protected ArrayList<AwardFundingProposal> transferFundingProposals(InstitutionalProposal oldIP, InstitutionalProposal newIP) {
        ArrayList<AwardFundingProposal> newFundingProposals = new ArrayList<AwardFundingProposal>();
        for (AwardFundingProposal afpp:oldIP.getAwardFundingProposals()) {
            newFundingProposals.add(new AwardFundingProposal(afpp.getAward(), newIP));
            afpp.setActive(false);
        }
        getBusinessObjectService().save(oldIP.getAwardFundingProposals());
        return newFundingProposals;
    }
    
    /* Service injection getters and setters */
    
    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
    
    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setVersioningService(VersioningService versioningService) {
        this.versioningService = versioningService;
    }

    public void setInstitutionalProposalVersioningService(InstitutionalProposalVersioningService institutionalProposalVersioningService) {
        this.institutionalProposalVersioningService = institutionalProposalVersioningService;
    }
    
    /**
     * Set the Sequence Accessor Service.
     * @param sequenceAccessorService the Sequence Accessor Service
     */
    public void setSequenceAccessorService(SequenceAccessorService sequenceAccessorService) {
        this.sequenceAccessorService = sequenceAccessorService;
    }

    /**
     * @see org.kuali.kra.institutionalproposal.service.InstitutionalProposalService#getValidFundingProposalStatusCodes()
     */
    @Override
    public Collection<String> getValidFundingProposalStatusCodes() {
        String value = getParameterService().getParameterValueAsString(InstitutionalProposalDocument.class, "validFundingProposalStatusCodes");
        return Arrays.asList(value.split(","));
    }

    protected ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public DocumentService getDocumentService() {
        return documentService;
    }

    public VersioningService getVersioningService() {
        return versioningService;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }
    
}
