/*
 * Decompiled with CFR 0.152.
 */
package org.kurento.room;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.PreDestroy;
import org.kurento.client.IceCandidate;
import org.kurento.client.KurentoClient;
import org.kurento.client.MediaElement;
import org.kurento.client.MediaPipeline;
import org.kurento.room.api.KurentoClientProvider;
import org.kurento.room.api.RoomEventHandler;
import org.kurento.room.api.UserNotificationService;
import org.kurento.room.api.pojo.ParticipantRequest;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.exception.AdminException;
import org.kurento.room.exception.RoomException;
import org.kurento.room.internal.DefaultRoomEventHandler;
import org.kurento.room.internal.Participant;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RoomManager {
    private final Logger log = LoggerFactory.getLogger(RoomManager.class);
    private RoomEventHandler roomEventHandler;
    private KurentoClientProvider kcProvider;
    private final ConcurrentMap<String, Room> rooms = new ConcurrentHashMap<String, Room>();

    public RoomManager(UserNotificationService notificationService, KurentoClientProvider kcProvider) {
        this.roomEventHandler = new DefaultRoomEventHandler(notificationService);
        this.kcProvider = kcProvider;
    }

    public RoomManager(RoomEventHandler roomEventHandler, KurentoClientProvider kcProvider) {
        this.roomEventHandler = roomEventHandler;
        this.kcProvider = kcProvider;
    }

    public void joinRoom(String userName, String roomName, ParticipantRequest request) {
        this.log.debug("Request [JOIN_ROOM] user={}, room={} ({})", new Object[]{userName, roomName, request});
        try {
            Room room = this.getRoom(roomName, request, true);
            if (room.isClosed()) {
                this.log.error("Trying to join room {} but it is closing", (Object)room.getName());
                throw new RoomException(RoomException.Code.ROOM_CLOSED_ERROR_CODE, "Trying to join room '" + room.getName() + "' but it is closing");
            }
            Set<UserParticipant> existingParticipants = this.getParticipants(roomName);
            room.join(request.getParticipantId(), userName);
            this.roomEventHandler.onParticipantJoined(request, roomName, userName, existingParticipants, null);
        }
        catch (RoomException e) {
            this.log.warn("PARTICIPANT {}: Error joining/creating room {}", new Object[]{userName, roomName, e});
            this.roomEventHandler.onParticipantJoined(request, roomName, userName, null, e);
        }
    }

    public void leaveRoom(ParticipantRequest request) {
        this.log.debug("Request [LEAVE_ROOM] ({})", (Object)request);
        try {
            Participant participant = this.getParticipant(request.getParticipantId());
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            Room room = participant.getRoom();
            String roomName = room.getName();
            if (!room.isClosed()) {
                room.leave(request.getParticipantId());
                Set<String> remainingParticipantIds = room.getParticipantIds();
                if (remainingParticipantIds.isEmpty()) {
                    room.close();
                    this.rooms.remove(room.getName());
                    this.log.warn("Room '{}' removed and closed", (Object)roomName);
                }
                this.roomEventHandler.onParticipantLeft(request, roomName, participant.getName(), remainingParticipantIds, null);
            } else {
                this.log.warn("Trying to leave from room '{}' but it is closing", (Object)room.getName());
            }
        }
        catch (RoomException e) {
            this.log.warn("Error leaving room", (Throwable)e);
            this.roomEventHandler.onParticipantLeft(request, null, null, null, e);
        }
    }

    public void publishMedia(ParticipantRequest request, String sdpOffer, boolean doLoopback, MediaElement ... mediaElements) {
        this.log.debug("Request [PUBLISH_MEDIA] sdpOffer={} dooLoopback={} mediaElements={} ({})", new Object[]{sdpOffer, doLoopback, mediaElements, request});
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            String name = participant.getName();
            Room room = participant.getRoom();
            participant.createPublishingEndpoint();
            for (MediaElement elem : mediaElements) {
                participant.getPublisher().apply(elem);
            }
            String sdpAnswer = participant.publishToRoom(sdpOffer, doLoopback);
            if (sdpAnswer == null) {
                throw new RoomException(RoomException.Code.SDP_ERROR_CODE, "Error generating SDP answer for publishing user " + name);
            }
            this.roomEventHandler.onPublishMedia(request, name, sdpAnswer, room.getParticipantIds(), null);
            room.newPublisher(participant);
        }
        catch (RoomException e) {
            this.log.warn("Error publishing media", (Throwable)e);
            this.roomEventHandler.onPublishMedia(request, null, null, null, e);
        }
    }

    public void unpublishMedia(ParticipantRequest request) {
        this.log.debug("Request [UNPUBLISH_MEDIA] ({})", (Object)request);
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            if (!participant.isStreaming()) {
                throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Participant with id '" + request.getParticipantId() + "' is not streaming media");
            }
            String name = participant.getName();
            Room room = participant.getRoom();
            participant.unpublishMedia();
            room.cancelPublisher(participant);
            this.roomEventHandler.onUnpublishMedia(request, name, room.getParticipantIds(), null);
        }
        catch (RoomException e) {
            this.log.warn("Error unpublishing media", (Throwable)e);
            this.roomEventHandler.onUnpublishMedia(request, null, null, e);
        }
    }

    public void subscribe(String remoteName, String sdpOffer, ParticipantRequest request) {
        this.log.debug("Request [SUBSCRIBE] remoteParticipant={} sdpOffer={} ({})", new Object[]{remoteName, sdpOffer, request});
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            String name = participant.getName();
            Room room = participant.getRoom();
            Participant senderParticipant = room.getParticipantByName(remoteName);
            if (senderParticipant == null) {
                this.log.warn("PARTICIPANT {}: Requesting to recv media from user {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
            }
            String sdpAnswer = participant.receiveMediaFrom(senderParticipant, sdpOffer);
            if (sdpAnswer == null) {
                throw new RoomException(RoomException.Code.SDP_ERROR_CODE, "Error generating SDP answer for receiving user " + name + " from " + remoteName);
            }
            this.roomEventHandler.onSubscribe(request, sdpAnswer, null);
        }
        catch (RoomException e) {
            this.log.warn("Error subscribing to {}", (Object)remoteName, (Object)e);
            this.roomEventHandler.onSubscribe(request, null, e);
        }
    }

    public void unsubscribe(String remoteName, ParticipantRequest request) {
        this.log.debug("Request [UNSUBSCRIBE] remoteParticipant={} ({})", (Object)remoteName, (Object)request);
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            String name = participant.getName();
            Room room = participant.getRoom();
            Participant senderParticipant = room.getParticipantByName(remoteName);
            if (senderParticipant == null) {
                this.log.warn("PARTICIPANT {}: Requesting to unsubscribe from user {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
            }
            participant.cancelReceivingMedia(remoteName);
            this.roomEventHandler.onUnsubscribe(request, null);
        }
        catch (RoomException e) {
            this.log.warn("Error unsubscribing from {}", (Object)remoteName, (Object)e);
            this.roomEventHandler.onUnsubscribe(request, e);
        }
    }

    public void sendMessage(String message, String userName, String roomName, ParticipantRequest request) {
        this.log.debug("Request [SEND_MESSAGE] message={} ({})", (Object)message, (Object)request);
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            String name = participant.getName();
            if (!name.equals(userName)) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "Provided username '" + userName + "' differs from the participant's name");
            }
            Room room = participant.getRoom();
            if (!room.getName().equals(roomName)) {
                throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Provided room name '" + roomName + "' differs from the participant's room");
            }
            this.roomEventHandler.onSendMessage(request, message, userName, roomName, room.getParticipantIds(), null);
        }
        catch (RoomException e) {
            this.log.warn("Error sending message", (Throwable)e);
            this.roomEventHandler.onSendMessage(request, null, null, null, null, e);
        }
    }

    public void onIceCandidate(String endpointName, String candidate, int sdpMLineIndex, String sdpMid, ParticipantRequest request) {
        this.log.debug("Request [ICE_CANDIDATE] endpoint={} candidate={} sdpMLineIdx={} sdpMid={} ({})", new Object[]{endpointName, candidate, sdpMLineIndex, sdpMid, request});
        try {
            Participant participant = this.getParticipant(request);
            if (participant == null) {
                throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + request.getParticipantId() + "' was found");
            }
            participant.addIceCandidate(endpointName, new IceCandidate(candidate, sdpMid, sdpMLineIndex));
            this.roomEventHandler.onRecvIceCandidate(request, null);
        }
        catch (RoomException e) {
            this.log.warn("Error receiving ICE candidate", (Throwable)e);
            this.roomEventHandler.onRecvIceCandidate(request, e);
        }
    }

    @PreDestroy
    public void close() {
        for (String roomName : this.rooms.keySet()) {
            try {
                this.closeRoom(roomName);
            }
            catch (Exception e) {
                this.log.warn("Error closing room '{}'", (Object)roomName, (Object)e);
            }
        }
    }

    public void evictParticipant(String participantId) throws AdminException {
        try {
            Participant participant = this.getParticipant(participantId);
            if (participant == null) {
                throw new AdminException("No participant with id '" + participantId + "' was found");
            }
            Room room = participant.getRoom();
            String roomName = room.getName();
            if (!room.isClosed()) {
                room.leave(participantId);
                Set<String> remainingParticipantIds = room.getParticipantIds();
                if (remainingParticipantIds.isEmpty()) {
                    room.close();
                    this.rooms.remove(room.getName());
                    this.log.warn("Room '{}' removed and closed", (Object)roomName);
                    this.roomEventHandler.onRoomClosed(roomName, remainingParticipantIds);
                } else {
                    this.roomEventHandler.onParticipantLeft(null, roomName, participant.getName(), remainingParticipantIds, null);
                }
            } else {
                this.log.warn("Trying to evict participant with id '{}' from room '{}' but it is closing", (Object)participantId, (Object)room.getName());
            }
        }
        catch (RoomException e) {
            this.log.warn("Error evicting participant with id {}", (Object)participantId, (Object)e);
            throw new AdminException("Unable to evict participant with id " + participantId + ": " + e.getMessage());
        }
    }

    public Set<String> getRooms() {
        return new HashSet<String>(this.rooms.keySet());
    }

    public Set<UserParticipant> getParticipants(String roomName) throws AdminException {
        Room room = (Room)this.rooms.get(roomName);
        if (room == null) {
            throw new AdminException("Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = room.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPublishers(String roomName) throws AdminException {
        Room r = (Room)this.rooms.get(roomName);
        if (r == null) {
            throw new AdminException("Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = r.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed() || !p.isStreaming()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), true));
        }
        return userParts;
    }

    public Set<UserParticipant> getSubscribers(String roomName) throws AdminException {
        Room r = (Room)this.rooms.get(roomName);
        if (r == null) {
            throw new AdminException("Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = r.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed() || !p.isSubscribed()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPeerPublishers(String participantId) throws AdminException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new AdminException("No participant with id '" + participantId + "' was found");
        }
        Set<String> subscribedEndpoints = participant.getConnectedSubscribedEndpoints();
        Room room = participant.getRoom();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (String epName : subscribedEndpoints) {
            Participant p = room.getParticipantByName(epName);
            userParts.add(new UserParticipant(p.getId(), p.getName()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPeerSubscribers(String participantId) throws AdminException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new AdminException("No participant with id '" + participantId + "' was found");
        }
        if (!participant.isStreaming()) {
            throw new AdminException("Participant with id '" + participantId + "' is not a publisher yet");
        }
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        Room room = participant.getRoom();
        String endpointName = participant.getName();
        for (Participant p : room.getParticipants()) {
            Set<String> subscribedEndpoints;
            if (p.equals(participant) || !(subscribedEndpoints = p.getConnectedSubscribedEndpoints()).contains(endpointName)) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName()));
        }
        return userParts;
    }

    public boolean createRoom(String roomName) throws AdminException {
        Room room = (Room)this.rooms.get(roomName);
        try {
            if (room != null) {
                this.log.info("Room '{}' already exists");
                return false;
            }
            KurentoClient kurentoClient = this.kcProvider.getKurentoClient(null);
            room = new Room(roomName, kurentoClient, this.roomEventHandler);
            Room oldRoom = this.rooms.putIfAbsent(roomName, room);
            if (oldRoom != null) {
                this.log.info("Room '{}' has just been created by another thread");
                return false;
            }
            this.log.warn("No room '{}' exists yet. Created one using KurentoClient '{}')", (Object)roomName, (Object)kurentoClient.getServerManager().getName());
            return true;
        }
        catch (RoomException e) {
            this.log.warn("Error creating room {}", (Object)roomName, (Object)e);
            throw new AdminException("Error creating room - " + e.toString());
        }
    }

    public MediaPipeline getPipeline(String participantId) throws AdminException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new AdminException("No participant with id '" + participantId + "' was found");
        }
        return participant.getPipeline();
    }

    public void closeRoom(String roomName) throws AdminException {
        Room room = (Room)this.rooms.get(roomName);
        if (room == null) {
            throw new AdminException("Room '" + roomName + "' not found");
        }
        if (room.isClosed()) {
            throw new AdminException("Room '" + roomName + "' already closed");
        }
        HashSet<String> pids = new HashSet<String>(room.getParticipantIds());
        for (String pid : pids) {
            try {
                room.leave(pid);
            }
            catch (RoomException e) {
                this.log.warn("Error evicting participant with id '{}' from room '{}'", new Object[]{pid, roomName, e});
            }
        }
        this.roomEventHandler.onRoomClosed(roomName, pids);
        room.close();
        this.rooms.remove(roomName);
        this.log.warn("Room '{}' removed and closed", (Object)roomName);
    }

    public void addMediaElement(String participantId, MediaElement element) throws AdminException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new AdminException("No participant with id '" + participantId + "' was found");
        }
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new AdminException("Participant '" + name + "' has been closed");
        }
        try {
            participant.getPublisher().apply(element);
        }
        catch (RoomException e) {
            throw new AdminException("Error connecting media element - " + e.toString());
        }
    }

    public void removeMediaElement(String participantId, MediaElement element) throws AdminException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new AdminException("No participant with id '" + participantId + "' was found");
        }
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new AdminException("Participant '" + name + "' has been closed");
        }
        try {
            participant.getPublisher().revert(element);
        }
        catch (RoomException e) {
            throw new AdminException("Error disconnecting media element - " + e.toString());
        }
    }

    private Room getRoom(String roomName, ParticipantRequest request, boolean getOrCreate) {
        if (roomName == null || roomName.trim().isEmpty()) {
            throw new RoomException(RoomException.Code.CANNOT_CREATE_ROOM_ERROR_CODE, "Empty room name is not allowed");
        }
        Room room = (Room)this.rooms.get(roomName);
        if (room == null && getOrCreate) {
            KurentoClient kurentoClient = this.kcProvider.getKurentoClient(request.getParticipantId());
            if (kurentoClient == null) {
                throw new RoomException(RoomException.Code.CANNOT_CREATE_ROOM_ERROR_CODE, "Unable to obtain a KurentoClient instance");
            }
            room = new Room(roomName, kurentoClient, this.roomEventHandler);
            Room oldRoom = this.rooms.putIfAbsent(roomName, room);
            if (oldRoom != null) {
                return oldRoom;
            }
            this.log.warn("Created room '{}' using the provided KurentoClient", (Object)roomName);
            this.roomEventHandler.onRoomCreated(request, roomName);
        }
        return room;
    }

    private Participant getParticipant(ParticipantRequest request) {
        return this.getParticipant(request.getParticipantId());
    }

    private Participant getParticipant(String pid) {
        for (Room r : this.rooms.values()) {
            if (r.isClosed() || !r.getParticipantIds().contains(pid)) continue;
            return r.getParticipant(pid);
        }
        return null;
    }
}

