/*
 * Decompiled with CFR 0.152.
 */
package org.kurento.room.internal;

import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.kurento.client.Continuation;
import org.kurento.client.ErrorEvent;
import org.kurento.client.IceCandidate;
import org.kurento.client.MediaElement;
import org.kurento.client.MediaPipeline;
import org.kurento.client.WebRtcEndpoint;
import org.kurento.client.internal.server.KurentoServerException;
import org.kurento.room.endpoint.PublisherEndpoint;
import org.kurento.room.endpoint.SubscriberEndpoint;
import org.kurento.room.exception.RoomException;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Participant {
    private static final Logger log = LoggerFactory.getLogger(Participant.class);
    private String id;
    private String name;
    private final Room room;
    private final MediaPipeline pipeline;
    private PublisherEndpoint publisher;
    private CountDownLatch endPointLatch = new CountDownLatch(1);
    private final ConcurrentMap<String, SubscriberEndpoint> subscribers = new ConcurrentHashMap<String, SubscriberEndpoint>();
    private volatile boolean streaming = false;
    private volatile boolean closed;

    public Participant(String id, String name, Room room, MediaPipeline pipeline) {
        this.id = id;
        this.name = name;
        this.pipeline = pipeline;
        this.room = room;
        this.publisher = new PublisherEndpoint(this, name, pipeline);
        for (Participant other : room.getParticipants()) {
            if (other.getName().equals(this.name)) continue;
            this.subscribers.put(other.getName(), new SubscriberEndpoint(this, other.getName(), pipeline));
        }
    }

    public void createPublishingEndpoint() {
        this.publisher.createEndpoint(this.endPointLatch);
        if (this.getPublisher().getEndpoint() == null) {
            throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Unable to create publisher endpoint");
        }
    }

    public String getId() {
        return this.id;
    }

    public String getName() {
        return this.name;
    }

    public PublisherEndpoint getPublisher() {
        try {
            if (!this.endPointLatch.await(30L, TimeUnit.SECONDS)) {
                throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Timeout reached while waiting for publisher endpoint to be ready");
            }
        }
        catch (InterruptedException e) {
            throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Interrupted while waiting for publisher endpoint to be ready: " + e.getMessage());
        }
        return this.publisher;
    }

    public Room getRoom() {
        return this.room;
    }

    public MediaPipeline getPipeline() {
        return this.pipeline;
    }

    public boolean isClosed() {
        return this.closed;
    }

    public boolean isStreaming() {
        return this.streaming;
    }

    public boolean isSubscribed() {
        for (SubscriberEndpoint se : this.subscribers.values()) {
            if (!se.isConnectedToPublisher()) continue;
            return true;
        }
        return false;
    }

    public Set<String> getConnectedSubscribedEndpoints() {
        HashSet<String> subscribedToSet = new HashSet<String>();
        for (SubscriberEndpoint se : this.subscribers.values()) {
            if (!se.isConnectedToPublisher()) continue;
            subscribedToSet.add(se.getEndpointName());
        }
        return subscribedToSet;
    }

    public String publishToRoom(String sdpOffer, boolean doLoopback) {
        log.info("USER {}: Request to publish video in room {}", (Object)this.name, (Object)this.room.getName());
        log.trace("USER {}: Publishing SdpOffer is {}", (Object)this.name, (Object)sdpOffer);
        String sdpAnswer = this.getPublisher().publish(sdpOffer, doLoopback);
        this.streaming = true;
        log.trace("USER {}: Publishing SdpAnswer is {}", (Object)this.name, (Object)sdpAnswer);
        log.info("USER {}: Is now publishing video in room {}", (Object)this.name, (Object)this.room.getName());
        return sdpAnswer;
    }

    public void unpublishMedia() {
        log.debug("PARTICIPANT {}: unpublishing media stream from room {}", (Object)this.name, (Object)this.room.getName());
        this.releasePublisherEndpoint();
        this.publisher = new PublisherEndpoint(this, this.name, this.pipeline);
        log.debug("PARTICIPANT {}: released publisher endpoint and left it initialized (ready for future streaming)", (Object)this.name);
    }

    public String receiveMediaFrom(Participant sender, String sdpOffer) {
        String senderName = sender.getName();
        log.info("USER {}: Request to receive media from {} in room {}", new Object[]{this.name, senderName, this.room.getName()});
        log.trace("USER {}: SdpOffer for {} is {}", new Object[]{this.name, senderName, sdpOffer});
        if (senderName.equals(this.name)) {
            log.warn("PARTICIPANT {}: trying to configure loopback by subscribing", (Object)this.name);
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Can loopback only when publishing media");
        }
        if (sender.getPublisher() == null) {
            log.warn("PARTICIPANT {}: Trying to connect to a user without a publishing endpoint", (Object)this.name);
            return null;
        }
        log.debug("PARTICIPANT {}: Creating a subscriber endpoint to user {}", (Object)this.name, (Object)senderName);
        SubscriberEndpoint subscriber = new SubscriberEndpoint(this, senderName, this.pipeline);
        SubscriberEndpoint oldSubscriber = this.subscribers.putIfAbsent(senderName, subscriber);
        if (oldSubscriber != null) {
            subscriber = oldSubscriber;
        }
        try {
            CountDownLatch subscriberLatch = new CountDownLatch(1);
            WebRtcEndpoint oldWrEndpoint = subscriber.createEndpoint(subscriberLatch);
            try {
                if (!subscriberLatch.await(30L, TimeUnit.SECONDS)) {
                    throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Timeout reached when creating subscriber endpoint");
                }
            }
            catch (InterruptedException e) {
                throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Interrupted when creating subscriber endpoint: " + e.getMessage());
            }
            if (oldWrEndpoint != null) {
                log.warn("PARTICIPANT {}: Two threads are trying to create at the same time a subscriber endpoint for user {}", (Object)this.name, (Object)senderName);
                return null;
            }
            if (subscriber.getEndpoint() == null) {
                throw new RoomException(RoomException.Code.WEBRTC_ENDPOINT_ERROR_CODE, "Unable to create subscriber endpoint");
            }
        }
        catch (RoomException e) {
            this.subscribers.remove(senderName);
            throw e;
        }
        log.debug("PARTICIPANT {}: Created subscriber endpoint for user {}", (Object)this.name, (Object)senderName);
        try {
            String sdpAnswer = subscriber.subscribe(sdpOffer, sender.getPublisher());
            log.trace("USER {}: Subscribing SdpAnswer is {}", (Object)this.name, (Object)sdpAnswer);
            log.info("USER {}: Is now receiving video from {} in room {}", new Object[]{this.name, senderName, this.room.getName()});
            return sdpAnswer;
        }
        catch (KurentoServerException e) {
            if (e.getCode() == 40101) {
                log.warn("Publisher endpoint was already released when trying to connect a subscriber endpoint to it", (Throwable)e);
            } else {
                log.error("Exception connecting subscriber endpoint to publisher endpoint", (Throwable)e);
            }
            this.subscribers.remove(senderName);
            this.releaseSubscriberEndpoint(senderName, subscriber);
            return null;
        }
    }

    public void cancelReceivingMedia(String senderName) {
        log.debug("PARTICIPANT {}: cancel receiving media from {}", (Object)this.name, (Object)senderName);
        SubscriberEndpoint subscriberEndpoint = (SubscriberEndpoint)this.subscribers.remove(senderName);
        if (subscriberEndpoint == null || subscriberEndpoint.getEndpoint() == null) {
            log.warn("PARTICIPANT {}: Trying to cancel receiving video from user {}. But there is no such subscriber endpoint.", (Object)this.name, (Object)senderName);
        } else {
            log.debug("PARTICIPANT {}: Cancel subscriber endpoint linked to user {}", (Object)this.name, (Object)senderName);
            this.releaseSubscriberEndpoint(senderName, subscriberEndpoint);
        }
    }

    public void close() {
        log.debug("PARTICIPANT {}: Closing user", (Object)this.name);
        this.closed = true;
        for (String remoteParticipantName : this.subscribers.keySet()) {
            SubscriberEndpoint subscriber = (SubscriberEndpoint)this.subscribers.get(remoteParticipantName);
            if (subscriber.getEndpoint() != null) {
                this.releaseSubscriberEndpoint(remoteParticipantName, subscriber);
                log.debug("PARTICIPANT {}: Released subscriber endpoint to {}", (Object)this.name, (Object)remoteParticipantName);
                continue;
            }
            log.warn("PARTICIPANT {}: Trying to close subscriber endpoint to {}. But the endpoint was never instantiated.", (Object)this.name, (Object)remoteParticipantName);
        }
        this.releasePublisherEndpoint();
    }

    public SubscriberEndpoint addSubscriber(String newUserName) {
        SubscriberEndpoint iceSendingEndpoint = new SubscriberEndpoint(this, newUserName, this.pipeline);
        SubscriberEndpoint existingIceSendingEndpoint = this.subscribers.putIfAbsent(newUserName, iceSendingEndpoint);
        if (existingIceSendingEndpoint != null) {
            iceSendingEndpoint = existingIceSendingEndpoint;
            log.trace("PARTICIPANT {}: There is an existing placeholder for WebRtcEndpoint with ICE candidates queue for user {}", (Object)this.name, (Object)newUserName);
        } else {
            log.debug("PARTICIPANT {}: New placeholder for WebRtcEndpoint with ICE candidates queue for user {}", (Object)this.name, (Object)newUserName);
        }
        return iceSendingEndpoint;
    }

    public void addIceCandidate(String endpointName, IceCandidate iceCandidate) {
        if (this.name.equals(endpointName)) {
            this.publisher.addIceCandidate(iceCandidate);
        } else {
            this.addSubscriber(endpointName).addIceCandidate(iceCandidate);
        }
    }

    public void sendIceCandidate(String endpointName, IceCandidate candidate) {
        this.room.sendIceCandidate(this.id, endpointName, candidate);
    }

    public void sendMediaError(ErrorEvent event) {
        String desc = event.getType() + ": " + event.getDescription() + "(errCode=" + event.getErrorCode() + ")";
        log.warn("PARTICIPANT {}: Media error encountered: {}", (Object)this.name, (Object)desc);
        this.room.sendMediaError(this.id, desc);
    }

    private void releasePublisherEndpoint() {
        if (this.publisher != null && this.publisher.getEndpoint() != null) {
            this.streaming = false;
            this.publisher.unregisterErrorListeners();
            for (MediaElement el : this.publisher.getMediaElements()) {
                this.releaseElement(this.name, el);
            }
            this.releaseElement(this.name, (MediaElement)this.publisher.getEndpoint());
            this.publisher = null;
        } else {
            log.warn("PARTICIPANT {}: Trying to release publisher endpoint but is null", (Object)this.name);
        }
    }

    private void releaseSubscriberEndpoint(String senderName, SubscriberEndpoint subscriber) {
        if (subscriber != null) {
            subscriber.unregisterErrorListeners();
            this.releaseElement(senderName, (MediaElement)subscriber.getEndpoint());
        } else {
            log.warn("PARTICIPANT {}: Trying to release subscriber endpoint for '{}' but is null", (Object)this.name, (Object)senderName);
        }
    }

    private void releaseElement(final String senderName, MediaElement element) {
        final String eid = element.getId();
        try {
            element.release((Continuation)new Continuation<Void>(){

                public void onSuccess(Void result) throws Exception {
                    log.debug("PARTICIPANT {}: Released successfully media element #{} for {}", new Object[]{Participant.this.name, eid, senderName});
                }

                public void onError(Throwable cause) throws Exception {
                    log.warn("PARTICIPANT {}: Could not release media element #{} for {}", new Object[]{Participant.this.name, eid, senderName, cause});
                }
            });
        }
        catch (Exception e) {
            log.error("PARTICIPANT {}: Error calling release on elem #{} for {}", new Object[]{this.name, eid, senderName, e});
        }
    }

    public String toString() {
        return "[User: " + this.name + "]";
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + (this.id == null ? 0 : this.id.hashCode());
        result = 31 * result + (this.name == null ? 0 : this.name.hashCode());
        return result;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Participant)) {
            return false;
        }
        Participant other = (Participant)obj;
        if (this.id == null ? other.id != null : !this.id.equals(other.id)) {
            return false;
        }
        return !(this.name == null ? other.name != null : !this.name.equals(other.name));
    }
}

