/*
 * Decompiled with CFR 0.152.
 */
package org.kurento.room.internal;

import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.kurento.client.Continuation;
import org.kurento.client.ErrorEvent;
import org.kurento.client.EventListener;
import org.kurento.client.IceCandidate;
import org.kurento.client.KurentoClient;
import org.kurento.client.MediaPipeline;
import org.kurento.room.api.RoomHandler;
import org.kurento.room.exception.RoomException;
import org.kurento.room.internal.Participant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Room {
    public static final int ASYNC_LATCH_TIMEOUT = 30;
    private static final Logger log = LoggerFactory.getLogger(Room.class);
    private final ConcurrentMap<String, Participant> participants = new ConcurrentHashMap<String, Participant>();
    private final String name;
    private MediaPipeline pipeline;
    private CountDownLatch pipelineLatch = new CountDownLatch(1);
    private KurentoClient kurentoClient;
    private RoomHandler roomHandler;
    private volatile boolean closed = false;
    private AtomicInteger activePublishers = new AtomicInteger(0);

    public Room(String roomName, KurentoClient kurentoClient, RoomHandler roomHandler) {
        this.name = roomName;
        this.kurentoClient = kurentoClient;
        this.roomHandler = roomHandler;
        log.info("ROOM {} has been created", (Object)roomName);
    }

    public String getName() {
        return this.name;
    }

    public MediaPipeline getPipeline() {
        try {
            this.pipelineLatch.await(30L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        return this.pipeline;
    }

    public void join(String participantId, String userName, boolean webParticipant) throws RoomException {
        this.checkClosed();
        if (userName == null || userName.isEmpty()) {
            throw new RoomException(RoomException.Code.GENERIC_ERROR_CODE, "Empty user name is not allowed");
        }
        for (Participant p : this.participants.values()) {
            if (!p.getName().equals(userName)) continue;
            throw new RoomException(RoomException.Code.EXISTING_USER_IN_ROOM_ERROR_CODE, "User '" + userName + "' already exists in room '" + this.name + "'");
        }
        if (this.pipeline == null) {
            log.info("ROOM {}: Creating MediaPipeline", (Object)this.name);
            try {
                this.kurentoClient.createMediaPipeline((Continuation)new Continuation<MediaPipeline>(){

                    public void onSuccess(MediaPipeline result) throws Exception {
                        Room.this.pipeline = result;
                        Room.this.pipelineLatch.countDown();
                        log.debug("ROOM {}: Created MediaPipeline", (Object)Room.this.name);
                    }

                    public void onError(Throwable cause) throws Exception {
                        Room.this.pipelineLatch.countDown();
                        log.error("ROOM {}: Failed to create MediaPipeline", (Object)Room.this.name, (Object)cause);
                    }
                });
            }
            catch (Exception e) {
                log.error("Unable to create media pipeline for room '{}'", (Object)this.name, (Object)e);
                this.pipelineLatch.countDown();
            }
            if (this.getPipeline() == null) {
                throw new RoomException(RoomException.Code.CANNOT_CREATE_ROOM_ERROR_CODE, "Unable to create media pipeline for room '" + this.name + "'");
            }
            this.pipeline.addErrorListener((EventListener)new EventListener<ErrorEvent>(){

                public void onEvent(ErrorEvent event) {
                    String desc = event.getType() + ": " + event.getDescription() + "(errCode=" + event.getErrorCode() + ")";
                    log.warn("ROOM {}: Pipeline error encountered: {}", (Object)Room.this.name, (Object)desc);
                    Room.this.roomHandler.onPipelineError(Room.this.name, Room.this.getParticipantIds(), desc);
                }
            });
        }
        this.participants.put(participantId, new Participant(participantId, userName, this, this.pipeline, webParticipant));
        log.info("ROOM {}: Added participant {}", (Object)this.name, (Object)userName);
    }

    public void newPublisher(Participant participant) {
        this.registerPublisher();
        for (Participant participant1 : this.participants.values()) {
            if (participant.equals(participant1)) continue;
            participant1.addSubscriber(participant.getName());
        }
        log.debug("ROOM {}: Virtually subscribed other participants {} to new publisher {}", new Object[]{this.name, this.participants.values(), participant.getName()});
    }

    public void cancelPublisher(Participant participant) {
        this.deregisterPublisher();
        for (Participant subscriber : this.participants.values()) {
            if (participant.equals(subscriber)) continue;
            subscriber.cancelReceivingMedia(participant.getName());
        }
        log.debug("ROOM {}: Unsubscribed other participants {} from the publisher {}", new Object[]{this.name, this.participants.values(), participant.getName()});
    }

    public void leave(String participantId) throws RoomException {
        this.checkClosed();
        Participant participant = (Participant)this.participants.get(participantId);
        if (participant == null) {
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User #" + participantId + " not found in room '" + this.name + "'");
        }
        log.info("PARTICIPANT {}: Leaving room {}", (Object)participant.getName(), (Object)this.name);
        if (participant.isStreaming()) {
            this.deregisterPublisher();
        }
        this.removeParticipant(participant);
        participant.close();
    }

    public Collection<Participant> getParticipants() {
        this.checkClosed();
        return this.participants.values();
    }

    public Set<String> getParticipantIds() {
        this.checkClosed();
        return this.participants.keySet();
    }

    public Participant getParticipant(String participantId) {
        this.checkClosed();
        return (Participant)this.participants.get(participantId);
    }

    public Participant getParticipantByName(String userName) {
        this.checkClosed();
        for (Participant p : this.participants.values()) {
            if (!p.getName().equals(userName)) continue;
            return p;
        }
        return null;
    }

    public void close() {
        if (!this.closed) {
            for (Participant user : this.participants.values()) {
                user.close();
            }
            this.participants.clear();
            if (this.pipeline != null) {
                this.pipeline.release((Continuation)new Continuation<Void>(){

                    public void onSuccess(Void result) throws Exception {
                        log.trace("ROOM {}: Released Pipeline", (Object)Room.this.name);
                    }

                    public void onError(Throwable cause) throws Exception {
                        log.warn("PARTICIPANT " + Room.this.name + ": Could not release Pipeline", cause);
                    }
                });
            }
            log.debug("Room {} closed", (Object)this.name);
            this.closed = true;
        } else {
            log.warn("Closing an already closed room {}", (Object)this.name);
        }
    }

    public void sendIceCandidate(String participantId, String endpointName, IceCandidate candidate) {
        this.roomHandler.onIceCandidate(this.name, participantId, endpointName, candidate);
    }

    public void sendMediaError(String participantId, String description) {
        this.roomHandler.onMediaElementError(this.name, participantId, description);
    }

    public boolean isClosed() {
        return this.closed;
    }

    private void checkClosed() {
        if (this.closed) {
            throw new RoomException(RoomException.Code.ROOM_CLOSED_ERROR_CODE, "The room '" + this.name + "' is closed");
        }
    }

    private void removeParticipant(Participant participant) {
        this.checkClosed();
        this.participants.remove(participant.getId());
        log.debug("ROOM {}: Cancel receiving media from user '{}' for other users", (Object)this.name, (Object)participant.getName());
        for (Participant other : this.participants.values()) {
            other.cancelReceivingMedia(participant.getName());
        }
    }

    public int getActivePublishers() {
        return this.activePublishers.get();
    }

    public void registerPublisher() {
        this.activePublishers.incrementAndGet();
    }

    public void deregisterPublisher() {
        this.activePublishers.decrementAndGet();
    }
}

