/*
 * Decompiled with CFR 0.152.
 */
package org.kurento.room;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.annotation.PreDestroy;
import org.kurento.client.IceCandidate;
import org.kurento.client.KurentoClient;
import org.kurento.client.MediaElement;
import org.kurento.client.MediaPipeline;
import org.kurento.client.MediaType;
import org.kurento.room.api.KurentoClientProvider;
import org.kurento.room.api.KurentoClientSessionInfo;
import org.kurento.room.api.MutedMediaType;
import org.kurento.room.api.RoomHandler;
import org.kurento.room.api.pojo.UserParticipant;
import org.kurento.room.endpoint.SdpType;
import org.kurento.room.exception.RoomException;
import org.kurento.room.internal.Participant;
import org.kurento.room.internal.Room;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RoomManager {
    private final Logger log = LoggerFactory.getLogger(RoomManager.class);
    private RoomHandler roomHandler;
    private KurentoClientProvider kcProvider;
    private final ConcurrentMap<String, Room> rooms = new ConcurrentHashMap<String, Room>();
    private volatile boolean closed = false;

    public RoomManager(RoomHandler roomHandler, KurentoClientProvider kcProvider) {
        this.roomHandler = roomHandler;
        this.kcProvider = kcProvider;
    }

    public Set<UserParticipant> joinRoom(String userName, String roomName, boolean dataChannels, boolean webParticipant, KurentoClientSessionInfo kcSessionInfo, String participantId) throws RoomException {
        this.log.debug("Request [JOIN_ROOM] user={}, room={}, web={} kcSessionInfo.room={} ({})", new Object[]{userName, roomName, webParticipant, kcSessionInfo != null ? kcSessionInfo.getRoomName() : null, participantId});
        Room room = (Room)this.rooms.get(roomName);
        if (room == null && kcSessionInfo != null) {
            this.createRoom(kcSessionInfo);
        }
        if ((room = (Room)this.rooms.get(roomName)) == null) {
            this.log.warn("Room '{}' not found");
            throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' was not found, must be created before '" + userName + "' can join");
        }
        if (room.isClosed()) {
            this.log.warn("'{}' is trying to join room '{}' but it is closing", (Object)userName, (Object)roomName);
            throw new RoomException(RoomException.Code.ROOM_CLOSED_ERROR_CODE, "'" + userName + "' is trying to join room '" + roomName + "' but it is closing");
        }
        Set<UserParticipant> existingParticipants = this.getParticipants(roomName);
        room.join(participantId, userName, dataChannels, webParticipant);
        return existingParticipants;
    }

    public Set<UserParticipant> leaveRoom(String participantId) throws RoomException {
        this.log.debug("Request [LEAVE_ROOM] ({})", (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        Room room = participant.getRoom();
        String roomName = room.getName();
        if (room.isClosed()) {
            this.log.warn("'{}' is trying to leave from room '{}' but it is closing", (Object)participant.getName(), (Object)roomName);
            throw new RoomException(RoomException.Code.ROOM_CLOSED_ERROR_CODE, "'" + participant.getName() + "' is trying to leave from room '" + roomName + "' but it is closing");
        }
        room.leave(participantId);
        Set<UserParticipant> remainingParticipants = null;
        try {
            remainingParticipants = this.getParticipants(roomName);
        }
        catch (RoomException e) {
            this.log.debug("Possible collision when closing the room '{}' (not found)");
            remainingParticipants = Collections.emptySet();
        }
        if (remainingParticipants.isEmpty()) {
            this.log.debug("No more participants in room '{}', removing it and closing it", (Object)roomName);
            room.close();
            this.rooms.remove(roomName);
            this.log.warn("Room '{}' removed and closed", (Object)roomName);
        }
        return remainingParticipants;
    }

    public String publishMedia(String participantId, boolean isOffer, String sdp, MediaElement loopbackAlternativeSrc, MediaType loopbackConnectionType, boolean doLoopback, MediaElement ... mediaElements) throws RoomException {
        this.log.debug("Request [PUBLISH_MEDIA] isOffer={} sdp={} loopbackAltSrc={} lpbkConnType={} doLoopback={} mediaElements={} ({})", new Object[]{isOffer, sdp, loopbackAlternativeSrc == null, loopbackConnectionType, doLoopback, mediaElements, participantId});
        SdpType sdpType = isOffer ? SdpType.OFFER : SdpType.ANSWER;
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        participant.createPublishingEndpoint();
        for (MediaElement elem : mediaElements) {
            participant.getPublisher().apply(elem);
        }
        String sdpResponse = participant.publishToRoom(sdpType, sdp, doLoopback, loopbackAlternativeSrc, loopbackConnectionType);
        if (sdpResponse == null) {
            throw new RoomException(RoomException.Code.MEDIA_SDP_ERROR_CODE, "Error generating SDP response for publishing user " + name);
        }
        room.newPublisher(participant);
        return sdpResponse;
    }

    public String publishMedia(String participantId, String sdp, boolean doLoopback, MediaElement ... mediaElements) throws RoomException {
        return this.publishMedia(participantId, true, sdp, null, null, doLoopback, mediaElements);
    }

    public String publishMedia(String participantId, boolean isOffer, String sdp, boolean doLoopback, MediaElement ... mediaElements) throws RoomException {
        return this.publishMedia(participantId, isOffer, sdp, null, null, doLoopback, mediaElements);
    }

    public String generatePublishOffer(String participantId) throws RoomException {
        this.log.debug("Request [GET_PUBLISH_SDP_OFFER] ({})", (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        participant.createPublishingEndpoint();
        String sdpOffer = participant.preparePublishConnection();
        if (sdpOffer == null) {
            throw new RoomException(RoomException.Code.MEDIA_SDP_ERROR_CODE, "Error generating SDP offer for publishing user " + name);
        }
        room.newPublisher(participant);
        return sdpOffer;
    }

    public void unpublishMedia(String participantId) throws RoomException {
        this.log.debug("Request [UNPUBLISH_MEDIA] ({})", (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        if (!participant.isStreaming()) {
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + participant.getName() + "' is not streaming media");
        }
        Room room = participant.getRoom();
        participant.unpublishMedia();
        room.cancelPublisher(participant);
    }

    public String subscribe(String remoteName, String sdpOffer, String participantId) throws RoomException {
        this.log.debug("Request [SUBSCRIBE] remoteParticipant={} sdpOffer={} ({})", new Object[]{remoteName, sdpOffer, participantId});
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        Participant senderParticipant = room.getParticipantByName(remoteName);
        if (senderParticipant == null) {
            this.log.warn("PARTICIPANT {}: Requesting to recv media from user {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User '" + remoteName + " not found in room '" + room.getName() + "'");
        }
        if (!senderParticipant.isStreaming()) {
            this.log.warn("PARTICIPANT {}: Requesting to recv media from user {} in room {} but user is not streaming media", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "User '" + remoteName + " not streaming media in room '" + room.getName() + "'");
        }
        String sdpAnswer = participant.receiveMediaFrom(senderParticipant, sdpOffer);
        if (sdpAnswer == null) {
            throw new RoomException(RoomException.Code.MEDIA_SDP_ERROR_CODE, "Unable to generate SDP answer when subscribing '" + name + "' to '" + remoteName + "'");
        }
        return sdpAnswer;
    }

    public void unsubscribe(String remoteName, String participantId) throws RoomException {
        this.log.debug("Request [UNSUBSCRIBE] remoteParticipant={} ({})", (Object)remoteName, (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        Participant senderParticipant = room.getParticipantByName(remoteName);
        if (senderParticipant == null) {
            this.log.warn("PARTICIPANT {}: Requesting to unsubscribe from user {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
        }
        participant.cancelReceivingMedia(remoteName);
    }

    public void onIceCandidate(String endpointName, String candidate, int sdpMLineIndex, String sdpMid, String participantId) throws RoomException {
        this.log.debug("Request [ICE_CANDIDATE] endpoint={} candidate={} sdpMLineIdx={} sdpMid={} ({})", new Object[]{endpointName, candidate, sdpMLineIndex, sdpMid, participantId});
        Participant participant = this.getParticipant(participantId);
        participant.addIceCandidate(endpointName, new IceCandidate(candidate, sdpMid, sdpMLineIndex));
    }

    public void addMediaElement(String participantId, MediaElement element) throws RoomException {
        this.addMediaElement(participantId, element, null);
    }

    public void addMediaElement(String participantId, MediaElement element, MediaType type) throws RoomException {
        this.log.debug("Add media element {} (connection type: {}) to participant {}", new Object[]{element.getId(), type, participantId});
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new RoomException(RoomException.Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
        }
        participant.shapePublisherMedia(element, type);
    }

    public void removeMediaElement(String participantId, MediaElement element) throws RoomException {
        this.log.debug("Remove media element {} from participant {}", (Object)element.getId(), (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new RoomException(RoomException.Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
        }
        participant.getPublisher().revert(element);
    }

    public void mutePublishedMedia(MutedMediaType muteType, String participantId) throws RoomException {
        this.log.debug("Request [MUTE_PUBLISHED] muteType={} ({})", (Object)muteType, (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new RoomException(RoomException.Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
        }
        if (!participant.isStreaming()) {
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + name + "' is not streaming media");
        }
        participant.mutePublishedMedia(muteType);
    }

    public void unmutePublishedMedia(String participantId) throws RoomException {
        this.log.debug("Request [UNMUTE_PUBLISHED] muteType={} ({})", (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        if (participant.isClosed()) {
            throw new RoomException(RoomException.Code.USER_CLOSED_ERROR_CODE, "Participant '" + name + "' has been closed");
        }
        if (!participant.isStreaming()) {
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Participant '" + name + "' is not streaming media");
        }
        participant.unmutePublishedMedia();
    }

    public void muteSubscribedMedia(String remoteName, MutedMediaType muteType, String participantId) throws RoomException {
        this.log.debug("Request [MUTE_SUBSCRIBED] remoteParticipant={} muteType={} ({})", new Object[]{remoteName, muteType, participantId});
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        Participant senderParticipant = room.getParticipantByName(remoteName);
        if (senderParticipant == null) {
            this.log.warn("PARTICIPANT {}: Requesting to mute streaming from {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
        }
        if (!senderParticipant.isStreaming()) {
            this.log.warn("PARTICIPANT {}: Requesting to mute streaming from {} in room {} but user is not streaming media", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "User '" + remoteName + " not streaming media in room '" + room.getName() + "'");
        }
        participant.muteSubscribedMedia(senderParticipant, muteType);
    }

    public void unmuteSubscribedMedia(String remoteName, String participantId) throws RoomException {
        this.log.debug("Request [UNMUTE_SUBSCRIBED] remoteParticipant={} ({})", (Object)remoteName, (Object)participantId);
        Participant participant = this.getParticipant(participantId);
        String name = participant.getName();
        Room room = participant.getRoom();
        Participant senderParticipant = room.getParticipantByName(remoteName);
        if (senderParticipant == null) {
            this.log.warn("PARTICIPANT {}: Requesting to unmute streaming from {} in room {} but user could not be found", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "User " + remoteName + " not found in room " + room.getName());
        }
        if (!senderParticipant.isStreaming()) {
            this.log.warn("PARTICIPANT {}: Requesting to unmute streaming from {} in room {} but user is not streaming media", new Object[]{name, remoteName, room.getName()});
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "User '" + remoteName + " not streaming media in room '" + room.getName() + "'");
        }
        participant.unmuteSubscribedMedia(senderParticipant);
    }

    @PreDestroy
    public void close() {
        this.closed = true;
        this.log.info("Closing all rooms");
        for (String roomName : this.rooms.keySet()) {
            try {
                this.closeRoom(roomName);
            }
            catch (Exception e) {
                this.log.warn("Error closing room '{}'", (Object)roomName, (Object)e);
            }
        }
    }

    public boolean isClosed() {
        return this.closed;
    }

    public Set<String> getRooms() {
        return new HashSet<String>(this.rooms.keySet());
    }

    public Set<UserParticipant> getParticipants(String roomName) throws RoomException {
        Room room = (Room)this.rooms.get(roomName);
        if (room == null) {
            throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = room.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPublishers(String roomName) throws RoomException {
        Room r = (Room)this.rooms.get(roomName);
        if (r == null) {
            throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = r.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed() || !p.isStreaming()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), true));
        }
        return userParts;
    }

    public Set<UserParticipant> getSubscribers(String roomName) throws RoomException {
        Room r = (Room)this.rooms.get(roomName);
        if (r == null) {
            throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
        }
        Collection<Participant> participants = r.getParticipants();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (Participant p : participants) {
            if (p.isClosed() || !p.isSubscribed()) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName(), p.isStreaming()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPeerPublishers(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + participantId + "' was found");
        }
        Set<String> subscribedEndpoints = participant.getConnectedSubscribedEndpoints();
        Room room = participant.getRoom();
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        for (String epName : subscribedEndpoints) {
            Participant p = room.getParticipantByName(epName);
            userParts.add(new UserParticipant(p.getId(), p.getName()));
        }
        return userParts;
    }

    public Set<UserParticipant> getPeerSubscribers(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + participantId + "' was found");
        }
        if (!participant.isStreaming()) {
            throw new RoomException(RoomException.Code.USER_NOT_STREAMING_ERROR_CODE, "Participant with id '" + participantId + "' is not a publisher yet");
        }
        HashSet<UserParticipant> userParts = new HashSet<UserParticipant>();
        Room room = participant.getRoom();
        String endpointName = participant.getName();
        for (Participant p : room.getParticipants()) {
            Set<String> subscribedEndpoints;
            if (p.equals(participant) || !(subscribedEndpoints = p.getConnectedSubscribedEndpoints()).contains(endpointName)) continue;
            userParts.add(new UserParticipant(p.getId(), p.getName()));
        }
        return userParts;
    }

    public boolean isPublisherStreaming(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + participantId + "' was found");
        }
        if (participant.isClosed()) {
            throw new RoomException(RoomException.Code.USER_CLOSED_ERROR_CODE, "Participant '" + participant.getName() + "' has been closed");
        }
        return participant.isStreaming();
    }

    public void createRoom(KurentoClientSessionInfo kcSessionInfo) throws RoomException {
        String roomName = kcSessionInfo.getRoomName();
        Room room = (Room)this.rooms.get(kcSessionInfo);
        if (room != null) {
            throw new RoomException(RoomException.Code.ROOM_CANNOT_BE_CREATED_ERROR_CODE, "Room '" + roomName + "' already exists");
        }
        KurentoClient kurentoClient = this.kcProvider.getKurentoClient(kcSessionInfo);
        room = new Room(roomName, kurentoClient, this.roomHandler, this.kcProvider.destroyWhenUnused());
        Room oldRoom = this.rooms.putIfAbsent(roomName, room);
        if (oldRoom != null) {
            this.log.warn("Room '{}' has just been created by another thread", (Object)roomName);
            return;
        }
        String kcName = "[NAME NOT AVAILABLE]";
        if (kurentoClient.getServerManager() != null) {
            kcName = kurentoClient.getServerManager().getName();
        }
        this.log.warn("No room '{}' exists yet. Created one using KurentoClient '{}'.", (Object)roomName, (Object)kcName);
    }

    public Set<UserParticipant> closeRoom(String roomName) throws RoomException {
        Room room = (Room)this.rooms.get(roomName);
        if (room == null) {
            throw new RoomException(RoomException.Code.ROOM_NOT_FOUND_ERROR_CODE, "Room '" + roomName + "' not found");
        }
        if (room.isClosed()) {
            throw new RoomException(RoomException.Code.ROOM_CLOSED_ERROR_CODE, "Room '" + roomName + "' already closed");
        }
        Set<UserParticipant> participants = this.getParticipants(roomName);
        HashSet<String> pids = new HashSet<String>(room.getParticipantIds());
        for (String pid : pids) {
            try {
                room.leave(pid);
            }
            catch (RoomException e) {
                this.log.warn("Error evicting participant with id '{}' from room '{}'", new Object[]{pid, roomName, e});
            }
        }
        room.close();
        this.rooms.remove(roomName);
        this.log.warn("Room '{}' removed and closed", (Object)roomName);
        return participants;
    }

    public MediaPipeline getPipeline(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        if (participant == null) {
            throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + participantId + "' was found");
        }
        return participant.getPipeline();
    }

    public String getRoomName(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        return participant.getRoom().getName();
    }

    public String getParticipantName(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        return participant.getName();
    }

    public UserParticipant getParticipantInfo(String participantId) throws RoomException {
        Participant participant = this.getParticipant(participantId);
        return new UserParticipant(participantId, participant.getName());
    }

    private Participant getParticipant(String pid) throws RoomException {
        for (Room r : this.rooms.values()) {
            if (r.isClosed() || !r.getParticipantIds().contains(pid) || r.getParticipant(pid) == null) continue;
            return r.getParticipant(pid);
        }
        throw new RoomException(RoomException.Code.USER_NOT_FOUND_ERROR_CODE, "No participant with id '" + pid + "' was found");
    }
}

