/*
 * Decompiled with CFR 0.152.
 */
package org.kurento.test.monitor;

import java.io.BufferedReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.ServerSocket;
import java.net.Socket;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.kurento.test.monitor.NetInfo;
import org.kurento.test.monitor.SystemInfo;

public class SystemMonitor {
    private Thread thread;
    private Map<Long, SystemInfo> infoMap = Collections.synchronizedSortedMap(new TreeMap());
    private long samplingTime = 100L;
    private double prevTotal = 0.0;
    private double prevIdle = 0.0;
    private int numClients = 0;
    private double currentLatency = 0.0;
    private int latencyHints = 0;
    private int latencyErrors = 0;
    private boolean showLantency = false;
    private static final String OK = "ok";
    private static final String ERR = "error: ";
    public static final String MONITOR_PORT_PROP = "kms.monitor.port";
    public static final int MONITOR_PORT_DEFAULT = 12345;
    public static final int KMS_WAIT_TIMEOUT = 60;
    public static final String OUTPUT_CSV = "/kms-monitor.csv";

    public SystemMonitor() {
    }

    public SystemMonitor(long samplingTime) {
        this();
        this.samplingTime = samplingTime;
    }

    public static void main(String[] args) throws InterruptedException, IOException {
        int monitorPort = args.length > 0 ? Integer.parseInt(args[0]) : 12345;
        SystemMonitor monitor = new SystemMonitor();
        ServerSocket server = new ServerSocket(monitorPort);
        System.out.println("Waiting for incoming messages...");
        boolean run = true;
        while (run) {
            Socket socket = server.accept();
            String result = OK;
            PrintWriter output = null;
            BufferedReader input = null;
            try {
                output = new PrintWriter(socket.getOutputStream(), true);
                input = new BufferedReader(new InputStreamReader(socket.getInputStream()));
                String message = input.readLine();
                System.out.println("Message received " + message);
                if (message != null) {
                    String[] commands = message.split(" ");
                    switch (commands[0]) {
                        case "start": {
                            monitor.start();
                            break;
                        }
                        case "stop": {
                            monitor.stop();
                            break;
                        }
                        case "destroy": {
                            run = false;
                            break;
                        }
                        case "writeResults": {
                            monitor.writeResults(commands[1] + OUTPUT_CSV);
                            break;
                        }
                        case "incrementNumClients": {
                            monitor.incrementNumClients();
                            break;
                        }
                        case "decrementNumClients": {
                            monitor.decrementNumClients();
                            break;
                        }
                        case "addCurrentLatency": {
                            monitor.addCurrentLatency(Double.parseDouble(commands[1]));
                            break;
                        }
                        case "setSamplingTime": {
                            monitor.setSamplingTime(Long.parseLong(commands[1]));
                            break;
                        }
                        case "incrementLatencyErrors": {
                            monitor.incrementLatencyErrors();
                            break;
                        }
                        default: {
                            result = "error: Invalid command: " + message;
                        }
                    }
                    System.out.println("Sending back message " + result);
                    output.println(result);
                }
                output.close();
                input.close();
                socket.close();
            }
            catch (IOException e) {
                result = ERR + e.getMessage();
                e.printStackTrace();
            }
        }
        server.close();
    }

    public void start() {
        final long start = new Date().getTime();
        final NetInfo initNetInfo = this.getInitNetInfo();
        ExecutorService executor = Executors.newSingleThreadExecutor();
        Callable<Integer> callable = new Callable<Integer>(){

            @Override
            public Integer call() {
                return SystemMonitor.this.getKmsPid();
            }
        };
        final Future<Integer> future = executor.submit(callable);
        this.thread = new Thread(){

            @Override
            public void run() {
                try {
                    int kmsPid = (Integer)future.get();
                    while (true) {
                        SystemInfo info = new SystemInfo();
                        NetInfo newNetInfo = SystemMonitor.this.getNetInfo(initNetInfo);
                        info.setNetInfo(newNetInfo);
                        info.setCpuPercent(SystemMonitor.this.getCpuUsage());
                        double[] mem = SystemMonitor.this.getMemSwap();
                        info.setMem((long)mem[0]);
                        info.setSwap((long)mem[1]);
                        info.setMemPercent(mem[2]);
                        info.setSwapPercent(mem[3]);
                        info.setNumClients(SystemMonitor.this.numClients);
                        double latency = SystemMonitor.this.getLatency();
                        info.setLatency(latency);
                        info.setLatencyErrors(SystemMonitor.this.latencyErrors);
                        if (latency > 0.0) {
                            SystemMonitor.this.showLantency = true;
                        }
                        info.setNumThreadsKms(SystemMonitor.this.getNumThreads(kmsPid));
                        SystemMonitor.this.infoMap.put(new Date().getTime() - start, info);
                        Thread.sleep(SystemMonitor.this.samplingTime);
                    }
                }
                catch (Exception e) {
                    System.out.println(e.getMessage());
                    return;
                }
            }
        };
        this.thread.setDaemon(true);
        this.thread.start();
    }

    public void stop() {
        this.thread.interrupt();
        this.thread.stop();
    }

    public NetInfo getNetInfo(NetInfo initNetInfo, NetInfo lastNetInfo) {
        String[] lines;
        NetInfo netInfo = new NetInfo();
        String out = this.runAndWait("/bin/sh", "-c", "cat /proc/net/dev | awk 'NR > 2'");
        for (String line : lines = out.split("\n")) {
            String[] split = line.trim().replaceAll(" +", " ").split(" ");
            String iface = split[0].replace(":", "");
            long rxBytes = Long.parseLong(split[1]);
            long txBytes = Long.parseLong(split[9]);
            netInfo.putNetInfo(iface, rxBytes, txBytes);
        }
        if (initNetInfo != null) {
            netInfo.decrementInitInfo(initNetInfo);
        }
        if (lastNetInfo != null) {
            netInfo.decrementInitInfo(lastNetInfo);
        }
        return netInfo;
    }

    public NetInfo getNetInfo(NetInfo initNetInfo) {
        return this.getNetInfo(initNetInfo, null);
    }

    public NetInfo getInitNetInfo() {
        return this.getNetInfo(null, null);
    }

    public void writeResults(String csvTitle) {
        PrintWriter pw;
        try {
            pw = new PrintWriter(new FileWriter(csvTitle));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        boolean header = false;
        for (long time : this.infoMap.keySet()) {
            if (!header) {
                pw.print("time, cpu_percetage, mem_bytes, mem_percentage, swap_bytes, swap_percentage, clients_number, kms_threads_number");
                if (this.showLantency) {
                    pw.print(", latency_ms_avg, latency_errors_number");
                }
                pw.print(this.infoMap.get(time).getNetInfo().parseHeaderEntry());
                pw.println("");
                header = true;
            }
            String parsedtime = new SimpleDateFormat("mm:ss.SSS").format(time);
            double cpu = this.infoMap.get(time).getCpuPercent();
            long mem = this.infoMap.get(time).getMem();
            double memPercent = this.infoMap.get(time).getMemPercent();
            long swap = this.infoMap.get(time).getSwap();
            double swapPercent = this.infoMap.get(time).getSwapPercent();
            pw.print(parsedtime + "," + cpu + "," + mem + "," + memPercent + "," + swap + "," + swapPercent + "," + this.infoMap.get(time).getNumClients() + "," + this.infoMap.get(time).getNumThreadsKms());
            if (this.showLantency) {
                pw.print("," + this.infoMap.get(time).getLatency() + "," + this.infoMap.get(time).getLatencyErrors());
            }
            pw.print(this.infoMap.get(time).getNetInfo().parseNetEntry());
            pw.println("");
        }
        pw.close();
    }

    public double getCpuUsage() {
        String[] cpu = this.runAndWait("/bin/sh", "-c", "cat /proc/stat | grep '^cpu ' | awk '{print substr($0, index($0, $2))}'").replaceAll("\n", "").split(" ");
        double idle = Double.parseDouble(cpu[3]);
        double total = 0.0;
        for (String s : cpu) {
            total += Double.parseDouble(s);
        }
        double diffIdle = idle - this.prevIdle;
        double diffTotal = total - this.prevTotal;
        double diffUsage = (1000.0 * (diffTotal - diffIdle) / diffTotal + 5.0) / 10.0;
        this.prevTotal = total;
        this.prevIdle = idle;
        return diffUsage;
    }

    public double[] getMemSwap() {
        String[] mem = this.runAndWait("free").replaceAll("\n", ",").replaceAll(" +", " ").split(" ");
        long usedMem = Long.parseLong(mem[15]);
        long usedSwap = Long.parseLong(mem[19]);
        long totalMem = Long.parseLong(mem[7]);
        long totalSwap = Long.parseLong(mem[20]);
        double percetageMem = (double)usedMem / (double)totalMem * 100.0;
        double percetageSwap = (double)usedSwap / (double)totalSwap * 100.0;
        if (Double.isNaN(percetageMem)) {
            percetageMem = 0.0;
        }
        if (Double.isNaN(percetageSwap)) {
            percetageSwap = 0.0;
        }
        double[] out = new double[]{usedMem, usedSwap, percetageMem, percetageSwap};
        return out;
    }

    private int getKmsPid() {
        String kmsPid;
        boolean reachable = false;
        long endTimeMillis = System.currentTimeMillis() + 60000L;
        do {
            boolean bl = reachable = !(kmsPid = this.runAndWait("/bin/sh", "-c", "ps axf | grep kurento-media-server | grep -v grep | awk '{print $1}'").replaceAll("\n", "")).equals("");
            if (kmsPid.contains(" ")) {
                throw new RuntimeException("More than one KMS process are started (PIDs:" + kmsPid + ")");
            }
            if (reachable) break;
            try {
                Thread.sleep(TimeUnit.SECONDS.toMillis(1L));
            }
            catch (InterruptedException ie) {
                ie.printStackTrace();
            }
        } while (System.currentTimeMillis() <= endTimeMillis);
        if (!reachable) {
            throw new RuntimeException("KMS is not started in the local machine");
        }
        return Integer.parseInt(kmsPid);
    }

    private int getNumThreads(int kmsPid) {
        return Integer.parseInt(this.runAndWait("/bin/sh", "-c", "cat /proc/" + kmsPid + "/stat | awk '{print $20}'").replaceAll("\n", ""));
    }

    private String runAndWait(String ... command) {
        try {
            Process p = new ProcessBuilder(command).redirectErrorStream(true).start();
            return this.inputStreamToString(p.getInputStream());
        }
        catch (IOException e) {
            throw new RuntimeException("Exception executing command on the shell: " + Arrays.toString(command), e);
        }
    }

    private String inputStreamToString(InputStream in) throws IOException {
        InputStreamReader is = new InputStreamReader(in);
        StringBuilder sb = new StringBuilder();
        BufferedReader br = new BufferedReader(is);
        String read = br.readLine();
        while (read != null) {
            sb.append(read);
            read = br.readLine();
            sb.append('\n');
            sb.append(' ');
        }
        return sb.toString().trim();
    }

    public int getNumClients() {
        return this.numClients;
    }

    public synchronized void incrementNumClients() {
        ++this.numClients;
    }

    public synchronized void decrementNumClients() {
        --this.numClients;
    }

    public int getLatencyErrors() {
        return this.latencyErrors;
    }

    public synchronized void incrementLatencyErrors() {
        ++this.latencyErrors;
    }

    public double getLatency() {
        double latency = this.latencyHints > 0 ? this.currentLatency / (double)this.latencyHints : 0.0;
        this.currentLatency = 0.0;
        this.latencyHints = 0;
        return latency;
    }

    public synchronized void addCurrentLatency(double currentLatency) {
        this.currentLatency += currentLatency;
        ++this.latencyHints;
    }

    public void setSamplingTime(long samplingTime) {
        this.samplingTime = samplingTime;
    }
}

