/*
 * Decompiled with CFR 0.152.
 */
package org.lable.oss.dynamicconfig.zookeeper;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.lable.oss.dynamicconfig.zookeeper.ZookeeperLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MonitoringZookeeperConnection
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(MonitoringZookeeperConnection.class);
    static final int MAX_RETRY_WAIT_MINUTES = 5;
    static final String LOCKING_NODES = "/locks";
    static final long MAINTENANCE_TIMER_INTERVAL = TimeUnit.MINUTES.toMillis(5L);
    final String connectString;
    final Map<String, NodeState> monitoredFiles;
    final NodeChangeListener changeListener;
    final String identityString;
    final Watcher watcher;
    final Queue<Task> jobQueue;
    final ScheduledExecutorService executor;
    CompletableFuture<Void> connectionTask;
    Future<?> jobRunner;
    ZooKeeper zooKeeper;
    State state;
    int retryCounter;
    int retryWait;

    public MonitoringZookeeperConnection(String[] quorum, String chroot, NodeChangeListener changeListener) {
        if (chroot == null) {
            this.connectString = String.join((CharSequence)",", quorum);
        } else {
            if (!chroot.startsWith("/")) {
                chroot = "/" + chroot;
            }
            this.connectString = String.join((CharSequence)",", quorum) + chroot;
        }
        logger.info("Monitoring: {}", (Object)this.connectString);
        this.state = State.CONNECTING;
        this.changeListener = changeListener;
        this.jobQueue = new ConcurrentLinkedQueue<Task>();
        this.monitoredFiles = new HashMap<String, NodeState>();
        this.identityString = "MonitoringZKConn " + chroot;
        this.watcher = new MZKWatcher();
        this.executor = Executors.newSingleThreadScheduledExecutor();
        this.resetRetryCounters();
        this.connectionTask = CompletableFuture.runAsync(this::connect, this.executor).thenRun(() -> {
            this.jobRunner = this.executor.scheduleWithFixedDelay(new JobRunner(this), 50L, 1000L, TimeUnit.MILLISECONDS);
        });
    }

    public Optional<InputStream> load(String node) {
        if (!MonitoringZookeeperConnection.isLegalName(node)) {
            logger.error(this.identityString + ": ZooKeeper node name is not valid ({}).", (Object)node);
            return Optional.empty();
        }
        switch (this.state) {
            case CONNECTING: {
                logger.info(this.identityString + ": Connection to ZooKeeper not established yet; waiting\u2026");
                try {
                    TimeUnit.MILLISECONDS.sleep(300L);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return Optional.empty();
                }
                return this.load(node);
            }
            case LIVE: {
                break;
            }
            case CLOSED: {
                return Optional.empty();
            }
        }
        try {
            byte[] data = this.zooKeeper.getData(node, false, null);
            return data == null || data.length == 0 ? Optional.empty() : Optional.of(new ByteArrayInputStream(data));
        }
        catch (KeeperException e) {
            logger.error(this.identityString + ": Failure during getData on " + node, (Throwable)e);
            return Optional.empty();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return Optional.empty();
        }
    }

    public void set(String node, String value, boolean createIfMissing) {
        if (!MonitoringZookeeperConnection.isLegalName(node)) {
            logger.error(this.identityString + ": ZooKeeper node name is not valid ({}).", (Object)node);
            return;
        }
        byte[] data = value.getBytes(Charset.forName("UTF-8"));
        Task task = () -> {
            try {
                this.zooKeeper.setData(node, data, -1);
            }
            catch (KeeperException.NoNodeException e) {
                if (createIfMissing) {
                    this.zooKeeper.create(node, data, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
                }
                logger.error("Failed to set node " + node + ", it is missing.", (Throwable)e);
            }
        };
        this.jobQueue.add(task);
    }

    public State getState() {
        return this.state;
    }

    synchronized void connect() {
        if (this.state != State.CONNECTING) {
            return;
        }
        if (this.zooKeeper != null) {
            try {
                this.zooKeeper.close();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
        if (this.retryCounter > 0) {
            logger.warn(this.identityString + ": Failed to connect to Zookeeper quorum, retrying (" + this.retryCounter + ").");
        }
        try {
            this.zooKeeper = new ZooKeeper(this.connectString, 3000, this.watcher);
        }
        catch (IOException e) {
            this.waitBeforeRetrying();
            this.connect();
        }
    }

    public void listen(String node) {
        if (!MonitoringZookeeperConnection.isLegalName(node)) {
            logger.error("Configuration source name is not valid ({}).", (Object)node);
            return;
        }
        if (this.state != State.LIVE) {
            try {
                logger.info("Not connected.");
                TimeUnit.MILLISECONDS.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.listen(node);
        }
        this.exclusiveListen(node);
    }

    synchronized void exclusiveListen(String node) {
        NodeState nodeState;
        if (this.monitoredFiles.containsKey(node)) {
            nodeState = this.monitoredFiles.get(node);
            nodeState.monitored = true;
        } else {
            nodeState = new NodeState(node);
            this.monitoredFiles.put(node, nodeState);
        }
        this.attemptToSetWatcher(nodeState);
    }

    synchronized void processPart(WatchedEvent event) {
        if (this.state == State.CLOSED) {
            return;
        }
        Watcher.Event.EventType eventType = event.getType();
        String znode = event.getPath();
        if (znode == null) {
            return;
        }
        NodeState nodeState = this.monitoredFiles.get(znode);
        if (nodeState == null) {
            logger.info("Watcher triggered ({}) for unknown node {}.", (Object)eventType, (Object)znode);
            return;
        }
        if (!nodeState.monitored) {
            nodeState.watcherState = NodeState.WatcherState.NO_WATCHER;
            return;
        }
        nodeState.watcherState = NodeState.WatcherState.NEEDS_WATCHER;
        switch (eventType) {
            case None: 
            case NodeChildrenChanged: {
                break;
            }
            case NodeDeleted: {
                logger.error("Watched configuration part deleted! Keeping the last-known version in memory until this part is restored, or if all references to this part are removed from the config.");
                break;
            }
            case NodeCreated: 
            case NodeDataChanged: {
                try {
                    byte[] data = this.zooKeeper.getData(znode, false, null);
                    this.changeListener.changed(znode, new ByteArrayInputStream(data));
                    break;
                }
                catch (KeeperException e) {
                    logger.error("Failed to read data from znode " + znode + "! Will attempt to re-set later.", (Throwable)e);
                    break;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
        this.attemptToSetWatcher(nodeState);
    }

    synchronized void attemptToSetWatcher(NodeState nodeState) {
        if (nodeState == null || nodeState.watcherState == NodeState.WatcherState.HAS_WATCHER) {
            return;
        }
        try {
            this.zooKeeper.exists(nodeState.znode, this::processPart);
            nodeState.watcherState = NodeState.WatcherState.HAS_WATCHER;
        }
        catch (KeeperException e) {
            logger.error("Failed to set watcher for node " + nodeState.znode + "! Will attempt to re-set later.", (Throwable)e);
            nodeState.watcherState = NodeState.WatcherState.NEEDS_WATCHER;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public synchronized void stopListening(String part) {
        if (this.monitoredFiles.containsKey(part)) {
            this.monitoredFiles.get((Object)part).monitored = false;
        }
    }

    public ZookeeperLock prepareLock(String znode) {
        return new ZookeeperLock(this.zooKeeper, LOCKING_NODES + znode);
    }

    @Override
    public void close() throws IOException {
        this.state = State.CLOSED;
        if (this.connectionTask != null && !this.connectionTask.isDone()) {
            this.connectionTask.cancel(true);
        }
        if (this.jobRunner != null) {
            this.jobRunner.cancel(false);
        }
        if (this.zooKeeper != null) {
            try {
                this.zooKeeper.close();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    synchronized void performMaintenanceTasks() {
        this.monitoredFiles.values().stream().filter(nodeState -> nodeState.monitored).filter(nodeState -> nodeState.watcherState == NodeState.WatcherState.NEEDS_WATCHER).forEach(nodeState -> {
            logger.warn("Reset watcher on " + ((NodeState)nodeState).znode);
            this.attemptToSetWatcher((NodeState)nodeState);
        });
    }

    void waitBeforeRetrying() {
        if (this.retryWait < 300) {
            this.retryWait *= 2;
            if (this.retryWait > 300) {
                this.retryWait = 300;
            }
        }
        ++this.retryCounter;
        try {
            logger.info("Failed to connect to ZooKeeper quorum, waiting " + this.retryWait + "s before retrying.");
            TimeUnit.SECONDS.sleep(this.retryWait);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
    }

    void resetRetryCounters() {
        this.retryCounter = 0;
        this.retryWait = 10;
    }

    static boolean isLegalName(String name) {
        return name != null && !name.isEmpty();
    }

    @FunctionalInterface
    public static interface NodeChangeListener {
        public void changed(String var1, InputStream var2);
    }

    static interface Task {
        public void execute() throws KeeperException, InterruptedException;
    }

    static class NodeState {
        private final String znode;
        boolean monitored;
        WatcherState watcherState;

        public NodeState(String znode) {
            this.znode = znode;
            this.monitored = true;
            this.watcherState = WatcherState.NO_WATCHER;
        }

        static enum WatcherState {
            NEEDS_WATCHER,
            HAS_WATCHER,
            NO_WATCHER;

        }
    }

    static class JobRunner
    implements Runnable {
        private static final Logger logger = LoggerFactory.getLogger(JobRunner.class);
        MonitoringZookeeperConnection monitoringZookeeperConnection;
        long maintenanceTaskLastRan;

        public JobRunner(MonitoringZookeeperConnection monitoringZookeeperConnection) {
            this.monitoringZookeeperConnection = monitoringZookeeperConnection;
            this.maintenanceTaskLastRan = System.currentTimeMillis();
        }

        @Override
        public void run() {
            switch (this.monitoringZookeeperConnection.getState()) {
                case CONNECTING: 
                case CLOSED: {
                    break;
                }
                case LIVE: {
                    while (!this.monitoringZookeeperConnection.jobQueue.isEmpty()) {
                        Task job = this.monitoringZookeeperConnection.jobQueue.poll();
                        if (job == null) continue;
                        try {
                            job.execute();
                        }
                        catch (KeeperException.ConnectionLossException | KeeperException.OperationTimeoutException | KeeperException.SessionExpiredException e) {
                            this.monitoringZookeeperConnection.jobQueue.add(job);
                        }
                        catch (KeeperException e) {
                            logger.error("Unexpected ZooKeeper error.", (Throwable)e);
                        }
                        catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                    }
                    long now = System.currentTimeMillis();
                    if (this.maintenanceTaskLastRan + MAINTENANCE_TIMER_INTERVAL >= now) break;
                    this.monitoringZookeeperConnection.performMaintenanceTasks();
                    this.maintenanceTaskLastRan = now;
                }
            }
        }
    }

    private class MZKWatcher
    implements Watcher {
        private MZKWatcher() {
        }

        public void process(WatchedEvent event) {
            if (MonitoringZookeeperConnection.this.state != State.LIVE && MonitoringZookeeperConnection.this.state != State.CONNECTING) {
                return;
            }
            Watcher.Event.KeeperState state = event.getState();
            Watcher.Event.EventType type = event.getType();
            block1 : switch (state) {
                case SyncConnected: {
                    MonitoringZookeeperConnection.this.resetRetryCounters();
                    MonitoringZookeeperConnection.this.state = State.LIVE;
                    switch (type) {
                        case NodeCreated: {
                            logger.info("Our configuration parent znode was created (why was it gone?).");
                            break block1;
                        }
                        case NodeDeleted: {
                            logger.error("Our configuration parent znode was deleted! Waiting for it to be recreated\u2026");
                            break block1;
                        }
                    }
                    break;
                }
                case Disconnected: {
                    logger.warn("Disconnected from Zookeeper quorum, reconnecting\u2026");
                    MonitoringZookeeperConnection.this.state = State.CONNECTING;
                    MonitoringZookeeperConnection.this.waitBeforeRetrying();
                    break;
                }
                case Expired: {
                    logger.warn("Connection to Zookeeper quorum expired. Attempting to reconnect\u2026");
                    MonitoringZookeeperConnection.this.state = State.CONNECTING;
                    MonitoringZookeeperConnection.this.connect();
                    return;
                }
            }
            try {
                MonitoringZookeeperConnection.this.zooKeeper.getChildren("/", (Watcher)this);
            }
            catch (KeeperException.SessionExpiredException e) {
                MonitoringZookeeperConnection.this.connect();
            }
            catch (KeeperException e) {
                logger.error("KeeperException caught, retrying\u2026", (Throwable)e);
                MonitoringZookeeperConnection.this.waitBeforeRetrying();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
    }

    static enum State {
        CONNECTING,
        LIVE,
        CLOSED;

    }
}

