/*
 * Decompiled with CFR 0.152.
 */
package org.lable.oss.uniqueid.zookeeper;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;
import org.lable.oss.uniqueid.zookeeper.ZooKeeperHelper;
import org.lable.oss.uniqueid.zookeeper.connection.ZooKeeperConnection;
import org.lable.oss.uniqueid.zookeeper.connection.ZooKeeperConnectionObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ResourceClaim
implements ZooKeeperConnectionObserver,
Closeable {
    static final Logger logger = LoggerFactory.getLogger(ResourceClaim.class);
    static String ZNODE;
    static String QUEUE_NODE;
    static String POOL_NODE;
    static final String LOCKING_TICKET = "nr-00000000000000";
    static final Random random;
    final int resource;
    final int poolSize;
    final ZooKeeper zookeeper;
    final ZooKeeperConnection zooKeeperConnection;
    protected State state = State.UNCLAIMED;

    ResourceClaim(ZooKeeperConnection zooKeeperConnection, int poolSize, String znode) throws IOException {
        logger.debug("Acquiring resource-claim\u2026");
        ZNODE = znode;
        QUEUE_NODE = znode + "/queue";
        POOL_NODE = znode + "/pool";
        zooKeeperConnection.registerObserver(this);
        this.poolSize = poolSize;
        this.zooKeeperConnection = zooKeeperConnection;
        this.zookeeper = zooKeeperConnection.get();
        if (this.zookeeper.getState() != ZooKeeper.States.CONNECTED) {
            throw new IOException("Not connected to ZooKeeper quorum.");
        }
        try {
            this.ensureRequiredZnodesExist(this.zookeeper, znode);
            String placeInLine = ResourceClaim.acquireLock(this.zookeeper, QUEUE_NODE);
            this.resource = this.claimResource(this.zookeeper, POOL_NODE, poolSize);
            ResourceClaim.releaseTicket(this.zookeeper, QUEUE_NODE, placeInLine);
        }
        catch (KeeperException e) {
            throw new IOException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException(e);
        }
        this.state = State.HAS_CLAIM;
        logger.debug("Resource-claim acquired ({}).", (Object)this.resource);
    }

    void ensureRequiredZnodesExist(ZooKeeper zookeeper, String znode) throws KeeperException, InterruptedException {
        ZooKeeperHelper.mkdirp(zookeeper, znode);
        ZooKeeperHelper.createIfNotThere(zookeeper, QUEUE_NODE);
        ZooKeeperHelper.createIfNotThere(zookeeper, POOL_NODE);
    }

    public static ResourceClaim claim(ZooKeeperConnection zooKeeperConnection, int poolSize, String znode) throws IOException {
        return new ResourceClaim(zooKeeperConnection, poolSize, znode);
    }

    public int get() {
        if (this.state != State.HAS_CLAIM) {
            throw new IllegalStateException("Resource claim not held.");
        }
        return this.resource;
    }

    @Override
    public void close() {
        this.close(false);
    }

    public void close(boolean nodeAlreadyDeleted) {
        if (this.state == State.CLAIM_RELINQUISHED) {
            return;
        }
        this.state = State.CLAIM_RELINQUISHED;
        this.zooKeeperConnection.deregisterObserver(this);
        logger.debug("Closing resource-claim ({}).", (Object)this.resource);
        if (nodeAlreadyDeleted) {
            return;
        }
        new Timer().schedule(new TimerTask(){

            @Override
            public void run() {
                ResourceClaim.this.relinquishResource(ResourceClaim.this.zookeeper, POOL_NODE, ResourceClaim.this.resource);
            }
        }, TimeUnit.SECONDS.toMillis(2L));
    }

    static String acquireLock(ZooKeeper zookeeper, String lockNode) throws KeeperException, InterruptedException {
        String placeInLine = ResourceClaim.takeQueueTicket(zookeeper, lockNode);
        logger.debug("Acquiring lock, waiting in queue: {}.", (Object)placeInLine);
        return ResourceClaim.waitInLine(zookeeper, lockNode, placeInLine);
    }

    static String takeQueueTicket(ZooKeeper zookeeper, String lockNode) throws InterruptedException, KeeperException {
        String ticket = String.format("nr-%014d-%04d", System.currentTimeMillis(), random.nextInt(10000));
        if (ResourceClaim.grabTicket(zookeeper, lockNode, ticket)) {
            return ticket;
        }
        return ResourceClaim.takeQueueTicket(zookeeper, lockNode);
    }

    static void releaseTicket(ZooKeeper zookeeper, String lockNode, String ticket) throws KeeperException, InterruptedException {
        block2: {
            logger.debug("Releasing ticket {}.", (Object)ticket);
            try {
                zookeeper.delete(lockNode + "/" + ticket, -1);
            }
            catch (KeeperException e) {
                if (e.code() == KeeperException.Code.NONODE) break block2;
                throw e;
            }
        }
    }

    static String waitInLine(ZooKeeper zookeeper, String lockNode, String placeInLine) throws KeeperException, InterruptedException {
        String placeBeforeUs;
        List children = zookeeper.getChildren(lockNode, false);
        Collections.sort(children);
        if (children.size() == 0) {
            logger.warn("getChildren() returned empty list, but we created a ticket.");
            return ResourceClaim.acquireLock(zookeeper, lockNode);
        }
        boolean lockingTicketExists = ((String)children.get(0)).equals(LOCKING_TICKET);
        if (lockingTicketExists) {
            children.remove(0);
        }
        int positionInQueue = -1;
        int i = 0;
        for (String child : children) {
            if (child.equals(placeInLine)) {
                positionInQueue = i;
                break;
            }
            ++i;
        }
        if (positionInQueue < 0) {
            throw new RuntimeException("Created node (" + placeInLine + ") not found in getChildren().");
        }
        if (positionInQueue == 0) {
            if (ResourceClaim.grabTicket(zookeeper, lockNode, LOCKING_TICKET)) {
                ResourceClaim.releaseTicket(zookeeper, lockNode, placeInLine);
                return LOCKING_TICKET;
            }
            placeBeforeUs = LOCKING_TICKET;
        } else {
            placeBeforeUs = (String)children.get(positionInQueue - 1);
        }
        CountDownLatch latch = new CountDownLatch(1);
        Stat stat = zookeeper.exists(lockNode + "/" + placeBeforeUs, event -> latch.countDown());
        if (stat != null) {
            logger.debug("Watching place in queue before us ({})", (Object)placeBeforeUs);
            latch.await();
        }
        return ResourceClaim.waitInLine(zookeeper, lockNode, placeInLine);
    }

    static boolean grabTicket(ZooKeeper zookeeper, String lockNode, String ticket) throws InterruptedException, KeeperException {
        try {
            zookeeper.create(lockNode + "/" + ticket, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL);
        }
        catch (KeeperException e) {
            if (e.code() == KeeperException.Code.NODEEXISTS) {
                logger.debug("Failed to claim ticket {}.", (Object)ticket);
                return false;
            }
            throw e;
        }
        logger.debug("Claimed ticket {}.", (Object)ticket);
        return true;
    }

    int claimResource(ZooKeeper zookeeper, String poolNode, int poolSize) throws KeeperException, InterruptedException {
        logger.debug("Trying to claim a resource.");
        List claimedResources = zookeeper.getChildren(poolNode, false);
        if (claimedResources.size() >= poolSize) {
            logger.debug("No resources available at the moment (pool size: {}), waiting.", (Object)poolSize);
            CountDownLatch latch = new CountDownLatch(1);
            zookeeper.getChildren(poolNode, event -> latch.countDown());
            latch.await();
            return this.claimResource(zookeeper, poolNode, poolSize);
        }
        for (int i = 0; i < poolSize; ++i) {
            String node;
            String resourcePath = Integer.toString(i);
            if (claimedResources.contains(resourcePath)) continue;
            try {
                logger.debug("Trying to claim seemingly available resource {}.", (Object)resourcePath);
                node = zookeeper.create(poolNode + "/" + resourcePath, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL);
            }
            catch (KeeperException e) {
                if (e.code() == KeeperException.Code.NODEEXISTS) continue;
                throw e;
            }
            zookeeper.exists(node, event -> {
                if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                    logger.debug("Resource-claim node unexpectedly deleted ({})", (Object)this.resource);
                    this.close(true);
                }
            });
            logger.debug("Successfully claimed resource {}.", (Object)resourcePath);
            return i;
        }
        return this.claimResource(zookeeper, poolNode, poolSize);
    }

    private void relinquishResource(ZooKeeper zookeeper, String poolNode, int resource) {
        logger.debug("Relinquishing claimed resource {}.", (Object)resource);
        try {
            zookeeper.delete(poolNode + "/" + Integer.toString(resource), -1);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        catch (KeeperException e) {
            logger.error("Failed to remove resource claim node {}/{}", (Object)poolNode, (Object)resource);
        }
    }

    public String getConfiguredZNode() {
        return ZNODE;
    }

    @Override
    public void disconnected() {
        logger.debug("Disconnected from ZooKeeper quorum, this invalidates the claim to resource {}.", (Object)this.resource);
        this.state = State.CLAIM_RELINQUISHED;
        this.zooKeeperConnection.deregisterObserver(this);
    }

    @Override
    public void connected() {
    }

    static {
        random = new Random();
    }

    public static enum State {
        UNCLAIMED,
        HAS_CLAIM,
        CLAIM_RELINQUISHED;

    }
}

