/*
 * Copyright (C) 2014 Lable (info@lable.nl)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.lable.oss.uniqueid.zookeeper;

import org.lable.oss.dynamicconfig.zookeeper.MonitoringZookeeperConnection;
import org.lable.oss.uniqueid.BaseUniqueIDGenerator;
import org.lable.oss.uniqueid.GeneratorIdentityHolder;
import org.lable.oss.uniqueid.IDGenerator;
import org.lable.oss.uniqueid.bytes.Mode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Create an {@link IDGenerator} capable of generating unique identifiers in a distributed environment with multiple
 * services generating them. To do this, the {@link GeneratorIdentityHolder} it uses acquires a temporary claim on a
 * generator ID negotiated via a ZooKeeper quorum.
 * <p>
 * Because claimed generator IDs are automatically returned to the pool after a set time
 * ({@link ExpiringResourceClaim#DEFAULT_CLAIM_HOLD}), there is no guarantee that IDs generated by the same
 * {@link IDGenerator} instance share the same generator ID.
 */
public class SynchronizedUniqueIDGeneratorFactory {
    final static Logger logger = LoggerFactory.getLogger(SynchronizedUniqueIDGeneratorFactory.class);
    final static Map<String, IDGenerator> instances = new HashMap<>();

    /**
     * Get the synchronized ID generator instance.
     *
     * @param zooKeeperConnection Connection to the ZooKeeper quorum.
     * @param znode               Base-path of the resource pool in ZooKeeper.
     * @param mode                Generator mode.
     * @return An instance of this class.
     * @throws IOException Thrown when something went wrong trying to find the cluster ID or trying to claim a
     *                     generator ID.
     */
    public static synchronized IDGenerator generatorFor(MonitoringZookeeperConnection zooKeeperConnection,
                                                        String znode,
                                                        Mode mode)
            throws IOException {

        if (!instances.containsKey(znode)) {
            final int clusterId = ClusterID.get(zooKeeperConnection.getActiveConnection(), znode);
            SynchronizedGeneratorIdentity generatorIdentityHolder =
                    new SynchronizedGeneratorIdentity(zooKeeperConnection, znode, clusterId, null, null);

            return generatorFor(generatorIdentityHolder, mode);
        }
        return instances.get(znode);
    }

    /**
     * Get the synchronized ID generator instance.
     *
     * @param synchronizedGeneratorIdentity An instance of {@link SynchronizedGeneratorIdentity} to (re)use for
     *                                      acquiring the generator ID.
     * @param mode                          Generator mode.
     * @return An instance of this class.
     * @throws IOException Thrown when something went wrong trying to find the cluster ID or trying to claim a
     *                     generator ID.
     */
    public static synchronized IDGenerator generatorFor(SynchronizedGeneratorIdentity synchronizedGeneratorIdentity,
                                                        Mode mode)
            throws IOException {

        String instanceKey = synchronizedGeneratorIdentity.getZNode();
        if (!instances.containsKey(instanceKey)) {
            logger.debug("Creating new instance.");
            instances.putIfAbsent(instanceKey, new BaseUniqueIDGenerator(synchronizedGeneratorIdentity, mode));
        }
        return instances.get(instanceKey);
    }
}
