package org.laoruga.dtogenerator.api.rules;

import org.laoruga.dtogenerator.api.rules.meta.Rule;
import org.laoruga.dtogenerator.constants.Boundary;
import org.laoruga.dtogenerator.constants.Bounds;
import org.laoruga.dtogenerator.constants.Group;

import java.lang.annotation.Repeatable;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.math.BigInteger;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.TYPE_USE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Put on the integral field to generate random integral value.
 * <p>
 * Supported field types:
 * <ul>
 *     <li>{@link Byte} and primitive {@code byte}</li>
 *     <li>{@link Short} and primitive {@code short}</li>
 *     <li>{@link Integer} and primitive {@code int}</li>
 *     <li>{@link Long} and primitive {@code long}</li>
 *     <li>{@link BigInteger}</li>
 *     <li>{@link AtomicInteger}</li>
 *     <li>{@link AtomicLong}</li>
 * </ul>
 * You may specify values range depended on field's type.
 *
 * @see Boundary
 * @see Group
 */

@Retention(RUNTIME)
@Target({FIELD, TYPE_USE})
@Rule
@Repeatable(IntegralRules.class)
public @interface IntegralRule {

    Class<?>[] GENERATED_TYPES = new Class<?>[]
            {Byte.class, Short.class, Integer.class, Long.class, BigInteger.class, AtomicInteger.class, AtomicLong.class};

    /**
     * Default value equals to maximum long value: 9,223,372,036,854,775,807
     *
     * @return maximum value of generated long
     */
    long maxLong() default Long.MAX_VALUE;

    /**
     * Default value equals to minimum long value: -9,223,372,036,854,775,808
     *
     * @return minimum value of generated long
     */
    long minLong() default Long.MIN_VALUE;

    /**
     * Default value equals to maximum integer value: 2,147,483,647
     *
     * @return maximum value of generated integer
     */
    int maxInt() default Integer.MAX_VALUE;

    /**
     * Default value equals to minimum integer value: -2,147,483,648
     *
     * @return minimum value of generated integer
     */
    int minInt() default Integer.MIN_VALUE;

    /**
     * Default value equals to maximum short value: 32767
     *
     * @return maximum value of generated short
     */
    short maxShort() default Short.MAX_VALUE;

    /**
     * Default value equals to minimum short value: -32768
     *
     * @return minimum value of generated short
     */
    short minShort() default Short.MIN_VALUE;

    /**
     * Default value equals to maximum byte value: 127
     *
     * @return maximum value of generated byte
     */
    byte maxByte() default Byte.MAX_VALUE;

    /**
     * Default value equals to minimum byte value: -128
     *
     * @return minimum value of generated byte
     */
    byte minByte() default Byte.MIN_VALUE;

    /**
     * Default value equals to maximum value of long type: 9,223,372,036,854,775,807
     *
     * @return maximum value of big integer byte
     */
    String maxBigInt() default Bounds.BIG_INTEGER_MAX_VALUE;

    /**
     * Default value equals to maximum value of long type but negative: -9,223,372,036,854,775,807
     *
     * @return minimum value of big integer byte
     */
    String minBigInt() default Bounds.BIG_INTEGER_MIN_VALUE;

    /**
     * @return boundary parameter
     * @see Boundary
     */
    Boundary boundary() default Boundary.RANDOM_VALUE;

    /**
     * @return group of the generators
     * @see Group
     */
    String group() default Group.DEFAULT;

}
