/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.storage.contrib.nio;

import com.google.cloud.storage.StorageOptions;
import com.google.cloud.storage.contrib.nio.CloudStorageConfiguration;
import com.google.cloud.storage.contrib.nio.CloudStorageFileAttributes;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystem;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystemProvider;
import com.google.cloud.storage.contrib.nio.CloudStorageOptions;
import com.google.cloud.storage.contrib.nio.CloudStoragePath;
import com.google.cloud.storage.contrib.nio.CloudStoragePseudoDirectoryException;
import com.google.cloud.storage.contrib.nio.testing.LocalStorageHelper;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import com.google.common.testing.NullPointerTester;
import com.google.common.truth.Truth;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.List;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class CloudStorageFileSystemProviderTest {
    private static final List<String> FILE_CONTENTS = ImmutableList.of((Object)"Fanatics have their dreams, wherewith they weave", (Object)"A paradise for a sect; the savage too", (Object)"From forth the loftiest fashion of his sleep", (Object)"Guesses at Heaven; pity these have not", (Object)"Trac'd upon vellum or wild Indian leaf", (Object)"The shadows of melodious utterance.", (Object)"But bare of laurel they live, dream, and die;", (Object)"For Poesy alone can tell her dreams,", (Object)"With the fine spell of words alone can save", (Object)"Imagination from the sable charm", (Object)"And dumb enchantment. Who alive can say,", (Object)"'Thou art no Poet may'st not tell thy dreams?'", (Object[])new String[]{"Since every man whose soul is not a clod", "Hath visions, and would speak, if he had loved", "And been well nurtured in his mother tongue.", "Whether the dream now purpos'd to rehearse", "Be poet's or fanatic's will be known", "When this warm scribe my hand is in the grave."});
    private static final String SINGULARITY = "A string";
    @Rule
    public final ExpectedException thrown = ExpectedException.none();

    @Before
    public void before() {
        CloudStorageFileSystemProvider.setStorageOptions((StorageOptions)LocalStorageHelper.getOptions());
    }

    @Test
    public void testSize() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Truth.assertThat((Long)Files.size(path)).isEqualTo((Object)SINGULARITY.getBytes(StandardCharsets.UTF_8).length);
    }

    @Test
    public void testSize_trailingSlash_returnsFakePseudoDirectorySize() throws Exception {
        Truth.assertThat((Long)Files.size(Paths.get(URI.create("gs://bucket/wat/")))).isEqualTo((Object)1);
    }

    @Test
    public void testSize_trailingSlash_disablePseudoDirectories() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.usePseudoDirectories(false));){
            CloudStoragePath path = fs.getPath("wat/", new String[0]);
            byte[] rapture = SINGULARITY.getBytes(StandardCharsets.UTF_8);
            Files.write((Path)path, rapture, new OpenOption[0]);
            Truth.assertThat((Long)Files.size((Path)path)).isEqualTo((Object)rapture.length);
        }
    }

    @Test
    public void testReadAllBytes() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Truth.assertThat((String)new String(Files.readAllBytes(path), StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
    }

    @Test
    public void testReadAllBytes_trailingSlash() throws Exception {
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        Files.readAllBytes(Paths.get(URI.create("gs://bucket/wat/")));
    }

    @Test
    public void testNewByteChannelRead() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        byte[] data = SINGULARITY.getBytes(StandardCharsets.UTF_8);
        Files.write(path, data, new OpenOption[0]);
        try (SeekableByteChannel input = Files.newByteChannel(path, new OpenOption[0]);){
            ByteBuffer buffer = ByteBuffer.allocate(data.length);
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)data.length);
            Truth.assertThat((String)new String(buffer.array(), StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
            buffer.rewind();
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)-1);
        }
    }

    @Test
    public void testNewByteChannelRead_seeking() throws Exception {
        Path path = Paths.get(URI.create("gs://lol/cat"));
        Files.write(path, "helloworld".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (SeekableByteChannel input = Files.newByteChannel(path, new OpenOption[0]);){
            ByteBuffer buffer = ByteBuffer.allocate(5);
            input.position(5L);
            Truth.assertThat((Long)input.position()).isEqualTo((Object)5);
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)5);
            Truth.assertThat((Long)input.position()).isEqualTo((Object)10);
            Truth.assertThat((String)new String(buffer.array(), StandardCharsets.UTF_8)).isEqualTo((Object)"world");
            buffer.rewind();
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)-1);
            input.position(0L);
            Truth.assertThat((Long)input.position()).isEqualTo((Object)0);
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)5);
            Truth.assertThat((Long)input.position()).isEqualTo((Object)5);
            Truth.assertThat((String)new String(buffer.array(), StandardCharsets.UTF_8)).isEqualTo((Object)"hello");
        }
    }

    @Test
    public void testNewByteChannelRead_seekBeyondSize_reportsEofOnNextRead() throws Exception {
        Path path = Paths.get(URI.create("gs://lol/cat"));
        Files.write(path, "hellocat".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (SeekableByteChannel input = Files.newByteChannel(path, new OpenOption[0]);){
            ByteBuffer buffer = ByteBuffer.allocate(5);
            input.position(10L);
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)-1);
            input.position(11L);
            Truth.assertThat((Integer)input.read(buffer)).isEqualTo((Object)-1);
            Truth.assertThat((Long)input.size()).isEqualTo((Object)8);
        }
    }

    @Test
    public void testNewByteChannelRead_trailingSlash() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat/"));
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        Files.newByteChannel(path, new OpenOption[0]);
    }

    @Test
    public void testNewByteChannelRead_notFound() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wednesday"));
        this.thrown.expect(NoSuchFileException.class);
        Files.newByteChannel(path, new OpenOption[0]);
    }

    @Test
    public void testNewByteChannelWrite() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/tests"));
        try (SeekableByteChannel output = Files.newByteChannel(path, StandardOpenOption.WRITE);){
            Truth.assertThat((Long)output.position()).isEqualTo((Object)0);
            Truth.assertThat((Long)output.size()).isEqualTo((Object)0);
            ByteBuffer buffer = ByteBuffer.wrap("filec".getBytes(StandardCharsets.UTF_8));
            Truth.assertThat((Integer)output.write(buffer)).isEqualTo((Object)5);
            Truth.assertThat((Long)output.position()).isEqualTo((Object)5);
            Truth.assertThat((Long)output.size()).isEqualTo((Object)5);
            buffer = ByteBuffer.wrap("onten".getBytes(StandardCharsets.UTF_8));
            Truth.assertThat((Integer)output.write(buffer)).isEqualTo((Object)5);
            Truth.assertThat((Long)output.position()).isEqualTo((Object)10);
            Truth.assertThat((Long)output.size()).isEqualTo((Object)10);
        }
        Truth.assertThat((String)new String(Files.readAllBytes(path), StandardCharsets.UTF_8)).isEqualTo((Object)"fileconten");
    }

    @Test
    public void testNewInputStream() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (InputStream input = Files.newInputStream(path, new OpenOption[0]);){
            byte[] data = new byte[SINGULARITY.getBytes(StandardCharsets.UTF_8).length];
            input.read(data);
            Truth.assertThat((String)new String(data, StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
        }
    }

    @Test
    public void testNewInputStream_trailingSlash() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat/"));
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        try (InputStream input = Files.newInputStream(path, new OpenOption[0]);){
            input.read();
        }
    }

    @Test
    public void testNewInputStream_notFound() throws Exception {
        Path path = Paths.get(URI.create("gs://cry/wednesday"));
        this.thrown.expect(NoSuchFileException.class);
        try (InputStream input = Files.newInputStream(path, new OpenOption[0]);){
            input.read();
        }
    }

    @Test
    public void testNewOutputStream() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (OutputStream output = Files.newOutputStream(path, new OpenOption[0]);){
            output.write(SINGULARITY.getBytes(StandardCharsets.UTF_8));
        }
        Truth.assertThat((String)new String(Files.readAllBytes(path), StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
    }

    @Test
    public void testNewOutputStream_truncateByDefault() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.write(path, "hello".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (OutputStream output = Files.newOutputStream(path, new OpenOption[0]);){
            output.write(SINGULARITY.getBytes(StandardCharsets.UTF_8));
        }
        Truth.assertThat((String)new String(Files.readAllBytes(path), StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
    }

    @Test
    public void testNewOutputStream_truncateExplicitly() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.write(path, "hello".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (OutputStream output = Files.newOutputStream(path, StandardOpenOption.TRUNCATE_EXISTING);){
            output.write(SINGULARITY.getBytes(StandardCharsets.UTF_8));
        }
        Truth.assertThat((String)new String(Files.readAllBytes(path), StandardCharsets.UTF_8)).isEqualTo((Object)SINGULARITY);
    }

    @Test
    public void testNewOutputStream_trailingSlash() throws Exception {
        Path path = Paths.get(URI.create("gs://bucket/wat/"));
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        Files.newOutputStream(path, new OpenOption[0]);
    }

    @Test
    public void testNewOutputStream_createNew() throws Exception {
        Path path = Paths.get(URI.create("gs://cry/wednesday"));
        Files.newOutputStream(path, StandardOpenOption.CREATE_NEW);
    }

    @Test
    public void testNewOutputStream_createNew_alreadyExists() throws Exception {
        Path path = Paths.get(URI.create("gs://cry/wednesday"));
        Files.write(path, SINGULARITY.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        this.thrown.expect(FileAlreadyExistsException.class);
        Files.newOutputStream(path, StandardOpenOption.CREATE_NEW);
    }

    @Test
    public void testWrite_objectNameWithExtraSlashes_throwsIae() throws Exception {
        Path path = Paths.get(URI.create("gs://double/slash//yep"));
        this.thrown.expect(IllegalArgumentException.class);
        Files.write(path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
    }

    @Test
    public void testWrite_objectNameWithExtraSlashes_canBeNormalized() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"greenbean", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.permitEmptyPathComponents(false));){
            CloudStoragePath path = fs.getPath("adipose//yep", new String[0]).normalize();
            Files.write((Path)path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
            Truth.assertThat(Files.readAllLines((Path)path, StandardCharsets.UTF_8)).isEqualTo(FILE_CONTENTS);
            Truth.assertThat((Boolean)Files.exists((Path)fs.getPath("adipose", new String[]{"yep"}), new LinkOption[0])).isTrue();
        }
    }

    @Test
    public void testWrite_objectNameWithExtraSlashes_permitEmptyPathComponents() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"greenbean", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.permitEmptyPathComponents(true));){
            CloudStoragePath path = fs.getPath("adipose//yep", new String[0]);
            Files.write((Path)path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
            Truth.assertThat(Files.readAllLines((Path)path, StandardCharsets.UTF_8)).isEqualTo(FILE_CONTENTS);
            Truth.assertThat((Boolean)Files.exists((Path)path, new LinkOption[0])).isTrue();
        }
    }

    @Test
    public void testWrite_absoluteObjectName_prefixSlashGetsRemoved() throws Exception {
        Path path = Paths.get(URI.create("gs://greenbean/adipose/yep"));
        Files.write(path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
        Truth.assertThat(Files.readAllLines(path, StandardCharsets.UTF_8)).isEqualTo(FILE_CONTENTS);
        Truth.assertThat((Boolean)Files.exists(path, new LinkOption[0])).isTrue();
    }

    @Test
    public void testWrite_absoluteObjectName_disableStrip_slashGetsPreserved() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"greenbean", (CloudStorageConfiguration)CloudStorageConfiguration.builder().stripPrefixSlash(false).build());){
            CloudStoragePath path = fs.getPath("/adipose/yep", new String[0]);
            Files.write((Path)path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
            Truth.assertThat(Files.readAllLines((Path)path, StandardCharsets.UTF_8)).isEqualTo(FILE_CONTENTS);
            Truth.assertThat((Boolean)Files.exists((Path)path, new LinkOption[0])).isTrue();
        }
    }

    @Test
    public void testWrite() throws Exception {
        Path path = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
        Truth.assertThat(Files.readAllLines(path, StandardCharsets.UTF_8)).isEqualTo(FILE_CONTENTS);
    }

    @Test
    public void testWriteOnClose() throws Exception {
        Path path = Paths.get(URI.create("gs://greenbean/adipose"));
        try (SeekableByteChannel chan = Files.newByteChannel(path, StandardOpenOption.WRITE);){
            for (int i = 0; i < 9999; ++i) {
                for (String s : FILE_CONTENTS) {
                    chan.write(ByteBuffer.wrap(s.getBytes(StandardCharsets.UTF_8)));
                }
            }
            try {
                Files.size(path);
                Truth.assertThat((Boolean)false).isTrue();
            }
            catch (NoSuchFileException noSuchFileException) {
                // empty catch block
            }
        }
        Truth.assertThat((Boolean)Files.exists(path, new LinkOption[0])).isTrue();
        Truth.assertThat((Long)Files.size(path)).isGreaterThan((Comparable)Long.valueOf(100L));
    }

    @Test
    public void testWrite_trailingSlash() throws Exception {
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        Files.write(Paths.get(URI.create("gs://greenbean/adipose/")), FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
    }

    @Test
    public void testExists() throws Exception {
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://military/fashion")), new LinkOption[0])).isFalse();
        Files.write(Paths.get(URI.create("gs://military/fashion")), "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://military/fashion")), new LinkOption[0])).isTrue();
    }

    @Test
    public void testExists_trailingSlash() {
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://military/fashion/")), new LinkOption[0])).isTrue();
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://military/fashion/.")), new LinkOption[0])).isTrue();
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://military/fashion/..")), new LinkOption[0])).isTrue();
    }

    @Test
    public void testExists_trailingSlash_disablePseudoDirectories() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"military", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.usePseudoDirectories(false));){
            Truth.assertThat((Boolean)Files.exists((Path)fs.getPath("fashion/", new String[0]), new LinkOption[0])).isFalse();
        }
    }

    @Test
    public void testDelete() throws Exception {
        Files.write(Paths.get(URI.create("gs://love/fashion")), "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://love/fashion")), new LinkOption[0])).isTrue();
        Files.delete(Paths.get(URI.create("gs://love/fashion")));
        Truth.assertThat((Boolean)Files.exists(Paths.get(URI.create("gs://love/fashion")), new LinkOption[0])).isFalse();
    }

    @Test
    public void testDelete_dotDirNotNormalized_throwsIae() throws Exception {
        this.thrown.expect(IllegalArgumentException.class);
        Files.delete(Paths.get(URI.create("gs://love/fly/../passion")));
    }

    @Test
    public void testDelete_trailingSlash() throws Exception {
        Files.delete(Paths.get(URI.create("gs://love/passion/")));
    }

    @Test
    public void testDelete_trailingSlash_disablePseudoDirectories() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"pumpkin", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.usePseudoDirectories(false));){
            CloudStoragePath path = fs.getPath("wat/", new String[0]);
            Files.write((Path)path, FILE_CONTENTS, StandardCharsets.UTF_8, new OpenOption[0]);
            Truth.assertThat((Boolean)Files.exists((Path)path, new LinkOption[0]));
            Files.delete((Path)path);
            Truth.assertThat((Boolean)(!Files.exists((Path)path, new LinkOption[0]) ? 1 : 0));
        }
    }

    @Test
    public void testDelete_notFound() throws Exception {
        this.thrown.expect(NoSuchFileException.class);
        Files.delete(Paths.get(URI.create("gs://loveh/passionehu")));
    }

    @Test
    public void testDeleteIfExists() throws Exception {
        Files.write(Paths.get(URI.create("gs://love/passionz")), "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Truth.assertThat((Boolean)Files.deleteIfExists(Paths.get(URI.create("gs://love/passionz")))).isTrue();
        Files.deleteIfExists(Paths.get(URI.create("gs://love/passion/")));
    }

    @Test
    public void testDeleteIfExists_trailingSlash_disablePseudoDirectories() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.usePseudoDirectories(false));){
            Files.deleteIfExists(Paths.get(URI.create("gs://love/passion/")));
        }
    }

    @Test
    public void testCopy() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.copy(source, target, new CopyOption[0]);
        Truth.assertThat((String)new String(Files.readAllBytes(target), StandardCharsets.UTF_8)).isEqualTo((Object)"(\u273f\u25d5 \u203f\u25d5 )\u30ce");
        Truth.assertThat((Boolean)Files.exists(source, new LinkOption[0])).isTrue();
        Truth.assertThat((Boolean)Files.exists(target, new LinkOption[0])).isTrue();
    }

    @Test
    public void testCopy_sourceMissing_throwsNoSuchFileException() throws Exception {
        this.thrown.expect(NoSuchFileException.class);
        Files.copy(Paths.get(URI.create("gs://military/fashion.show")), Paths.get(URI.create("gs://greenbean/adipose")), new CopyOption[0]);
    }

    @Test
    public void testCopy_targetExists_throwsFileAlreadyExistsException() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.write(target, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        this.thrown.expect(FileAlreadyExistsException.class);
        Files.copy(source, target, new CopyOption[0]);
    }

    @Test
    public void testCopyReplace_targetExists_works() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.write(target, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.copy(source, target, StandardCopyOption.REPLACE_EXISTING);
    }

    @Test
    public void testCopy_directory_doesNothing() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fundir/"));
        Path target = Paths.get(URI.create("gs://greenbean/loldir/"));
        Files.copy(source, target, new CopyOption[0]);
    }

    @Test
    public void testCopy_atomic_throwsUnsupported() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        this.thrown.expect(UnsupportedOperationException.class);
        Files.copy(source, target, StandardCopyOption.ATOMIC_MOVE);
    }

    @Test
    public void testMove() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.move(source, target, new CopyOption[0]);
        Truth.assertThat((String)new String(Files.readAllBytes(target), StandardCharsets.UTF_8)).isEqualTo((Object)"(\u273f\u25d5 \u203f\u25d5 )\u30ce");
        Truth.assertThat((Boolean)Files.exists(source, new LinkOption[0])).isFalse();
        Truth.assertThat((Boolean)Files.exists(target, new LinkOption[0])).isTrue();
    }

    @Test
    public void testCreateDirectory() throws Exception {
        Path path = Paths.get(URI.create("gs://greenbean/dir/"));
        Files.createDirectory(path, new FileAttribute[0]);
        Truth.assertThat((Boolean)Files.exists(path, new LinkOption[0])).isTrue();
    }

    @Test
    public void testMove_atomicMove_notSupported() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        this.thrown.expect(AtomicMoveNotSupportedException.class);
        Files.move(source, target, StandardCopyOption.ATOMIC_MOVE);
    }

    @Test
    public void testIsDirectory() throws Exception {
        try (FileSystem fs = FileSystems.getFileSystem(URI.create("gs://doodle"));){
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("/", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath(".", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("./", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("cat/..", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("hello/cat/..", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("cat/../", new String[0]), new LinkOption[0])).isTrue();
            Truth.assertThat((Boolean)Files.isDirectory(fs.getPath("hello/cat/../", new String[0]), new LinkOption[0])).isTrue();
        }
    }

    @Test
    public void testIsDirectory_trailingSlash_alwaysTrue() {
        Truth.assertThat((Boolean)Files.isDirectory(Paths.get(URI.create("gs://military/fundir/")), new LinkOption[0])).isTrue();
    }

    @Test
    public void testIsDirectory_trailingSlash_pseudoDirectoriesDisabled_false() throws Exception {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStorageFileSystemProviderTest.usePseudoDirectories(false));){
            Truth.assertThat((Boolean)Files.isDirectory((Path)fs.getPath("fundir/", new String[0]), new LinkOption[0])).isFalse();
        }
    }

    @Test
    public void testCopy_withCopyAttributes_preservesAttributes() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[]{CloudStorageOptions.withMimeType((String)"text/lolcat"), CloudStorageOptions.withCacheControl((String)"public; max-age=666"), CloudStorageOptions.withContentEncoding((String)"foobar"), CloudStorageOptions.withContentDisposition((String)"my-content-disposition"), CloudStorageOptions.withUserMetadata((String)"answer", (String)"42")});
        Files.copy(source, target, StandardCopyOption.COPY_ATTRIBUTES);
        CloudStorageFileAttributes attributes = Files.readAttributes(target, CloudStorageFileAttributes.class, new LinkOption[0]);
        Truth.assertThat((Optional)attributes.mimeType()).hasValue((Object)"text/lolcat");
        Truth.assertThat((Optional)attributes.cacheControl()).hasValue((Object)"public; max-age=666");
        Truth.assertThat((Optional)attributes.contentEncoding()).hasValue((Object)"foobar");
        Truth.assertThat((Optional)attributes.contentDisposition()).hasValue((Object)"my-content-disposition");
        Truth.assertThat((Boolean)attributes.userMetadata().containsKey("answer")).isTrue();
        Truth.assertThat((String)((String)attributes.userMetadata().get("answer"))).isEqualTo((Object)"42");
    }

    @Test
    public void testCopy_withoutOptions_doesntPreservesAttributes() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target = Paths.get(URI.create("gs://greenbean/adipose"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[]{CloudStorageOptions.withMimeType((String)"text/lolcat"), CloudStorageOptions.withCacheControl((String)"public; max-age=666"), CloudStorageOptions.withUserMetadata((String)"answer", (String)"42")});
        Files.copy(source, target, new CopyOption[0]);
        CloudStorageFileAttributes attributes = Files.readAttributes(target, CloudStorageFileAttributes.class, new LinkOption[0]);
        String mimeType = (String)attributes.mimeType().orNull();
        String cacheControl = (String)attributes.cacheControl().orNull();
        Truth.assertThat((String)mimeType).isNotEqualTo((Object)"text/lolcat");
        Truth.assertThat((String)cacheControl).isNull();
        Truth.assertThat((Boolean)attributes.userMetadata().containsKey("answer")).isFalse();
    }

    @Test
    public void testCopy_overwriteAttributes() throws Exception {
        Path source = Paths.get(URI.create("gs://military/fashion.show"));
        Path target1 = Paths.get(URI.create("gs://greenbean/adipose"));
        Path target2 = Paths.get(URI.create("gs://greenbean/round"));
        Files.write(source, "(\u273f\u25d5 \u203f\u25d5 )\u30ce".getBytes(StandardCharsets.UTF_8), new OpenOption[]{CloudStorageOptions.withMimeType((String)"text/lolcat"), CloudStorageOptions.withCacheControl((String)"public; max-age=666")});
        Files.copy(source, target1, StandardCopyOption.COPY_ATTRIBUTES);
        Files.copy(source, target2, new CopyOption[]{StandardCopyOption.COPY_ATTRIBUTES, CloudStorageOptions.withMimeType((String)"text/palfun")});
        CloudStorageFileAttributes attributes = Files.readAttributes(target1, CloudStorageFileAttributes.class, new LinkOption[0]);
        Truth.assertThat((Optional)attributes.mimeType()).hasValue((Object)"text/lolcat");
        Truth.assertThat((Optional)attributes.cacheControl()).hasValue((Object)"public; max-age=666");
        attributes = Files.readAttributes(target2, CloudStorageFileAttributes.class, new LinkOption[0]);
        Truth.assertThat((Optional)attributes.mimeType()).hasValue((Object)"text/palfun");
        Truth.assertThat((Optional)attributes.cacheControl()).hasValue((Object)"public; max-age=666");
    }

    @Test
    public void testNullness() throws Exception {
        try (FileSystem fs = FileSystems.getFileSystem(URI.create("gs://blood"));){
            NullPointerTester tester = new NullPointerTester();
            tester.ignore(CloudStorageFileSystemProvider.class.getMethod("equals", Object.class));
            tester.setDefault(URI.class, (Object)URI.create("gs://blood"));
            tester.setDefault(Path.class, (Object)fs.getPath("and/one", new String[0]));
            tester.setDefault(OpenOption.class, (Object)StandardOpenOption.CREATE);
            tester.setDefault(CopyOption.class, (Object)StandardCopyOption.COPY_ATTRIBUTES);
            tester.testAllPublicStaticMethods(CloudStorageFileSystemProvider.class);
            tester.testAllPublicInstanceMethods((Object)new CloudStorageFileSystemProvider());
        }
    }

    @Test
    public void testProviderEquals() {
        Path path1 = Paths.get(URI.create("gs://bucket/tuesday"));
        Path path2 = Paths.get(URI.create("gs://blood/wednesday"));
        Path path3 = Paths.get("tmp", new String[0]);
        Truth.assertThat((Object)path1.getFileSystem().provider()).isEqualTo((Object)path2.getFileSystem().provider());
        Truth.assertThat((Object)path1.getFileSystem().provider()).isNotEqualTo((Object)path3.getFileSystem().provider());
    }

    @Test
    public void testNewFileSystem() throws Exception {
        HashMap env = new HashMap();
        FileSystems.newFileSystem(URI.create("gs://bucket/path/to/file"), env);
    }

    @Test
    public void testFromSpace() throws Exception {
        Path path1 = Paths.get(URI.create("gs://bucket/with/a%20space"));
        CloudStorageFileSystemProvider provider = (CloudStorageFileSystemProvider)path1.getFileSystem().provider();
        CloudStoragePath path3 = provider.getPath("gs://bucket/with/a space");
        Truth.assertThat((Object)path1.getFileSystem().provider()).isEqualTo((Object)path3.getFileSystem().provider());
        Truth.assertThat((Comparable)path1.toUri()).isEqualTo((Object)path3.toUri());
        CloudStoragePath path4 = provider.getPath("gs://bucket/with/a%20percent");
        Truth.assertThat((String)path4.toString()).isEqualTo((Object)"/with/a%20percent");
    }

    private static CloudStorageConfiguration permitEmptyPathComponents(boolean value) {
        return CloudStorageConfiguration.builder().permitEmptyPathComponents(value).build();
    }

    private static CloudStorageConfiguration usePseudoDirectories(boolean value) {
        return CloudStorageConfiguration.builder().usePseudoDirectories(value).build();
    }
}

