/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.storage.contrib.nio;

import com.google.cloud.storage.StorageOptions;
import com.google.cloud.storage.contrib.nio.CloudStorageConfiguration;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystem;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystemProvider;
import com.google.cloud.storage.contrib.nio.CloudStoragePseudoDirectoryException;
import com.google.cloud.storage.contrib.nio.CloudStorageReadChannel;
import com.google.cloud.storage.contrib.nio.testing.LocalStorageHelper;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.NullPointerTester;
import com.google.common.truth.Truth;
import java.io.IOException;
import java.net.URI;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class CloudStorageFileSystemTest {
    private static final String ALONE = "To be, or not to be, that is the question\u2014\nWhether 'tis Nobler in the mind to suffer\nThe Slings and Arrows of outrageous Fortune,\nOr to take Arms against a Sea of troubles,\nAnd by opposing, end them? To die, to sleep\u2014\nNo more; and by a sleep, to say we end\nThe Heart-ache, and the thousand Natural shocks\nThat Flesh is heir to? 'Tis a consummation\n";
    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Before
    public void before() {
        CloudStorageFileSystemProvider.setStorageOptions((StorageOptions)LocalStorageHelper.getOptions());
    }

    @Test
    public void checkDefaultOptions() throws IOException {
        Path path = Paths.get(URI.create("gs://bucket/file"));
        CloudStorageFileSystem gcs = (CloudStorageFileSystem)path.getFileSystem();
        Truth.assertThat((Integer)gcs.config().maxChannelReopens()).isEqualTo((Object)0);
        CloudStorageFileSystemProvider.setDefaultCloudStorageConfiguration((CloudStorageConfiguration)CloudStorageConfiguration.builder().maxChannelReopens(123).build());
        Path path2 = Paths.get(URI.create("gs://newbucket/file"));
        CloudStorageFileSystem gcs2 = (CloudStorageFileSystem)path2.getFileSystem();
        Truth.assertThat((Integer)gcs2.config().maxChannelReopens()).isEqualTo((Object)123);
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            CloudStorageFileSystem csfs = fs;
            Truth.assertThat((Integer)csfs.config().maxChannelReopens()).isEqualTo((Object)123);
            Files.write(fs.getPath("/angel", new String[0]), ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            path2 = Paths.get(URI.create("gs://bucket/angel"));
            try (SeekableByteChannel seekableByteChannel = Files.newByteChannel(path2, new OpenOption[0]);){
                CloudStorageReadChannel cloudChannel = (CloudStorageReadChannel)seekableByteChannel;
                Truth.assertThat((Integer)cloudChannel.maxChannelReopens).isEqualTo((Object)123);
            }
        }
        CloudStorageFileSystemProvider.setDefaultCloudStorageConfiguration(null);
        Path path3 = Paths.get(URI.create("gs://newbucket/file"));
        CloudStorageFileSystem gcs3 = (CloudStorageFileSystem)path3.getFileSystem();
        Truth.assertThat((Integer)gcs3.config().maxChannelReopens()).isEqualTo((Object)0);
    }

    @Test
    public void canOverrideDefaultOptions() throws IOException {
        CloudStorageFileSystemProvider.setDefaultCloudStorageConfiguration((CloudStorageConfiguration)CloudStorageConfiguration.builder().maxChannelReopens(123).build());
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket", (CloudStorageConfiguration)CloudStorageConfiguration.builder().maxChannelReopens(7).build());){
            CloudStorageFileSystem csfs = fs;
            Truth.assertThat((Integer)csfs.config().maxChannelReopens()).isEqualTo((Object)7);
        }
        CloudStorageFileSystemProvider.setDefaultCloudStorageConfiguration(null);
    }

    @Test
    public void testGetPath() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Truth.assertThat((String)fs.getPath("/angel", new String[0]).toString()).isEqualTo((Object)"/angel");
            Truth.assertThat((String)fs.getPath("/angel", new String[0]).toUri().toString()).isEqualTo((Object)"gs://bucket/angel");
        }
    }

    @Test
    public void testWrite() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Files.write(fs.getPath("/angel", new String[0]), ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
        Truth.assertThat((String)new String(Files.readAllBytes(Paths.get(URI.create("gs://bucket/angel"))), StandardCharsets.UTF_8)).isEqualTo((Object)ALONE);
    }

    @Test
    public void testRead() throws IOException {
        Files.write(Paths.get(URI.create("gs://bucket/angel")), ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Truth.assertThat((String)new String(Files.readAllBytes(fs.getPath("/angel", new String[0])), StandardCharsets.UTF_8)).isEqualTo((Object)ALONE);
        }
    }

    @Test
    public void testExists_false() throws IOException {
        try (FileSystem fs = FileSystems.getFileSystem(URI.create("gs://bucket"));){
            Truth.assertThat((Boolean)Files.exists(fs.getPath("/angel", new String[0]), new LinkOption[0])).isFalse();
        }
    }

    @Test
    public void testExists_true() throws IOException {
        Files.write(Paths.get(URI.create("gs://bucket/angel")), ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Truth.assertThat((Boolean)Files.exists(fs.getPath("/angel", new String[0]), new LinkOption[0])).isTrue();
        }
    }

    @Test
    public void testGetters() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Truth.assertThat((Boolean)fs.isOpen()).isTrue();
            Truth.assertThat((Boolean)fs.isReadOnly()).isFalse();
            Truth.assertThat(fs.getRootDirectories()).containsExactly(new Object[]{fs.getPath("/", new String[0])});
            Truth.assertThat(fs.getFileStores()).isEmpty();
            Truth.assertThat((String)fs.getSeparator()).isEqualTo((Object)"/");
            Truth.assertThat(fs.supportedFileAttributeViews()).containsExactly(new Object[]{"basic", "gcs"});
        }
    }

    @Test
    public void testEquals() throws IOException {
        try (CloudStorageFileSystem bucket1 = CloudStorageFileSystem.forBucket((String)"bucket");
             FileSystem bucket2 = FileSystems.getFileSystem(URI.create("gs://bucket"));
             CloudStorageFileSystem doge1 = CloudStorageFileSystem.forBucket((String)"doge");
             FileSystem doge2 = FileSystems.getFileSystem(URI.create("gs://doge"));){
            new EqualsTester().addEqualityGroup(new Object[]{bucket1, bucket2}).addEqualityGroup(new Object[]{doge1, doge2}).testEquals();
        }
    }

    @Test
    public void testNullness() throws IOException, NoSuchMethodException, SecurityException {
        try (FileSystem fs = FileSystems.getFileSystem(URI.create("gs://bucket"));){
            NullPointerTester tester = new NullPointerTester().ignore(CloudStorageFileSystem.class.getMethod("equals", Object.class)).setDefault(CloudStorageConfiguration.class, (Object)CloudStorageConfiguration.DEFAULT).setDefault(StorageOptions.class, (Object)LocalStorageHelper.getOptions());
            tester.testAllPublicStaticMethods(CloudStorageFileSystem.class);
            tester.testAllPublicInstanceMethods((Object)fs);
        }
    }

    @Test
    public void testListFiles() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            ArrayList<Path> goodPaths = new ArrayList<Path>();
            ArrayList<Path> paths = new ArrayList<Path>();
            goodPaths.add(fs.getPath("dir/angel", new String[0]));
            goodPaths.add(fs.getPath("dir/alone", new String[0]));
            paths.add(fs.getPath("dir/dir2/another_angel", new String[0]));
            paths.add(fs.getPath("atroot", new String[0]));
            paths.addAll(goodPaths);
            goodPaths.add(fs.getPath("dir/dir2/", new String[0]));
            for (Path path : paths) {
                Files.write(path, ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            ArrayList<Path> got = new ArrayList<Path>();
            for (Path path : Files.newDirectoryStream(fs.getPath("/dir/", new String[0]))) {
                got.add(path);
            }
            Truth.assertThat(got).containsExactlyElementsIn(goodPaths);
            got.clear();
            for (Path path : Files.newDirectoryStream(fs.getPath("dir/", new String[0]))) {
                got.add(path);
            }
            Truth.assertThat(got).containsExactlyElementsIn(goodPaths);
        }
    }

    @Test
    public void testMatcher() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            String pattern1 = "glob:*.java";
            PathMatcher javaFileMatcher = fs.getPathMatcher(pattern1);
            this.assertMatches((FileSystem)fs, javaFileMatcher, "a.java", true);
            this.assertMatches((FileSystem)fs, javaFileMatcher, "a.text", false);
            this.assertMatches((FileSystem)fs, javaFileMatcher, "folder/c.java", true);
            this.assertMatches((FileSystem)fs, javaFileMatcher, "d", false);
            String pattern2 = "glob:*.{java,text}";
            PathMatcher javaAndTextFileMatcher = fs.getPathMatcher(pattern2);
            this.assertMatches((FileSystem)fs, javaAndTextFileMatcher, "a.java", true);
            this.assertMatches((FileSystem)fs, javaAndTextFileMatcher, "a.text", true);
            this.assertMatches((FileSystem)fs, javaAndTextFileMatcher, "folder/c.java", true);
            this.assertMatches((FileSystem)fs, javaAndTextFileMatcher, "d", false);
        }
    }

    @Test
    public void testDeleteEmptyFolder() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            ArrayList<Path> paths = new ArrayList<Path>();
            paths.add(fs.getPath("dir/angel", new String[0]));
            paths.add(fs.getPath("dir/dir2/another_angel", new String[0]));
            paths.add(fs.getPath("atroot", new String[0]));
            for (Path path : paths) {
                Files.write(path, ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            Files.delete(fs.getPath("ghost/", new String[0]));
            Files.delete(fs.getPath("dir/ghost/", new String[0]));
            Files.delete(fs.getPath("dir/dir2/ghost/", new String[0]));
            Files.deleteIfExists(fs.getPath("ghost/", new String[0]));
            Files.deleteIfExists(fs.getPath("dir/ghost/", new String[0]));
            Files.deleteIfExists(fs.getPath("dir/dir2/ghost/", new String[0]));
        }
    }

    @Test
    public void testDeleteFullFolder() throws IOException {
        this.thrown.expect(CloudStoragePseudoDirectoryException.class);
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            Files.write(fs.getPath("dir/angel", new String[0]), ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            Files.delete(fs.getPath("dir/", new String[0]));
        }
    }

    @Test
    public void testDelete() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            ArrayList<Path> paths = new ArrayList<Path>();
            paths.add(fs.getPath("dir/angel", new String[0]));
            paths.add(fs.getPath("dir/dir2/another_angel", new String[0]));
            paths.add(fs.getPath("atroot", new String[0]));
            for (Path path : paths) {
                Files.write(path, ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            Files.delete(fs.getPath("atroot", new String[0]));
            Files.delete(fs.getPath("dir/angel", new String[0]));
            Files.deleteIfExists(fs.getPath("dir/dir2/another_angel", new String[0]));
            for (Path path : paths) {
                Truth.assertThat((Boolean)Files.exists(path, new LinkOption[0])).isFalse();
            }
        }
    }

    @Test
    public void testDeleteEmptiedFolder() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            ArrayList<Path> paths = new ArrayList<Path>();
            paths.add(fs.getPath("dir/angel", new String[0]));
            paths.add(fs.getPath("dir/dir2/another_angel", new String[0]));
            for (Path path : paths) {
                Files.write(path, ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            Files.delete(fs.getPath("dir/angel", new String[0]));
            Files.deleteIfExists(fs.getPath("dir/dir2/another_angel", new String[0]));
            Path dir2 = fs.getPath("dir/dir2/", new String[0]);
            Files.deleteIfExists(dir2);
            Path dir = fs.getPath("dir/", new String[0]);
            Files.deleteIfExists(dir);
        }
    }

    @Test
    public void testDeleteRecursive() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"bucket");){
            ArrayList<Path> paths = new ArrayList<Path>();
            paths.add(fs.getPath("atroot", new String[0]));
            paths.add(fs.getPath("dir/angel", new String[0]));
            paths.add(fs.getPath("dir/dir2/another_angel", new String[0]));
            paths.add(fs.getPath("dir/dir2/angel3", new String[0]));
            paths.add(fs.getPath("dir/dir3/cloud", new String[0]));
            for (Path path : paths) {
                Files.write(path, ALONE.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            CloudStorageFileSystemTest.deleteRecursive(fs.getPath("dir/", new String[0]));
            Truth.assertThat((Boolean)Files.exists(fs.getPath("dir/angel", new String[0]), new LinkOption[0])).isFalse();
            Truth.assertThat((Boolean)Files.exists(fs.getPath("dir/dir3/cloud", new String[0]), new LinkOption[0])).isFalse();
            Truth.assertThat((Boolean)Files.exists(fs.getPath("atroot", new String[0]), new LinkOption[0])).isTrue();
        }
    }

    private static void deleteRecursive(Path directory) throws IOException {
        Files.walkFileTree(directory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void assertMatches(FileSystem fs, PathMatcher matcher, String toMatch, boolean expected) {
        Truth.assertThat((Boolean)matcher.matches(fs.getPath(toMatch, new String[0]).getFileName())).isEqualTo((Object)expected);
    }
}

