/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.storage.contrib.nio;

import com.google.cloud.storage.StorageOptions;
import com.google.cloud.storage.contrib.nio.CloudStorageConfiguration;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystem;
import com.google.cloud.storage.contrib.nio.CloudStorageFileSystemProvider;
import com.google.cloud.storage.contrib.nio.CloudStoragePath;
import com.google.cloud.storage.contrib.nio.testing.LocalStorageHelper;
import com.google.common.collect.Iterables;
import com.google.common.testing.EqualsTester;
import com.google.common.testing.NullPointerTester;
import com.google.common.truth.Truth;
import java.io.IOException;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.ProviderMismatchException;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class CloudStoragePathTest {
    @Rule
    public final ExpectedException thrown = ExpectedException.none();

    @Before
    public void before() {
        CloudStorageFileSystemProvider.setStorageOptions((StorageOptions)LocalStorageHelper.getOptions());
    }

    @Test
    public void testCreate_neverRemoveExtraSlashes() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("lol//cat", new String[0]).toString()).isEqualTo((Object)"lol//cat");
            Truth.assertThat((Object)fs.getPath("lol//cat", new String[0])).isEqualTo((Object)fs.getPath("lol//cat", new String[0]));
        }
    }

    @Test
    public void testCreate_preservesTrailingSlash() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("lol/cat/", new String[0]).toString()).isEqualTo((Object)"lol/cat/");
            Truth.assertThat((Object)fs.getPath("lol/cat/", new String[0])).isEqualTo((Object)fs.getPath("lol/cat/", new String[0]));
        }
    }

    @Test
    public void testGetGcsFilename_empty_notAllowed() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            fs.getPath("", new String[0]).getBlobId();
        }
    }

    @Test
    public void testGetGcsFilename_stripsPrefixSlash() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/hi", new String[0]).getBlobId().getName()).isEqualTo((Object)"hi");
        }
    }

    @Test
    public void testGetGcsFilename_overrideStripPrefixSlash_doesntStripPrefixSlash() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.stripPrefixSlash(false));){
            Truth.assertThat((String)fs.getPath("/hi", new String[0]).getBlobId().getName()).isEqualTo((Object)"/hi");
        }
    }

    @Test
    public void testGetGcsFilename_extraSlashes_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            fs.getPath("a//b", new String[0]).getBlobId().getName();
        }
    }

    @Test
    public void testGetGcsFilename_overridepermitEmptyPathComponents() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.permitEmptyPathComponents(true));){
            Truth.assertThat((String)fs.getPath("a//b", new String[0]).getBlobId().getName()).isEqualTo((Object)"a//b");
        }
    }

    @Test
    public void testGetGcsFilename_freaksOutOnExtraSlashesAndDotDirs() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            fs.getPath("a//b/..", new String[0]).getBlobId().getName();
        }
    }

    @Test
    public void testNameCount() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Integer)fs.getPath("", new String[0]).getNameCount()).isEqualTo((Object)1);
            Truth.assertThat((Integer)fs.getPath("/", new String[0]).getNameCount()).isEqualTo((Object)0);
            Truth.assertThat((Integer)fs.getPath("/hi/", new String[0]).getNameCount()).isEqualTo((Object)1);
            Truth.assertThat((Integer)fs.getPath("/hi/yo", new String[0]).getNameCount()).isEqualTo((Object)2);
            Truth.assertThat((Integer)fs.getPath("hi/yo", new String[0]).getNameCount()).isEqualTo((Object)2);
        }
    }

    @Test
    public void testGetName() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("", new String[0]).getName(0).toString()).isEqualTo((Object)"");
            Truth.assertThat((String)fs.getPath("/hi", new String[0]).getName(0).toString()).isEqualTo((Object)"hi");
            Truth.assertThat((String)fs.getPath("hi/there", new String[0]).getName(1).toString()).isEqualTo((Object)"there");
        }
    }

    @Test
    public void testGetName_negative_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            fs.getPath("angel", new String[0]).getName(-1);
        }
    }

    @Test
    public void testGetName_overflow_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            fs.getPath("angel", new String[0]).getName(1);
        }
    }

    @Test
    public void testIterator() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)((Path)Iterables.get((Iterable)fs.getPath("/dog/mog", new String[0]), (int)0)).toString()).isEqualTo((Object)"dog");
            Truth.assertThat((String)((Path)Iterables.get((Iterable)fs.getPath("/dog/mog", new String[0]), (int)1)).toString()).isEqualTo((Object)"mog");
            Truth.assertThat((Integer)Iterables.size((Iterable)fs.getPath("/", new String[0]))).isEqualTo((Object)0);
            Truth.assertThat((Integer)Iterables.size((Iterable)fs.getPath("", new String[0]))).isEqualTo((Object)1);
            Truth.assertThat((String)((Path)Iterables.get((Iterable)fs.getPath("", new String[0]), (int)0)).toString()).isEqualTo((Object)"");
        }
    }

    @Test
    public void testNormalize() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/", new String[0]).normalize().toString()).isEqualTo((Object)"/");
            Truth.assertThat((String)fs.getPath("a/x/../b/x/..", new String[0]).normalize().toString()).isEqualTo((Object)"a/b/");
            Truth.assertThat((String)fs.getPath("/x/x/../../\u2661", new String[0]).normalize().toString()).isEqualTo((Object)"/\u2661");
            Truth.assertThat((String)fs.getPath("/x/x/./.././.././\u2661", new String[0]).normalize().toString()).isEqualTo((Object)"/\u2661");
        }
    }

    @Test
    public void testNormalize_dot_becomesBlank() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("", new String[0]).normalize().toString()).isEqualTo((Object)"");
            Truth.assertThat((String)fs.getPath(".", new String[0]).normalize().toString()).isEqualTo((Object)"");
        }
    }

    @Test
    public void testNormalize_trailingSlash_isPreserved() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("o/", new String[0]).normalize().toString()).isEqualTo((Object)"o/");
        }
    }

    @Test
    public void testNormalize_doubleDot_becomesBlank() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("..", new String[0]).normalize().toString()).isEqualTo((Object)"");
            Truth.assertThat((String)fs.getPath("../..", new String[0]).normalize().toString()).isEqualTo((Object)"");
        }
    }

    @Test
    public void testNormalize_extraSlashes_getRemoved() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("//life///b/good//", new String[0]).normalize().toString()).isEqualTo((Object)"/life/b/good/");
        }
    }

    @Test
    public void testToRealPath_hasDotDir_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            fs.getPath("a/hi./b", new String[0]).toRealPath(new LinkOption[0]);
            fs.getPath("a/.hi/b", new String[0]).toRealPath(new LinkOption[0]);
            this.thrown.expect(IllegalArgumentException.class);
            this.thrown.expectMessage("dot-dir");
            fs.getPath("a/./b", new String[0]).toRealPath(new LinkOption[0]);
        }
    }

    @Test
    public void testToRealPath_hasDotDotDir_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            fs.getPath("a/hi../b", new String[0]).toRealPath(new LinkOption[0]);
            fs.getPath("a/..hi/b", new String[0]).toRealPath(new LinkOption[0]);
            this.thrown.expect(IllegalArgumentException.class);
            this.thrown.expectMessage("dot-dir");
            fs.getPath("a/../b", new String[0]).toRealPath(new LinkOption[0]);
        }
    }

    @Test
    public void testToRealPath_extraSlashes_throwsIae() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(IllegalArgumentException.class);
            this.thrown.expectMessage("extra slashes");
            fs.getPath("a//b", new String[0]).toRealPath(new LinkOption[0]);
        }
    }

    @Test
    public void testToRealPath_overridePermitEmptyPathComponents_extraSlashes_slashesRemain() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.permitEmptyPathComponents(true));){
            Truth.assertThat((String)fs.getPath("/life///b/./good/", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"life///b/./good/");
        }
    }

    @Test
    public void testToRealPath_permitEmptyPathComponents_doesNotNormalize() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.permitEmptyPathComponents(true));){
            Truth.assertThat((String)fs.getPath("a", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"a");
            Truth.assertThat((String)fs.getPath("a//b", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"a//b");
            Truth.assertThat((String)fs.getPath("a//./b//..", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"a//./b//..");
        }
    }

    @Test
    public void testToRealPath_withWorkingDirectory_makesAbsolute() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.workingDirectory("/lol"));){
            Truth.assertThat((String)fs.getPath("a", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"lol/a");
        }
    }

    @Test
    public void testToRealPath_disableStripPrefixSlash_makesPathAbsolute() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.stripPrefixSlash(false));){
            Truth.assertThat((String)fs.getPath("a", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"/a");
            Truth.assertThat((String)fs.getPath("/a", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"/a");
        }
    }

    @Test
    public void testToRealPath_trailingSlash_getsPreserved() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("a/b/", new String[0]).toRealPath(new LinkOption[0]).toString()).isEqualTo((Object)"a/b/");
        }
    }

    @Test
    public void testNormalize_empty_returnsEmpty() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("", new String[0]).normalize().toString()).isEqualTo((Object)"");
        }
    }

    @Test
    public void testNormalize_preserveTrailingSlash() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("a/b/../c/", new String[0]).normalize().toString()).isEqualTo((Object)"a/c/");
            Truth.assertThat((String)fs.getPath("a/b/./c/", new String[0]).normalize().toString()).isEqualTo((Object)"a/b/c/");
        }
    }

    @Test
    public void testGetParent_preserveTrailingSlash() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("a/b/c", new String[0]).getParent().toString()).isEqualTo((Object)"a/b/");
            Truth.assertThat((String)fs.getPath("a/b/c/", new String[0]).getParent().toString()).isEqualTo((Object)"a/b/");
            Truth.assertThat((Object)fs.getPath("", new String[0]).getParent()).isNull();
            Truth.assertThat((Object)fs.getPath("/", new String[0]).getParent()).isNull();
            Truth.assertThat((Object)fs.getPath("aaa", new String[0]).getParent()).isNull();
            Truth.assertThat((Object)fs.getPath("aaa/", new String[0]).getParent()).isNull();
        }
    }

    @Test
    public void testGetRoot() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/hello", new String[0]).getRoot().toString()).isEqualTo((Object)"/");
            Truth.assertThat((Object)fs.getPath("hello", new String[0]).getRoot()).isNull();
        }
    }

    @Test
    public void testRelativize() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/foo/bar/lol/cat", new String[0]).relativize((Path)fs.getPath("/foo/a/b/../../c", new String[0])).toString()).isEqualTo((Object)"../../../a/b/../../c");
        }
    }

    @Test
    public void testRelativize_providerMismatch() throws IOException {
        try (CloudStorageFileSystem gcs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(ProviderMismatchException.class);
            gcs.getPath("/etc", new String[0]).relativize(FileSystems.getDefault().getPath("/dog", new String[0]));
        }
    }

    @Test
    public void testRelativize_providerMismatch2() throws IOException {
        try (CloudStorageFileSystem gcs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(ProviderMismatchException.class);
            gcs.getPath("/dog", new String[0]).relativize(FileSystems.getDefault().getPath("/etc", new String[0]));
        }
    }

    @Test
    public void testResolve() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/hi", new String[0]).resolve("there").toString()).isEqualTo((Object)"/hi/there");
            Truth.assertThat((String)fs.getPath("hi", new String[0]).resolve("there").toString()).isEqualTo((Object)"hi/there");
        }
    }

    @Test
    public void testResolve_providerMismatch() throws IOException {
        try (CloudStorageFileSystem gcs = CloudStorageFileSystem.forBucket((String)"doodle");){
            this.thrown.expect(ProviderMismatchException.class);
            gcs.getPath("etc", new String[0]).resolve(FileSystems.getDefault().getPath("/dog", new String[0]));
        }
    }

    @Test
    public void testIsAbsolute() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Boolean)fs.getPath("/hi", new String[0]).isAbsolute()).isTrue();
            Truth.assertThat((Boolean)fs.getPath("hi", new String[0]).isAbsolute()).isFalse();
        }
    }

    @Test
    public void testToAbsolutePath() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Object)fs.getPath("/hi", new String[0]).toAbsolutePath()).isEqualTo((Object)fs.getPath("/hi", new String[0]));
            Truth.assertThat((Object)fs.getPath("hi", new String[0]).toAbsolutePath()).isEqualTo((Object)fs.getPath("/hi", new String[0]));
        }
    }

    @Test
    public void testToAbsolutePath_withWorkingDirectory() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.workingDirectory("/lol"));){
            Truth.assertThat((String)fs.getPath("a", new String[0]).toAbsolutePath().toString()).isEqualTo((Object)"/lol/a");
        }
    }

    @Test
    public void testGetFileName() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((String)fs.getPath("/hi/there", new String[0]).getFileName().toString()).isEqualTo((Object)"there");
            Truth.assertThat((String)fs.getPath("military/fashion/show", new String[0]).getFileName().toString()).isEqualTo((Object)"show");
        }
    }

    @Test
    public void testCompareTo() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Integer)fs.getPath("/hi/there", new String[0]).compareTo((Path)fs.getPath("/hi/there", new String[0]))).isEqualTo((Object)0);
            Truth.assertThat((Integer)fs.getPath("/hi/there", new String[0]).compareTo((Path)fs.getPath("/hi/therf", new String[0]))).isEqualTo((Object)-1);
            Truth.assertThat((Integer)fs.getPath("/hi/there", new String[0]).compareTo((Path)fs.getPath("/hi/therd", new String[0]))).isEqualTo((Object)1);
        }
    }

    @Test
    public void testStartsWith() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("/hi/there", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("/hi/therf", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("/hi", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("/hi/", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("hi", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("/", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).startsWith((Path)fs.getPath("", new String[0]))).isFalse();
        }
    }

    @Test
    public void testEndsWith() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("there", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("therf", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("/blag/therf", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("/hi/there", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("/there", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/human/that/you/cry", new String[0]).endsWith((Path)fs.getPath("that/you/cry", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/human/that/you/cry", new String[0]).endsWith((Path)fs.getPath("that/you/cry/", new String[0]))).isTrue();
            Truth.assertThat((Boolean)fs.getPath("/hi/there/", new String[0]).endsWith((Path)fs.getPath("/", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("/hi/there", new String[0]).endsWith((Path)fs.getPath("", new String[0]))).isFalse();
            Truth.assertThat((Boolean)fs.getPath("", new String[0]).endsWith((Path)fs.getPath("", new String[0]))).isTrue();
        }
    }

    @Test
    public void testResolve_willWorkWithRecursiveCopy() throws IOException {
        try (FileSystem fsSource = FileSystems.getFileSystem(URI.create("gs://hello"));
             FileSystem fsTarget = FileSystems.getFileSystem(URI.create("gs://cat"));){
            Path targetPath = fsTarget.getPath("/some/folder/", new String[0]);
            Path relSrcPath = fsSource.getPath("file.txt", new String[0]);
            Truth.assertThat((Object)targetPath.resolve(relSrcPath)).isEqualTo((Object)fsTarget.getPath("/some/folder/file.txt", new String[0]));
        }
    }

    @Test
    public void testRelativize_willWorkWithRecursiveCopy() throws IOException {
        try (FileSystem fsSource = FileSystems.getFileSystem(URI.create("gs://hello"));
             FileSystem fsTarget = FileSystems.getFileSystem(URI.create("gs://cat"));){
            Path targetPath = fsTarget.getPath("/some/folder/", new String[0]);
            Path sourcePath = fsSource.getPath("/sloth/", new String[0]);
            Path file = fsSource.getPath("/sloth/file.txt", new String[0]);
            Truth.assertThat((Object)targetPath.resolve(sourcePath.relativize(file))).isEqualTo((Object)fsTarget.getPath("/some/folder/file.txt", new String[0]));
        }
    }

    @Test
    public void testToFile_unsupported() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            CloudStoragePath path = fs.getPath("/lol", new String[0]);
            this.thrown.expect(UnsupportedOperationException.class);
            path.toFile();
        }
    }

    @Test
    public void testEquals() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            new EqualsTester().addEqualityGroup(new Object[]{fs.getPath("/hello/cat", new String[0]), fs.getPath("/hello/cat", new String[0])}).addEqualityGroup(new Object[]{fs.getPath("great/commandment", new String[0]), fs.getPath("/great/commandment", new String[0])}).addEqualityGroup(new Object[]{fs.getPath("great/commandment/", new String[0]), fs.getPath("/great/commandment/", new String[0])}).addEqualityGroup(new Object[]{fs.getPath("foo/../bar", new String[0]), fs.getPath("foo/../bar", new String[0])}).addEqualityGroup(new Object[]{fs.getPath("bar", new String[0])}).testEquals();
        }
    }

    @Test
    public void testEquals_currentDirectoryIsTakenIntoConsideration() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle", (CloudStorageConfiguration)CloudStoragePathTest.workingDirectory("/hello"));){
            new EqualsTester().addEqualityGroup(new Object[]{fs.getPath("cat", new String[0]), fs.getPath("/hello/cat", new String[0])}).addEqualityGroup(new Object[]{fs.getPath("", new String[0]), fs.getPath("/hello", new String[0])}).testEquals();
        }
    }

    @Test
    public void testNullness() throws IOException, NoSuchMethodException, SecurityException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            NullPointerTester tester = new NullPointerTester();
            tester.ignore(CloudStoragePath.class.getMethod("equals", Object.class));
            tester.setDefault(Path.class, (Object)fs.getPath("sup", new String[0]));
            tester.testAllPublicStaticMethods(CloudStoragePath.class);
            tester.testAllPublicInstanceMethods((Object)fs.getPath("sup", new String[0]));
        }
    }

    @Test
    public void testSpaces() throws IOException {
        try (CloudStorageFileSystem fs = CloudStorageFileSystem.forBucket((String)"doodle");){
            CloudStoragePath path = fs.getPath("/with/a space", new String[0]);
            String toUri = URLDecoder.decode(path.toUri().toString(), "UTF-8");
            Truth.assertThat((String)toUri).isEqualTo((Object)"gs://doodle/with/a space");
            CloudStoragePath path2 = fs.getPath("/with/a%20percent", new String[0]);
            String toUri2 = URLDecoder.decode(path2.toUri().toString(), "UTF-8");
            Truth.assertThat((String)toUri2).isEqualTo((Object)"gs://doodle/with/a%20percent");
        }
    }

    private static CloudStorageConfiguration stripPrefixSlash(boolean value) {
        return CloudStorageConfiguration.builder().stripPrefixSlash(value).build();
    }

    private static CloudStorageConfiguration permitEmptyPathComponents(boolean value) {
        return CloudStorageConfiguration.builder().permitEmptyPathComponents(value).build();
    }

    private static CloudStorageConfiguration workingDirectory(String value) {
        return CloudStorageConfiguration.builder().workingDirectory(value).build();
    }
}

