/*
 * Copyright 2019 Abdulla Abdurakhmanov (abdulla@latestbit.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 */

package org.latestbit.slack.morphism.concurrent

import cats._
import org.latestbit.slack.morphism.concurrent.impl.AsyncSeqIteratorImpl

import scala.language.implicitConversions

/**
 * Provides infinite and async computation results lazily iterating over some user defined function
 * Unlike standard Stream[]/LazyList[] from Scala, this implementation doesn't memorise previous values.
 * Unlike Future.sequence/fold we don't know beforehand how many async actions are coming
 *
 * Async iterator implements:
 *
 * - [[AsyncSeqIterator#foldLeft]] for accumulating batching results
 *
 * - [[AsyncSeqIterator#map]] to transform batch results
 *
 * - [[AsyncSeqIterator#foreach]] to iterate with effects
 *
 * For example:
 *
 * {{{
 *
 *   case class MyItem( value: String, cursor: Option[Int] )
 *
 *   def initialItem(): Future[MyItem] = Future.successful(
 *     MyItem( "initial", Some( 1 ) )
 *   )
 *
 *   def nextItem( position: Int ): Future[MyItem] = {
 *     if (position < 10) {
 *       Future.successful(
 *         MyItem( "next", Some( position + 1 ) )
 *       )
 *     } else {
 *       Future.successful(
 *         MyItem( "last", None )
 *       )
 *     }
 *   }
 *
 *   val iterator = AsyncSeqIterator.cons[Future,MyItem, String, Int](
 *     initial = initialItem(),
 *     toValue = _.value,
 *     getPos = _.cursor,
 *     producer = nextItem
 *   )
 *
 *  iterator
 *      .foldLeft( List[String]() ) {
 *         case ( all, itemValue ) =>
 *           all :+ itemValue
 *       }
 * }}}
 *
 * @note It is not possible to implement standard Iterator[] because of the sync nature of hasNext.
 * @tparam F async/effect monad kind (for example, standard scala.concurrent.Future or cats.effect.IO)
 * @tparam I iterating over item type which has a some position
 * @tparam A extracted value type (extracted from I)
 */
trait AsyncSeqIterator[F[_], I, A] {

  /**
   * Future of current item
   */
  def item(): F[I]

  /**
   * Future of value of item
   */
  def value(): F[Option[A]]

  /**
   * Future next iterator if it exists (depends on current item and its position/state)
   */
  def next(): F[Option[AsyncSeqIterator[F, I, A]]]

  /**
   * Iterate and fold (combining) values into the user specified structure and given function `f`
   *
   * @param initial initial value
   * @param f folding function
   * @return a folded value
   */
  def foldLeft[B](
      initial: B
  )( f: ( B, A ) => B ): F[B]

  /**
   * Mapping value of items using the given function `f`
   *
   * @param f a mapping function
   * @return a function result for value
   */
  def map[B](
      f: A => B
  ): AsyncSeqIterator[F, I, B]

  /**
   * Apply the given function `f` to each element of this linear sequence
   * (while respecting the order of the elements).
   *
   * @param f a function to apply
   */
  def foreach[U]( f: A => U ): Unit

  /**
   * Filter items using the given function `f`
   *
   * @param f a filter function
   * @return a new iterator with filtered items
   */
  def filter(
      f: A => Boolean
  ): AsyncSeqIterator[F, I, A]
}

/**
 * Async iterator constructors
 */
object AsyncSeqIterator {

  /**
   * Async iterator constructor
   */
  object cons {

    /**
     * Constructor of an async iterator over some sequence of futures,
     * generated by a provided function and previous state in a sequence
     *
     * @param initial initial async item
     * @param toValue function to extract value from a provided item
     * @param getPos function to extract the current state or position of current item. When a returned position is None than the whole sequence is completely finished
     * @param producer generator of next item based on a state/position in previous item
     * @tparam F async/effect monad kind (for example standard scala.concurrent.Future)
     * @tparam I item type
     * @tparam A value type
     * @tparam P state/position type
     * @return an async iterator instance
     */
    def apply[F[_] : Monad, I, A, P](
        initial: => F[I],
        toValue: I => A,
        getPos: I => Option[P],
        producer: P => F[I]
    ): AsyncSeqIterator[F, I, A] =
      new AsyncSeqIteratorImpl[F, I, A, P](
        initial,
        toValue.andThen( Some.apply ),
        getPos,
        producer
      )
  }

  /**
   * Support of Cats type classes instances for AsyncSeqIterator
   */
  implicit def asyncSeqIteratorInstances[F[_], I]: Functor[AsyncSeqIterator[F, I, *]] =
    new Functor[AsyncSeqIterator[F, I, *]] {

      override def map[A, B]( fa: AsyncSeqIterator[F, I, A] )( f: A => B ): AsyncSeqIterator[F, I, B] = {
        fa.map( f )
      }
    }

}
