import { HttpClient, HttpParams, HttpRequest, HttpErrorResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';

import { Observable } from 'rxjs/Observable';
import { ErrorObservable } from 'rxjs/observable/ErrorObservable';
import 'rxjs/add/operator/catch';
import 'rxjs/add/observable/throw';
import 'rxjs/add/operator/map';

import { Person } from './person.model.generated';
import { ServiceConfig } from './serviceconfig';

@Injectable()
export class PersonEndpoint {
    private get serviceBaseURL(): string {
        return this.serviceConfig.context + '/persons';
    }
    private get onError(): (error: HttpErrorResponse) => ErrorObservable {
        return this.serviceConfig.onError || this.handleError.bind(this);
    }

    constructor(private httpClient: HttpClient, private serviceConfig: ServiceConfig) { }
    /* GET */
    public readGet(id: string): Observable<Person> {
        const url = this.serviceBaseURL + '/' + id + '';
        const params = this.createHttpParams({});

        return this.httpClient.get<Person>(url, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }

    public neighboursGet(id: string): Observable<Person[]> {
        const url = this.serviceBaseURL + '/' + id + '/neighbours';
        const params = this.createHttpParams({});

        return this.httpClient.get<Person[]>(url, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }


    /* HEAD */

    /* POST */
    public movePost(id: string, cityId: string): Observable<void> {
        const url = this.serviceBaseURL + '/' + id + '/to/' + cityId + '';
        const params = this.createHttpParams({});

        return this.httpClient.post<void>(url, null, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }

    public createPost(newEntity: Person): Observable<Person> {
        const url = this.serviceBaseURL + '/';
        const params = this.createHttpParams({});

        return this.httpClient.post<Person>(url, newEntity, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }


    /* PUT */
    public updatePut(id: string, updatedEntity: Person): Observable<Person> {
        const url = this.serviceBaseURL + '/' + id + '';
        const params = this.createHttpParams({});

        return this.httpClient.put<Person>(url, updatedEntity, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }


    /* PATCH */

    /* DELETE */
    public deleteDelete(id: string): Observable<void> {
        const url = this.serviceBaseURL + '/' + id + '';
        const params = this.createHttpParams({});

        return this.httpClient.delete<void>(url, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }

    public clearAllDelete(): Observable<number> {
        const url = this.serviceBaseURL + '/';
        const params = this.createHttpParams({});

        return this.httpClient.delete<number>(url, {params: params})
            .catch((error: HttpErrorResponse) => this.onError(error));
    }


    /* OPTIONS */


    private handleError(error: HttpErrorResponse): ErrorObservable {
        // in a real world app, we may send the error to some remote logging infrastructure
        // instead of just logging it to the console
        this.log('error', error);

        return Observable.throw(error);
    }

    private log(level: string, message: any): void {
        if (this.serviceConfig.debug) {
            console[level](message);
        }
    }

    private createHttpParams(values: { [index: string]: any }): HttpParams {
        let params: HttpParams = new HttpParams();

        Object.keys(values).forEach((key: string) => {
            const value: any = values[key];
            if (value != undefined) {
                params = params.set(key, String(value));
            }
        });

        return params;
    }
}
